/* Schedwi
   Copyright (C) 2011-2014 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* cert_ca.c -- Tools for a certificate authority */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_TIME_H
#include <time.h>
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif

#if HAVE_NETDB_H
#include <netdb.h>
#endif

#if HAVE_IFADDRS_H
#include <ifaddrs.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <gcrypt.h>


#include <utils.h>
#include <lwc_log.h>
#include <conf.h>
#include <xmem.h>
#include <cert_utils.h>
#include <cert_ca.h>


#if GNUTLS_VERSION_NUMBER > 0x020b00
#include <gnutls/abstract.h>
#endif


static unsigned char buffer[4 * 1024];


/*
 * Get hostname.
 *
 * Return:
 *   The hostname (to be freed by free())
 */
static char *
get_hostname ()
{
	size_t len_name, len_domain;
	char *name, *s;
	char domain[256];

#ifdef HOST_NAME_MAX
	len_name = HOST_NAME_MAX;
#else
	len_name = 256;
#endif

	name = (char *) xmalloc (len_name);

	/*
	 * First get the hostname.  If it contain a `.' then it's
	 * probably the FQDN and we return that.
	 */
	name[0] = '\0';
	gethostname (name, len_name);
	name[len_name] = '\0';

	if (strchr (name, '.') != NULL) {
		return name;
	}

	/*
	 * Get the domainname.  If it's `(none)', then we return the hostname.
	 * Otherwise, we return `hostname.domainname'
	 */
	len_domain = sizeof (domain);
	domain[0] = '\0';
	getdomainname (domain, len_domain);
	domain[len_domain - 1] = '\0';

	if (domain[0] == '\0' || strchr (domain, '(') != NULL) {
		return name;
	}

	s = (char *) xmalloc (strlen (name) + strlen (domain) + 2);
	strcpy (s, name);
	strcat (s, ".");
	strcat (s, domain);
	free (name);
	return s;
}


/*
 * Free the NULL-terminated array of hostnames.
 */
static void
free_host_names (char **lst)
{
	int i;

	if (lst == NULL) {
		return;
	}
	for (i = 0; lst[i] != NULL; i++) {
		free (lst[i]);
		lst[i] = NULL;
	}
	free (lst);
}


/*
 * Return the know names of the current host (or the SSLCommonName
 * configuration parameter)
 *
 * Return:
 *  The NULL terminated array of hostnames (to be free by free_host_names()) OR
 *  NULL in case of error (a message has been logged)
 */
static char **
get_host_names ()
{
	char **lst, host[NI_MAXHOST], *hostname;
	struct ifaddrs *ifaddr, *ifa;
	int i, j, family, ret;
	const char *conf_cn;


	ret  = conf_get_param_string ("SSLCommonName", &conf_cn);
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif

	if (conf_cn != NULL && conf_cn[0] != '\0') {
		hostname = xstrdup (conf_cn);
	}
	else {
		hostname = get_hostname ();
	}

	if (getifaddrs (&ifaddr) < 0) {
		lwc_writeLog (	LOG_ERR,
				_("Cannot get network interfaces details: %s"),
				strerror (errno));
		free (hostname);
		return NULL;
	}

	for (i = 0, ifa = ifaddr; ifa != NULL; ifa = ifa->ifa_next) {
		i++;
	}

	lst = (char **) xmalloc (sizeof (char *) * (i + 2));
	lst[0] = hostname;
	for (i = 1, ifa = ifaddr; ifa != NULL; ifa = ifa->ifa_next) {
		family = ifa->ifa_addr->sa_family;
		if (family == AF_INET || family == AF_INET6) {
			ret = getnameinfo (ifa->ifa_addr,
				(family == AF_INET)?sizeof(struct sockaddr_in):
						sizeof(struct sockaddr_in6),
					host, NI_MAXHOST,
					NULL, 0, 0);
			if (ret != 0) {
				lwc_writeLog (	LOG_ERR, _("getnameinfo: %s"),
						gai_strerror (ret));
				while (--i >= 0) {
					if (lst[i] != NULL) {
						free (lst[i]);
					}
				}
				free (lst);
				freeifaddrs (ifaddr);
				return NULL;
			}

			/* Check if the host is already in the list */
			for (j = 0; j < i; j++) {
				if (strcmp (lst[j], host) == 0) {
					break;
				}
			}
			if (j == i) {
				lst[i] = xstrdup (host);
				i++;
			}
		}
	}
	lst[i] = NULL;
	freeifaddrs (ifaddr);
	return lst;
}


/*
 * Create a signed certificate
 *
 * Return:
 *   0 --> OK
 *  -1 --> Error (a message has been logged)
 */
static int
generate_signed_certificate (	gnutls_x509_privkey_t ca_key,
				gnutls_x509_crt_t ca_crt,
				gnutls_x509_crq_t crq,
				const char *outfile)
{
	gnutls_x509_crt_t crt;
	size_t size;
	int ret, days;
	FILE *file;
	int serial;
	char bin_serial[5];
	static char idx_serial = 0;

#if HAVE_ASSERT_H
	assert (ca_key != NULL && ca_crt != NULL && crq != NULL);
#endif

	ret = gnutls_x509_crt_init (&crt);
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot initialize the certificate: %s"),
			gnutls_strerror (ret));
		return -1;
	}

	ret = gnutls_x509_crt_set_crq (crt, crq);
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot use the certificate request: %s"),
			gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}

	/* Set the serial number */
	serial = (int) time(NULL);
	bin_serial[4] = serial & 0xff;
	bin_serial[3] = (serial >> 8) & 0xff;
	bin_serial[2] = (serial >> 16) & 0xff;
	bin_serial[1] = (serial >> 24) & 0xff;
	bin_serial[0] = idx_serial;
	idx_serial = (idx_serial + 1) % 100;

	ret = gnutls_x509_crt_set_serial (crt, bin_serial, 5);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: Serial number error: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}

	/* Activation and expiration */
	gnutls_x509_crt_set_activation_time (crt, time (NULL));
	days = 365 * 10;
	ret = gnutls_x509_crt_set_expiration_time (crt,
					time (NULL) + days * 24 * 60 * 60);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: Cannot set the expiration date: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}

	/* Honour the extensions from the request */
	ret = gnutls_x509_crt_set_crq_extensions (crt, crq);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: Cannot set the extensions: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}

	/* This is NOT a CA's certificate */
	ret = gnutls_x509_crt_set_basic_constraints (crt, 0, -1);
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot set basic constraints: %s"),
			gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}

	/* Certificate usages */
	ret = gnutls_x509_crt_set_key_usage (crt,
		GNUTLS_KEY_KEY_ENCIPHERMENT | GNUTLS_KEY_DIGITAL_SIGNATURE);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: Cannot specify the key usage: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}

	/* Subject Key ID  */
	size = sizeof (buffer);
	ret = gnutls_x509_crt_get_key_id (crt, 0, buffer, &size);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: Cannot get the key ID: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}
	ret = gnutls_x509_crt_set_subject_key_id (crt, buffer, size);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: Cannot set the key ID: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}

	/* Authority Key ID */
	size = sizeof (buffer);
	ret = gnutls_x509_crt_get_subject_key_id (ca_crt, buffer, &size, NULL);
	if (ret < 0) {
		size = sizeof (buffer);
		ret = gnutls_x509_crt_get_key_id (ca_crt, 0, buffer, &size);
	}
	if (ret >= 0) {
		ret = gnutls_x509_crt_set_authority_key_id (crt, buffer, size);
		if (ret < 0) {
			lwc_writeLog (LOG_ERR,
				_("SSL: Cannot set the authority key ID: %s"),
				gnutls_strerror (ret));
			gnutls_x509_crt_deinit (crt);
			return -1;
		}
	}

	/* Version  */
	ret = gnutls_x509_crt_set_version (crt, 3);
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot set the version: %s"),
			gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}

	/* Copy the CRL distribution points */
	gnutls_x509_crt_cpy_crl_dist_points (crt, ca_crt);

	/* Signing (SHA256) */
	ret = gnutls_x509_crt_sign2 (	crt, ca_crt, ca_key, GNUTLS_DIG_SHA256,
					0);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: Failed to sign the certificat: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}

	/* Write the certificate in PEM format */
	size = sizeof (buffer);
	ret = gnutls_x509_crt_export (crt, GNUTLS_X509_FMT_PEM, buffer, &size);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: Failed to export the certificate: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}

	gnutls_x509_crt_deinit (crt);

	file = fopen (outfile, "wb");
	if (file == NULL) {
		lwc_writeLog (LOG_ERR, _("%s: %s"), outfile, strerror (errno));
		return -1;
	}
	if (fwrite (buffer, 1, size, file) != size) {
		lwc_writeLog (	LOG_ERR, _("%s: write error: %s"),
				outfile, strerror (errno));
		fclose (file);
		my_unlink (outfile);
		return -1;
	}
	if (fclose(file) != 0) {
		my_unlink (outfile);
		lwc_writeLog (LOG_ERR, _("%s: %s"), outfile, strerror (errno));
		return -1;
	}

	return 0;
}


/*
 * Create a self-signed certificate
 *
 * Return:
 *   0 --> OK
 *  -1 --> Error (a message has been logged)
 */
static int
generate_self_signed_certificate (gnutls_x509_privkey_t key,
				  const char *outfile)
{
	gnutls_x509_crt_t crt;
	size_t size;
	int ret, days;
	char bin_serial[5];
	FILE *file;
	const char *conf_c, *conf_o, *conf_ou, *conf_l, *conf_st,
		   *conf_cn, *conf_email;

	if (key == NULL || outfile == NULL) {
		return 0;
	}

	ret  = conf_get_param_string ("SSLCACountry", &conf_c);
	ret += conf_get_param_string ("SSLCAOrganization", &conf_o);
	ret += conf_get_param_string ("SSLCAUnit", &conf_ou);
	ret += conf_get_param_string ("SSLCALocality", &conf_l);
	ret += conf_get_param_string ("SSLCAState", &conf_st);
	ret += conf_get_param_string ("SSLCACommonName", &conf_cn);
	ret += conf_get_param_string ("SSLCAEmail", &conf_email);
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif

	ret = gnutls_x509_crt_init (&crt);
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot initialize the certificate: %s"),
			gnutls_strerror (ret));
		return -1;
	}

	/* Set the DN */
	ret = gnutls_x509_crt_set_dn_by_oid (crt,
				GNUTLS_OID_X520_COUNTRY_NAME, 0,
				conf_c, strlen (conf_c));
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot set the COUNTRY (%s): %s"),
			conf_c, gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}

	ret = gnutls_x509_crt_set_dn_by_oid (crt,
			GNUTLS_OID_X520_ORGANIZATION_NAME, 0,
			conf_o, strlen (conf_o));
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot set the ORGANIZATION (%s): %s"),
			conf_o, gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}

	ret = gnutls_x509_crt_set_dn_by_oid (crt,
				GNUTLS_OID_X520_ORGANIZATIONAL_UNIT_NAME, 0,
				conf_ou, strlen (conf_ou));
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot set the ORGANIZATIONAL UNIT (%s): %s"),
			conf_ou, gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}

	ret = gnutls_x509_crt_set_dn_by_oid (crt,
				GNUTLS_OID_X520_LOCALITY_NAME, 0,
				conf_l, strlen (conf_l));
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot set the LOCALITY (%s): %s"),
			conf_l, gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}

	ret = gnutls_x509_crt_set_dn_by_oid (crt,
				GNUTLS_OID_X520_STATE_OR_PROVINCE_NAME, 0,
				conf_st, strlen (conf_st));
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot set the STATE (%s): %s"),
			conf_st, gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}

	ret = gnutls_x509_crt_set_dn_by_oid (crt, GNUTLS_OID_X520_COMMON_NAME,
					0, conf_cn, strlen (conf_cn));
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot set the COMMON NAME (%s): %s"),
			conf_cn, gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}

	/* Set the key */
	ret = gnutls_x509_crt_set_key (crt, key);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: Cannot set the key: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}

	/* Set the serial number (0) */
	bin_serial[4] = 0;
	bin_serial[3] = 0;
	bin_serial[2] = 0;
	bin_serial[1] = 0;
	bin_serial[0] = 0;

	ret = gnutls_x509_crt_set_serial (crt, bin_serial, 5);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: Serial number error: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}

	/* Activation and expiration */
	gnutls_x509_crt_set_activation_time (crt, time (NULL));
	days = 365 * 10;
	ret = gnutls_x509_crt_set_expiration_time (crt,
					time (NULL) + days * 24 * 60 * 60);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: Cannot set the expiration date: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}

	/* This is a CA's certificate */
	ret = gnutls_x509_crt_set_basic_constraints (crt, 1, -1);
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot set basic constraints: %s"),
			gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}

	/* Email */
	if (conf_email != NULL && conf_email[0] != '\0') {
		ret = gnutls_x509_crt_set_subject_alt_name (crt,
					GNUTLS_SAN_RFC822NAME,
					conf_email,
					strlen (conf_email),
					GNUTLS_FSAN_APPEND);
		if (ret < 0) {
			lwc_writeLog (LOG_ERR,
				_("SSL: Cannot set the alt name (%s): %s"),
				conf_email, gnutls_strerror (ret));
			gnutls_x509_crt_deinit (crt);
			return -1;
		}
	}

	/* Certificate usages */
	ret = gnutls_x509_crt_set_key_usage (crt,
			GNUTLS_KEY_KEY_CERT_SIGN | GNUTLS_KEY_CRL_SIGN);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: Cannot specify the key usage: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}

	/* Subject Key ID  */
	size = sizeof (buffer);
	ret = gnutls_x509_crt_get_key_id (crt, 0, buffer, &size);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: Cannot get the key ID: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}
	ret = gnutls_x509_crt_set_subject_key_id (crt, buffer, size);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: Cannot set the key ID: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}


	/* Version  */
	ret = gnutls_x509_crt_set_version (crt, 3);
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot set the version: %s"),
			gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}

	/* Self-signing (SHA256) */
	ret = gnutls_x509_crt_sign2 (crt, crt, key, GNUTLS_DIG_SHA256, 0);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
			_("SSL: Failed to self-sign the certificat: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}

	/* Write the certificate in PEM format */
	size = sizeof (buffer);
	ret = gnutls_x509_crt_export (crt, GNUTLS_X509_FMT_PEM, buffer, &size);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: Failed to export the certificate: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return -1;
	}

	gnutls_x509_crt_deinit (crt);

	file = fopen (outfile, "wb");
	if (file == NULL) {
		lwc_writeLog (LOG_ERR, _("%s: %s"), outfile, strerror (errno));
		return -1;
	}
	if (fwrite (buffer, 1, size, file) != size) {
		lwc_writeLog (	LOG_ERR, _("%s: write error: %s"),
				outfile, strerror (errno));
		fclose (file);
		my_unlink (outfile);
		return -1;
	}
	if (fclose(file) != 0) {
		my_unlink (outfile);
		lwc_writeLog (LOG_ERR, _("%s: %s"), outfile, strerror (errno));
		return -1;
	}

	return 0;
}


/*
 * Load a Certificate from memory.
 * The provided `file' is only used in error message and should contain the
 * file name from which the provided certificate has been read.
 *
 * Return:
 *   The certificate (to be freed by gnutls_x509_crt_deinit()) OR
 *   NULL in case of error (a message has been logged)
 */
gnutls_x509_crt_t
load_cert_from_mem (const char *certstr, size_t lengthstr, const char *infile)
{
	gnutls_x509_crt_t crt;
	int ret;
	gnutls_datum_t dat;
	time_t now, t;

#if HAVE_ASSERT_H
	assert (certstr != NULL);
#endif

	ret = gnutls_x509_crt_init (&crt);
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot initialize the certificate: %s"),
			gnutls_strerror (ret));
		return NULL;
	}

	dat.data = (unsigned char *)certstr;
	dat.size = lengthstr;

	ret = gnutls_x509_crt_import (crt, &dat, GNUTLS_X509_FMT_PEM);
	/* Try in DER format */
	if (	   ret < 0
		&& gnutls_x509_crt_import (crt, &dat, GNUTLS_X509_FMT_DER) < 0)
	{
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot import the certificate from %s: %s"),
			infile, gnutls_strerror (ret));
		gnutls_x509_crt_deinit (crt);
		return NULL;
	}

	/* Check the dates */
	now = time (NULL);

	t = gnutls_x509_crt_get_activation_time (crt);
	if (t > now) {
		lwc_writeLog (LOG_ERR,
		_("SSL: The certificate in %s is not yet activated: %s"),
			infile, ctime (&t));
		gnutls_x509_crt_deinit (crt);
		return NULL;
	}

	t = gnutls_x509_crt_get_expiration_time (crt);
	if (t < now) {
		lwc_writeLog (LOG_ERR,
			_("SSL: The certificate in %s has expired: %s"),
			infile, ctime (&t));
		gnutls_x509_crt_deinit (crt);
		return NULL;
	}

	if (t < now + 30 * 24 * 60 * 60) {
		lwc_writeLog (LOG_NOTICE,
_("SSL: The certificate in %s will expire in less than a month: %s"),
			infile, ctime (&t));
	}

	return crt;
}


/*
 * Load the Certificate Request from a file.
 *
 * Return:
 *   The certificate request (to be freed by gnutls_x509_crq_deinit()) OR
 *   NULL in case of error (a message has been logged)
 */
static gnutls_x509_crq_t
load_request_from_file (const char *infile)
{
	gnutls_x509_crq_t crq;
	int ret;
	gnutls_datum_t dat;
	size_t size;


	ret = gnutls_x509_crq_init (&crq);
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot initialize the certificate request: %s"),
			gnutls_strerror (ret));
		return NULL;
	}

	dat.data = (unsigned char *)read_file (infile, &size);
	dat.size = size;
	if (dat.data == NULL) {
		lwc_writeLog (	LOG_ERR, _("SSL: %s: read error: %s"),
				infile, strerror (errno));
		gnutls_x509_crq_deinit (crq);
		return NULL;
	}

	ret = gnutls_x509_crq_import (crq, &dat, GNUTLS_X509_FMT_PEM);
	/* Try in DER format */
	if (	   ret < 0
		&& gnutls_x509_crq_import (crq, &dat, GNUTLS_X509_FMT_DER) < 0)
	{
		free (dat.data);
		if (ret == GNUTLS_E_BASE64_UNEXPECTED_HEADER_ERROR) {
			lwc_writeLog (LOG_ERR,
				_("SSL: Cannot import the certificate request from %s: Could not find a valid PEM header"),
				infile);
			gnutls_x509_crq_deinit (crq);
			return NULL;
		}
		lwc_writeLog (LOG_ERR,
		_("SSL: Cannot import the certificate request from %s: %s"),
			infile, gnutls_strerror (ret));
		gnutls_x509_crq_deinit (crq);
		return NULL;
	}
	free (dat.data);

	return crq;
}


/*
 * Load a Certificate from a file.
 *
 * Return:
 *   The certificate (to be freed by gnutls_x509_crt_deinit()) OR
 *   NULL in case of error (a message has been logged)
 */
static gnutls_x509_crt_t
load_cert_from_file (const char *infile)
{
	char *s;
	size_t l;
	gnutls_x509_crt_t crt;

	s = read_file (infile, &l);
	if (s == NULL) {
		lwc_writeLog (	LOG_ERR, _("SSL: %s: read error: %s"),
				infile, strerror (errno));
		return NULL;
	}

	crt = load_cert_from_mem (s, l, infile);
	free (s);
	return crt;
}


/*
 * Create the CA key and certificate.  Set the owner of the private key file
 * to `user' and `group' (if `user' and `group' are not NULL)
 *
 * Return:
 *   0 --> OK
 *  -1 --> Error (a message has been logged)
 */
static int
build_ca (	const char *file_ca_cert, const char *file_ca_key,
		const char *user, const char *group)
{
	gnutls_x509_privkey_t key;

	key = generate_private_key (file_ca_key, user, group);
	if (key == NULL) {
		return -1;
	}
	if (generate_self_signed_certificate (key, file_ca_cert) != 0) {
		gnutls_x509_privkey_deinit (key);
		return -1;
	}
	gnutls_x509_privkey_deinit (key);
	return 0;
}


/*
 * Revoke the `cert_to_revoke' certificate and generate/update the provided
 * `crl_file'
 *
 * Return:
 *   0 --> OK
 *  -1 --> Error (a message has been logged)
 */
static int
cert_revoke (	gnutls_x509_privkey_t ca_key,
		gnutls_x509_crt_t ca_cert,
		gnutls_x509_crt_t cert_to_revoke,
		const char *crl_file)
{
	int ret;
	char from_file;
	gnutls_x509_crl_t crl;
	gnutls_datum_t dat;
	size_t size;
	time_t now;
	unsigned int serial;
	char bin_serial[5];
	static char idx_serial = 0;
	FILE *file;
	char *tmp_file;


	ret = gnutls_x509_crl_init (&crl);
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot initialize the CRL: %s"),
			gnutls_strerror (ret));
		return -1;
	}

	if (access (crl_file, F_OK) == 0) {
		dat.data = (unsigned char *)read_file (crl_file, &size);
		dat.size = size;
		if (dat.data == NULL) {
			lwc_writeLog (	LOG_ERR, _("SSL: %s: read error: %s"),
					crl_file, strerror (errno));
			gnutls_x509_crl_deinit (crl);
			return -1;
		}

		ret = gnutls_x509_crl_import (crl, &dat, GNUTLS_X509_FMT_PEM);
		if (ret < 0
			&& gnutls_x509_crl_import (crl, &dat,
						GNUTLS_X509_FMT_DER) < 0)
		{
			lwc_writeLog (LOG_ERR,
		_("SSL: Cannot import the CRL from %s: %s"),
				crl_file, gnutls_strerror (ret));
			free (dat.data);
			gnutls_x509_crl_deinit (crl);
			return -1;
		}
		free (dat.data);
		from_file = 1;
	}
	else {
		from_file = 0;
	}

	now = time (NULL);
	ret = gnutls_x509_crl_set_crt (crl, cert_to_revoke, now);
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot load the certificate to revoke: %s"),
			gnutls_strerror (ret));
		gnutls_x509_crl_deinit (crl);
		return -1;
	}

	ret = gnutls_x509_crl_set_this_update (crl, now);
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot set the update date in the CRL: %s"),
			gnutls_strerror (ret));
		gnutls_x509_crl_deinit (crl);
		return -1;
	}
							/* now + 1 day */
	ret = gnutls_x509_crl_set_next_update (crl, now + 1 * 24 * 60 * 60);
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
		_("SSL: Cannot set the next update date in the CRL: %s"),
			gnutls_strerror (ret));
		gnutls_x509_crl_deinit (crl);
		return -1;
	}

	ret = gnutls_x509_crl_set_version (crl, 2);
	if (ret < 0) {
		lwc_writeLog (LOG_ERR,
			_("SSL: Cannot set the version: %s"),
			gnutls_strerror (ret));
		gnutls_x509_crl_deinit (crl);
		return -1;
	}

	/* Authority Key ID */
	if (from_file == 0) {
		size = sizeof (buffer);
		ret = gnutls_x509_crt_get_subject_key_id (	ca_cert,
								buffer, &size,
								NULL);
		if (ret < 0) {
			size = sizeof (buffer);
			ret = gnutls_x509_crt_get_key_id (	ca_cert, 0,
								buffer, &size);
		}
		if (ret >= 0) {
			ret = gnutls_x509_crl_set_authority_key_id (crl,
								buffer, size);
			if (ret < 0) {
				lwc_writeLog (	LOG_ERR,
					_("SSL: Cannot set the key ID: %s"),
					gnutls_strerror (ret));
				gnutls_x509_crl_deinit (crl);
				return -1;
			}
		}

		serial = (unsigned int) now;
		bin_serial[4] = serial & 0xff;
		bin_serial[3] = (serial >> 8) & 0xff;
		bin_serial[2] = (serial >> 16) & 0xff;
		bin_serial[1] = (serial >> 24) & 0xff;
		bin_serial[0] = idx_serial;
		idx_serial = (idx_serial + 1) % 100;

		ret = gnutls_x509_crl_set_number (crl, bin_serial, 5);
		if (ret < 0) {
			lwc_writeLog (	LOG_ERR,
				_("SSL: Serial number error: %s"),
				gnutls_strerror (ret));
			gnutls_x509_crl_deinit (crl);
			return -1;
		}
	}

#if GNUTLS_VERSION_NUMBER <= 0x020b00
	ret = gnutls_x509_crl_sign (crl, ca_cert, ca_key);
#else
	{
		gnutls_privkey_t abs_key;

		ret = gnutls_privkey_init (&abs_key);
		if (ret < 0) {
			lwc_writeLog (	LOG_ERR,
			_("SSL: Cannot initialize the abstract key: %s"),
					gnutls_strerror (ret));
			gnutls_x509_crl_deinit (crl);
			return -1;
		}
		ret = gnutls_privkey_import_x509 (abs_key, ca_key, 0);
		if (ret < 0) {
			lwc_writeLog (	LOG_ERR,
				_("SSL: Cannot import the x509 key: %s"),
					gnutls_strerror (ret));
			gnutls_privkey_deinit (abs_key);
			gnutls_x509_crl_deinit (crl);
			return -1;
		}

		ret = gnutls_x509_crl_privkey_sign (	crl, ca_cert, abs_key,
							GNUTLS_DIG_SHA256, 0);
		gnutls_privkey_deinit (abs_key);
	}
#endif
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: Failed to sign the CRL: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crl_deinit (crl);
		return -1;
	}

	/* Write the CRL in PEM format */
	size = sizeof (buffer);
	ret = gnutls_x509_crl_export (crl, GNUTLS_X509_FMT_PEM, buffer, &size);
	if (ret < 0) {
		lwc_writeLog (	LOG_ERR,
				_("SSL: Failed to export the CRL: %s"),
				gnutls_strerror (ret));
		gnutls_x509_crl_deinit (crl);
		return -1;
	}

	gnutls_x509_crl_deinit (crl);

	tmp_file = (char *) xmalloc (strlen (crl_file) + strlen (TMP_EXTENSION)
					+ 1);
	strcpy (tmp_file, crl_file);
	strcat (tmp_file, TMP_EXTENSION);

	file = fopen (tmp_file, "wb");
	if (file == NULL) {
		lwc_writeLog (LOG_ERR, _("%s: %s"), tmp_file,
			      strerror (errno));
		free (tmp_file);
		return -1;
	}
	if (fwrite (buffer, 1, size, file) != size) {
		lwc_writeLog (	LOG_ERR, _("%s: write error: %s"),
				tmp_file, strerror (errno));
		fclose (file);
		my_unlink (tmp_file);
		free (tmp_file);
		return -1;
	}
	if (fclose(file) != 0) {
		my_unlink (tmp_file);
		lwc_writeLog (LOG_ERR, _("%s: %s"), tmp_file,
			      strerror (errno));
		free (tmp_file);
		return -1;
	}

	if (rename (tmp_file, crl_file) != 0) {
		my_unlink (tmp_file);
		lwc_writeLog (	LOG_ERR, _("Cannot rename %s in %s: %s"),
				tmp_file, crl_file, strerror (errno));
		free (tmp_file);
		return -1;
	}

	my_unlink (tmp_file);
	free (tmp_file);
	return 0;
}


/*
 * Check if the certificates and keys exist.  If not, create them.  `user' and
 * `group' are used to protect the private keys.
 *
 * Return:
 *   0 --> OK
 *  -1 --> Error (a message has been logged)
 */
int
cert_check_and_create (	const char *file_ca_cert,
			const char *file_ca_key,
			const char *file_cert,
			const char *file_key,
			const char *user,
			const char *group,
			char quick_random)
{
	int 	missing_file_ca_cert, missing_file_ca_key,
		missing_file_cert, missing_file_key;
	gnutls_x509_privkey_t key;
	gnutls_x509_crt_t ca_crt;
	gnutls_x509_crq_t crq;
	char **hostnames;

#if HAVE_ASSERT_H
	assert (   file_ca_cert != NULL && file_ca_key != NULL
		&& file_cert != NULL && file_key != NULL);
#endif

	/* Check which files already exist */
	missing_file_ca_cert = access (file_ca_cert, F_OK);
	missing_file_ca_key  = access (file_ca_key, F_OK);
	missing_file_cert    = access (file_cert, F_OK);
	missing_file_key     = access (file_key, F_OK);

	if (	   quick_random == 0
		&& (missing_file_ca_key != 0 || missing_file_key != 0))
	{
		lwc_writeLog (LOG_INFO,
	_("Creating SSL keys and certificates (may take several minutes)..."));
	}
	if (missing_file_ca_key) {
		if (build_ca (file_ca_cert, file_ca_key, user, group) != 0) {
			return -1;
		}
		/* We need to recreate the CA and the server certificates */
		missing_file_ca_cert = missing_file_cert = 1;
	}

	if (missing_file_ca_cert) {
		key = load_private_key (file_ca_key);
		if (key == NULL) {
			return -1;
		}
		if (generate_self_signed_certificate (key, file_ca_cert) != 0) {
			gnutls_x509_privkey_deinit (key);
			return -1;
		}
		gnutls_x509_privkey_deinit (key);
		/* We need to recreate the server certificate */
		missing_file_cert = 1;
	}

	if (missing_file_key) {
		key = generate_private_key (file_key, user, group);
		if (key == NULL) {
			return -1;
		}
		gnutls_x509_privkey_deinit (key);
		/* We need to recreate the server certificate */
		missing_file_cert = 1;
	}

	if (missing_file_cert) {
		key = load_private_key (file_key);
		if (key == NULL) {
			return -1;
		}
		hostnames = get_host_names ();
		if (hostnames == NULL) {
			gnutls_x509_privkey_deinit (key);
			return -1;
		}
		crq = generate_request (key, hostnames, NULL);
		free_host_names (hostnames);
		gnutls_x509_privkey_deinit (key);
		if (crq == NULL) {
			return -1;
		}

		key = load_private_key (file_ca_key);
		if (key == NULL) {
			gnutls_x509_crq_deinit (crq);
			return -1;
		}

		ca_crt = load_cert_from_file (file_ca_cert);
		if (ca_crt == NULL) {
			gnutls_x509_privkey_deinit (key);
			gnutls_x509_crq_deinit (crq);
			return -1;
		}

		if (generate_signed_certificate (	key, ca_crt,
							crq, file_cert) != 0)
		{
			gnutls_x509_crt_deinit (ca_crt);
			gnutls_x509_privkey_deinit (key);
			gnutls_x509_crq_deinit (crq);
			return -1;
		}
		gnutls_x509_crt_deinit (ca_crt);
		gnutls_x509_privkey_deinit (key);
		gnutls_x509_crq_deinit (crq);
	}
	if (	   quick_random == 0
		&& (missing_file_ca_key != 0 || missing_file_key != 0))
	{
		lwc_writeLog (LOG_INFO, _("Done"));
	}
	return 0;
}


/*
 * Revoke a certificate
 *
 * Return:
 *   0 --> OK
 *  -1 --> Error (a message has been logged)
 */
int
revoke_certificate (	const char *ca_keyfile, const char *ca_crtfile,
			const char *crt2revoke, const char *crl_file)
{
	gnutls_x509_privkey_t key;
	gnutls_x509_crt_t ca_crt, crt;

#if HAVE_ASSERT_H
	assert (   ca_keyfile != NULL && ca_crtfile != NULL
		&& crt2revoke != NULL && crl_file != NULL);
#endif

	key = load_private_key (ca_keyfile);
	if (key == NULL) {
		return -1;
	}

	ca_crt = load_cert_from_file (ca_crtfile);
	if (ca_crt == NULL) {
		gnutls_x509_privkey_deinit (key);
		return -1;
	}

	crt = load_cert_from_file (crt2revoke);
	if (crt == NULL) {
		gnutls_x509_crt_deinit (ca_crt);
		gnutls_x509_privkey_deinit (key);
		return -1;
	}

	if (cert_revoke (key, ca_crt, crt, crl_file) != 0) {
		gnutls_x509_crt_deinit (crt);
		gnutls_x509_crt_deinit (ca_crt);
		gnutls_x509_privkey_deinit (key);
		return -1;
	}
	gnutls_x509_crt_deinit (crt);
	gnutls_x509_crt_deinit (ca_crt);
	gnutls_x509_privkey_deinit (key);
	return 0;
}


/*
 * Sign a request (CSR)
 *
 * Return:
 *   0 --> OK
 *  -1 --> Error (a message has been logged)
 */
int
sign_request (const char *ca_keyfile, const char *ca_crtfile,
		const char *csrfile, const char *outsignedcrt)
{
	gnutls_x509_privkey_t key;
	gnutls_x509_crt_t ca_crt;
	gnutls_x509_crq_t crq;

#if HAVE_ASSERT_H
	assert (   ca_keyfile != NULL && ca_crtfile != NULL
		&& csrfile != NULL && outsignedcrt != NULL);
#endif

	key = load_private_key (ca_keyfile);
	if (key == NULL) {
		return -1;
	}

	ca_crt = load_cert_from_file (ca_crtfile);
	if (ca_crt == NULL) {
		gnutls_x509_privkey_deinit (key);
		return -1;
	}

	crq = load_request_from_file (csrfile);
	if (crq == NULL) {
		gnutls_x509_crt_deinit (ca_crt);
		gnutls_x509_privkey_deinit (key);
		return -1;
	}

	if (generate_signed_certificate (key, ca_crt, crq, outsignedcrt) != 0)
	{
		gnutls_x509_crq_deinit (crq);
		gnutls_x509_crt_deinit (ca_crt);
		gnutls_x509_privkey_deinit (key);
		return -1;
	}
	gnutls_x509_crq_deinit (crq);
	gnutls_x509_crt_deinit (ca_crt);
	gnutls_x509_privkey_deinit (key);
	return 0;
}


/*
 * Return the hostname stored in the certificate
 *
 * Return:
 *   The hostname (to be freed by free() OR
 *   NULL in case of error
 */
char *
get_hostname_from_cert (gnutls_x509_crt_t cert)
{
	char *dnsname;
	size_t dnsnamesize;
	int ret, i;

#if HAVE_ASSERT_H
	assert (cert != NULL);
#endif

	dnsname = (char *) xmalloc (512);

	/*
	 * First try to get a DNS name in an alternative name (subjectAltName)
	 * extension and then try the Common Name
	 */
	for (i = ret = 0; ret >= 0; i++) {
		dnsnamesize = 512;
		ret = gnutls_x509_crt_get_subject_alt_name (cert, i,
							dnsname, &dnsnamesize,
							NULL);
		if (ret == GNUTLS_SAN_DNSNAME) {
			return dnsname;
		}
	}

	dnsnamesize = 512;
	ret = gnutls_x509_crt_get_dn_by_oid (	cert,
						GNUTLS_OID_X520_COMMON_NAME, 0,
					 	0, dnsname, &dnsnamesize);
	if (ret == 0) {
		return dnsname;
	}

	free (dnsname);
	return NULL;
}

/*-----------------============== End Of File ==============-----------------*/
