/* Schedwi
   Copyright (C) 2007-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* calendar_list.c -- Management functions for the calendars */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <sql_common.h>
#include <sql_calendar.h>
#include <lwc_log.h>
#include <lwc_linkedlist.h>
#include <utils.h>
#include <xmem.h>
#include <calendar_list.h>


/*
 * Error callback function for the sql_cal_only_list() function
 */
static void
sql_error_logger (void *data, const char *msg, int err_code)
{
    if (msg != NULL) {
	lwc_writeLog (LOG_ERR, msg);
    }
    else {
	lwc_writeLog (LOG_ERR,
	    _("Database error while trying to retrieve the calendar list"));
    }
}


/*
 * Retrieve and build the calendar array. list and nmemb (number of elements
 * in the list) are set. The last element of the list has an id set to 0.
 *
 * Return:
 *   0 --> No error.  list is set and must be freed by the caller (free()).
 *	 Calendars with syntax error are ignored (but a message is displayed
 *	 by using lwc_writeLog())
 *  -1 --> Memory allocation error.  An error message has been displayed by
 *	 using lwc_writeLog().
 *  -2 --> Database error.  An error message has been displayed by
 *	 using lwc_writeLog().
 */
static int
get_calendar_list (	calendar_t_ptr *list, unsigned int *nmemb,
			const schedwi_date *workload_date)
{
   lwc_LL *cals;
   row_item_t *row;
   unsigned int i;
   int ret;
   calendar_t_ptr ptr;
   cal_errcode_t ret1, ret2, ret3;
   unsigned short int year;
   unsigned long long int id;
   char *name, *formula;


#if HAVE_ASSERT_H
   assert (list != NULL && workload_date != NULL);
#endif

   /* Retrieve all the calendars */
   ret = sql_cal_only_list (	schedwi_date_to_int (workload_date), &cals,
				sql_error_logger, NULL);
   if (ret != 0) {
      return ret;
   }

   /* Allocate the array */
   ptr = (calendar_t_ptr) xmalloc (	  sizeof (calendar_t)
					* (lwc_getNumNode (cals) + 1));

   /* Copy the list in the array */
   year = workload_date->year;
   i = 0;
   while ((row = (row_item_t *) lwc_delStartLL (cals)) != NULL) {
      /*
       * row[0] --> ID
       * row[1] --> Name
       * row[2] --> Formula
       */
      id      = (unsigned long long int)sql_row_item2ll (&(row[0]));
      name    = row[1].value_string;
      formula = row[2].value_string;

      ret1 = str2cal (&(ptr[i].calendar[0]), formula, year - 1, NULL);
      ret2 = str2cal (&(ptr[i].calendar[1]), formula, year, NULL);
      ret3 = str2cal (&(ptr[i].calendar[2]), formula, year + 1, NULL);
      switch	((ret1 == CAL_NOERROR) ?
		((ret2 == CAL_NOERROR) ? ret3 : ret2) : ret1)
      {
	case CAL_NOERROR:
	    ptr[i++].id = id;
	    break;

	 case CAL_MALLOC:
	    lwc_writeLog (LOG_ERR,
_("Memory allocation error when trying to parse the calendar `%s' (ID: %lld)"),
	                  name, id);
	    break;

	 case CAL_EMPTYFIELD:
	    lwc_writeLog (LOG_ERR,
_("Calendar syntax error: calendar `%s' (ID: %lld): a required field is empty"),
	                  name, id);
	    break;

	 case CAL_BADMONTHNAME:
	    lwc_writeLog (LOG_ERR,
_("Calendar syntax error: calendar `%s' (ID: %lld): invalid month name"),
	                  name, id);
	    break;

	 case CAL_BADDAYNAME:
	    lwc_writeLog (LOG_ERR,
_("Calendar syntax error: calendar `%s' (ID: %lld): invalid day name"),
	                  name, id);
	    break;

	 case CAL_MONTHOOR:
	    lwc_writeLog (LOG_ERR,
_("Calendar syntax error: calendar `%s' (ID: %lld): month number must be between 1 and 12"),
	                  name, id);
	    break;

	 case CAL_DAYOOR:
	    lwc_writeLog (LOG_ERR,
_("Calendar syntax error: calendar `%s' (ID: %lld): day number must be between 1 and 31"),
	                  name, id);
	    break;

	 case CAL_BADOPTION:
	    lwc_writeLog (LOG_ERR,
_("Calendar syntax error: calendar `%s' (ID: %lld): invalid option"),
	                  name, id);
	    break;

	 default:
	    lwc_writeLog (LOG_ERR,
_("Calendar syntax error: calendar `%s' (ID: %lld): syntax error"),
	                  name, id);
	    break;
      }
      sql_free_row (row);
   }
   lwc_delLL (cals, NULL);
   ptr[i].id = 0;

   *list = ptr;
   if (nmemb != NULL) {
      *nmemb = i;
   }
   return 0;
}


/*
 * Create a new calendar_list_t object and fill it with the compiled calendars
 * for the provided year, the previous one and the next one.
 *
 * Return:
 *   The new object (to be freed by destroy_calendar_list()) or
 *   NULL in case of error (an error message has been displayed by using
 *	lwc_writeLog())
 */
calendar_list_t_ptr
new_calendar_list (const schedwi_date *workload_date)
{
	calendar_list_t_ptr ptr;


#if HAVE_ASSERT_H
	assert (workload_date != NULL);
#endif

	ptr = (calendar_list_t_ptr) xmalloc (sizeof (calendar_list_t));
	if (get_calendar_list (	&(ptr->list), &(ptr->ncals),
				workload_date) != 0)
	{
		free (ptr);
		return NULL;
	}
	ptr->year = workload_date->year;
	return ptr;
}


/*
 * Free a calendar_list_t object
 */
void
destroy_calendar_list (calendar_list_t_ptr ptr)
{
	if (ptr != NULL) {
		if (ptr->list != NULL) {
			free (ptr->list);
		}
		free (ptr);
	}
}


/**
 * Retrieve all the active calendars for the provided day (day/month/year).
 * The returned string is a list of the calendar IDs (separated by `,')
 *
 * @param[in] ptr The list of calendars (as returned by new_calendar_list()).
 * @param[in] date The date.
 * @return The list of calendar IDs (separated by `,') as a string.  To be
 *         freed by the caller with free().
 */
char *
get_calendar_list_for_the_day (	calendar_list_t_ptr ptr,
				const schedwi_date *date)
{
	char *s;
	unsigned int i, j, n, year_idx;


#if HAVE_ASSERT_H
	assert (   ptr != NULL && date != NULL
		&& (	   ptr->year - 1 == date->year
			|| ptr->year == date->year
			|| ptr->year + 1 == date->year));
#endif

	/* Allocate the string */
	s = (char *) xmalloc (ptr->ncals * 25 + 1);

	/*
	 * In the calendar list, copy the ID of the calendars
	 * matching the provided day/month/year
	 */
	year_idx = 1 + date->year - ptr->year;
	for (i = j = 0; i < ptr->ncals; i++) {
		if (calmatch (&(ptr->list[i].calendar[year_idx]),
			date->day, date->month, date->year) == CAL_NOERROR)
		{
			n = copy_ulltostr (ptr->list[i].id, s + j);
			s[j + n] = ',';
			j += n + 1;
		}
	}

	/* Remove the last `,' */
	if (j == 0) {
		s[0] = '\0';
	}
	else {
		s[j - 1] = '\0';
	}

	return s;
}

/*-----------------============== End Of File ==============-----------------*/
