# Schedwi
# Copyright (C) 2012-2015 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Job/jobset status UI."""

from datetime import date, datetime
from babel import dates

from sqlalchemy import desc
from muntjac.api import Table
from muntjac.data.util.indexed_container import IndexedContainer
from muntjac.ui.table import ICellStyleGenerator

from web.jobtable import JobTableListener
from tables.job_status import job_status
import status_utils
import locale_utils


class Status(Table, JobTableListener):

    """UI component to display the status of a job or jobset."""

    def __init__(self, jobTable, sql_session, workload=None):
        """Build the job/jobset status UI.

        @param jobTable:
                    the associated L{web.jobtable.JobTable} object.
        @param sql_session:
                    SQLAlchemy session.
        @param workload:
                    workload to consider (ignored)
        """
        super(Status, self).__init__()
        self._sql_session = sql_session

        self.setSizeFull()
        self.setSelectable(False)
        self.setMultiSelect(False)
        self.setColumnReorderingAllowed(False)
        self.setColumnCollapsingAllowed(False)
        self._container = self._newContainer()
        self.setContainerDataSource(self._container)
        self.setColumnHeaders([_('Workload'), _('Status'), _('Date/time'),
                               _('Message')])
        self.setColumnAlignment('status', Table.ALIGN_CENTER)
        self.setColumnExpandRatio('message', 1.0)
        self.addStyleName("jobstatus")
        self.setCellStyleGenerator(TableStyleGenerator(self))

        jobTable.add_listener(self)

    def _newContainer(self):
        """Return a new container for the job status table.

        @return:    the new container.
        """
        jContainer = IndexedContainer()
        jContainer.addContainerProperty('workload', str, None)
        jContainer.addContainerProperty('status', str, None)
        jContainer.addContainerProperty('date', str, None)
        jContainer.addContainerProperty('message', str, None)
        return jContainer

    def refresh(self, job, jobsetIds):
        """Show the status of the given job/jobset.

        @param job:
                    the job ID for which the status must be displayed.
        @param jobsetIds:
                    an array of the IDs the parent jobsets (the last element
                    is the ID of the jobset that contains `job')
        """
        self._container = self._newContainer()
        if job:
            session = self._sql_session.open_session()
            query = session.query(job_status).filter(job_status.job_id == job)
            for s in query.order_by(desc(job_status.workload_date)):
                d = date(s.workload_date / 10000,
                         s.workload_date % 10000 / 100,
                         s.workload_date % 100)
                date_str = dates.format_date(d, format='short',
                                             locale=locale_utils.get_locale())
                dt = datetime.fromtimestamp(s.time_status_set)
                dt_str = dates.format_datetime(
                    dt,
                    format='full',
                    locale=locale_utils.get_locale())
                i = self._container.generateId()
                item = self._container.addItem(i)
                item.getItemProperty('workload').setValue(
                    '%s - %s' % (s.workload_date,
                                 date_str.encode('utf-8')))
                item.getItemProperty('status').setValue(
                    status_utils.status2string(s.status))
                item.getItemProperty('date').setValue(dt_str.encode('utf-8'))
                item.getItemProperty('message').setValue(
                    s.error_msg.encode('utf-8'))
            self._sql_session.close_session(session)
        self.setContainerDataSource(self._container)


class TableStyleGenerator(ICellStyleGenerator):

    """ICellStyleGenerator object to style de cells (status of the job)."""

    def __init__(self, c):
        self._c = c

    def getStyle(self, itemId, propertyId):
        if propertyId == 'status':
            item = self._c._container.getItem(itemId)
            s = item.getItemProperty('status').getValue()
            return status_utils.string2colorname(s)
