# Schedwi
# Copyright (C) 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""PAM user authentication."""


try:
    import PAM
except:
    WITH_PAM = False
else:
    WITH_PAM = True
import parse_config


def _pam_conversation(auth, query_list):
    """PAM conversation function.

    @param auth:
                The PAM handler.  auth.get_userdata() is the tuple
                [username, password]
    @param query_list:
                The PAM query list.
    @return:
                The response list
    """
    resp = []
    username, password = auth.get_userdata()
    for i in range(len(query_list)):
        query, type = query_list[i]
        if type == PAM.PAM_PROMPT_ECHO_ON:
            resp.append((username, 0))
        elif type == PAM.PAM_PROMPT_ECHO_OFF:
            resp.append((password, 0))
        elif (type == PAM.PAM_PROMPT_ERROR_MSG or
              type == PAM.PAM_PROMPT_TEXT_INFO):
            resp.append(('', 0))
        else:
            return None
    return resp


def pam_check_user(username, password):
    """Authenticate the given user with PAM.

    @param username:
                The name of the user to authenticate.
    @param password:
                The user password.
    @return:
                A tuple as follow:
                   - False on failed authentication, error, or if PyPAM is not
                     available.  True is the user has been successfully
                     authenticated.
                   - A message (a string)
    """
    if parse_config.PAM_AUTH is False or WITH_PAM is False:
        return (False, _("PAM disabled"))
    service = parse_config.PAM_AUTH_SERVICE
    auth = PAM.pam()
    auth.start(service)
    auth.set_item(PAM.PAM_USER, username)
    auth.set_item(PAM.PAM_CONV, _pam_conversation)
    auth.set_userdata([username, password])
    try:
        auth.authenticate()
        auth.acct_mgmt()
    except PAM.error as (resp, code):
        return (False, resp)
    except Exception as e:
        return (False, str(e))
    return (True, _("User authenticated"))
