# Schedwi
# Copyright (C) 2012, 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Main window UI."""

from muntjac.api import (HorizontalSplitPanel, VerticalSplitPanel,
                         VerticalLayout)
from muntjac.terminal.sizeable import ISizeable
from muntjac.addon.refresher.refresher import RefreshListener
from muntjac.ui.window import Notification

from web.jobsettree import JobsetTree, JobsetTreeListener
from web.jobtable import JobTable, JobTableRepaint, JobTableSelect
from web.linksandfiles import LinksAndFiles
from web.autorefresh import AutoRefresh
from web.register import get_pending
import parse_config


class MainWindow(HorizontalSplitPanel, RefreshListener):

    """Main window UI.  Left part shows the jobset tree and the right part
    shows the content (jobs/jobsets) of the selected jobset.
    """

    def __init__(self, application, sql_session, workload=None):
        """Create the main UI.

        @param application:
                    the L{web.main.Main} object.
        @type application:
                    L{muntjac.api.Application}
        @param sql_session:
                    SQLAlchemy session.
        @param workload:
                    workload to consider.
        """
        super(MainWindow, self).__init__()
        self._application = application
        self._sql_session = sql_session
        self._workload = workload
        self._pending_hosts = None

        # Layout
        self.setSizeFull()
        self.setSplitPosition(250, ISizeable.UNITS_PIXELS)

        #
        # Left side: jobset tree
        #
        self._tree = JobsetTree(application, sql_session, workload)
        self._tree.setSizeFull()
        self._tree.add_listener(NewSelection(self))
        self.addComponent(self._tree)

        #
        # Right side: job list
        #
        v = VerticalLayout()
        v.setSizeFull()
        self.addComponent(v)

        # Refresher (not shown)
        r = AutoRefresh(parse_config.GUI_REFRESH)
        r.addListener(self, RefreshListener)
        v.addComponent(r.get_refresher())

        # Bread crumb
        b = self._tree.get_breadcrumb()
        v.addComponent(b)

        vsplit = VerticalSplitPanel()
        vsplit.setSizeFull()
        vsplit.setSplitPosition(65, ISizeable.UNITS_PERCENTAGE)
        v.addComponent(vsplit)
        v.setExpandRatio(vsplit, 1.0)

        # Jobs
        self._table = JobTable(sql_session, workload)
        self._table.setSizeFull()
        self._table.add_listener(RefreshJobsetTree(self))
        self._table.add_listener(SelectJobsetTree(self))
        vsplit.addComponent(self._table)

        # Links, constraint files, status and stat
        self._tabs = LinksAndFiles(self._table, sql_session, workload)
        self._tabs.setSizeFull()
        vsplit.addComponent(self._tabs)

        # Refresh to display the initial page (the root jobset)
        self._tree.refresh()

    def check_pending_host(self):
        """Display a tray notification if registration requests are pending."""
        if self._workload is not None:
            return
        lst_host = get_pending()
        if lst_host and (not self._pending_hosts or
                         len(self._pending_hosts) < len(lst_host)):
            self._application.getMainWindow().showNotification(
                _("Pending host requests (open the Host \
                   Window for approval):"),
                '<br/>'.join(lst_host),
                Notification.TYPE_TRAY_NOTIFICATION)
        self._pending_hosts = lst_host

    def refresh(self, source=None):
        """Callback for the auto-refresh."""
        self._tree.refresh()
        self.check_pending_host()

    def refresh_job_view(self, jobset_hierarchy):
        """Refresh the job table view.

        @param jobset_hierarchy:
                    the list of all the jobset IDs from the root jobset up to
                    the jobset for which the content must be displayed.
        """
        self._table.display_jobset_content(jobset_hierarchy)

    def refresh_jobset_view(self):
        """Refresh the jobset tree view."""
        self._tree.refresh()
        self.check_pending_host()

    def select_jobset(self, jobset_hierarchy):
        """Select/display a jobset in the tree.

        @param jobset_hierarchy:
                    the list of all the jobset IDs from the root jobset up to
                    the jobset for which the content must be displayed.
        """
        self._tree.select_jobset(jobset_hierarchy)
        self.check_pending_host()


class NewSelection(JobsetTreeListener):

    """Callback for when a new jobset is selected in the jobset tree view."""

    def __init__(self, main_window_obj):
        """Initialize the callback.

        @param main_window_obj:
                    the associated L{MainWindow} object.
        """
        super(NewSelection, self).__init__()
        self._c = main_window_obj

    def refresh(self, jobset_hierarchy):
        self._c.refresh_job_view(jobset_hierarchy)


class RefreshJobsetTree(JobTableRepaint):

    """Callback for when a refresh request is sent by the job table."""

    def __init__(self, main_window_obj):
        """Initialize the callback.

        @param main_window_obj:
                    the associated L{MainWindow} object.
        """
        super(RefreshJobsetTree, self).__init__()
        self._c = main_window_obj

    def repaint(self):
        self._c.refresh_jobset_view()


class SelectJobsetTree(JobTableSelect):

    """Callback for when a new jobset must be displayed."""

    def __init__(self, main_window_obj):
        """Initialize the callback.

        @param main_window_obj:
                    the associated L{MainWindow} object.
        """
        super(SelectJobsetTree, self).__init__()
        self._c = main_window_obj

    def select(self, jobset_hierarchy):
        self._c.select_jobset(jobset_hierarchy)
