# Schedwi
# Copyright (C) 2012, 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Job/jobset dependency links UI."""

from muntjac.api import (VerticalLayout, Table, HorizontalLayout, Button,
                         Window, Alignment, NativeSelect)
from muntjac.ui.window import (Notification, IResizeListener)
from muntjac.data.util.indexed_container import IndexedContainer
from muntjac.terminal.theme_resource import ThemeResource
from muntjac.data.property import IValueChangeListener
from muntjac.ui.button import IClickListener
from muntjac.ui.table import ICellStyleGenerator
from muntjac.terminal.sizeable import ISizeable

from web.jobtable import JobTableListener
from web.cutcopypastemenu import GenericContextMenu
from web.selectjobwidget import SelectJob
from web.boxes import DeleteBox, BoxCloseListener
from web.autorefresh import AutoRefresh
from web.link_graph import LinkGraph
import path
import status_utils
from simple_queries_job import sql_get_job
from cmd_job.ln import detect_loop
from tables.link import link
from tables.link_s import link_s
from tables.job_main_s import job_main_s


class Links(HorizontalLayout, JobTableListener):

    """Dependency links UI."""

    def __init__(self, jobTable, sql_session, workload=None):
        """Build the dependency links table.

        @param jobTable:
                    the associated L{JobTable} object.
        @param sql_session:
                    SQLAlchemy session.
        @param workload:
                    workload to consider.
        """
        super(Links, self).__init__()
        self._sql_session = sql_session
        self._workload = workload
        self._selected = None
        self._job = None
        self._jobsetIds = None

        self.setMargin(True)
        self.setSpacing(True)
        self.setSizeFull()

        self._table = Table()
        self._table.setSizeFull()
        self._table.setSelectable(False)
        self._table.setMultiSelect(False)
        self._table.setColumnReorderingAllowed(False)
        self._table.setColumnCollapsingAllowed(False)
        self._table.setImmediate(True)
        self._table.addStyleName("jobstatus")
        self._table.setCellStyleGenerator(TableStyleGenerator(self))
        self._container = self._newContainer()
        self._table.setContainerDataSource(self._container)
        self._table.setVisibleColumns(['name', 'mustbe'])
        self._table.setColumnHeaders([_('Job/Jobset'), _('Must be')])
        self._table.setColumnExpandRatio('name', 1)
        self._table.setColumnAlignment('mustbe', Table.ALIGN_CENTER)
        self.addComponent(self._table)
        self.setExpandRatio(self._table, 1.0)

        btbox = VerticalLayout()
        btbox.setSpacing(True)
        btbox.setMargin(False)
        btbox.setWidth('110px')

        if workload is None or job_main_s.rw_links is True:
            self._table.setSelectable(True)
            self._table.addListener(SelectionHandler(self),
                                    IValueChangeListener)
            self._table.addActionHandler(LinkContextMenu(self))

            b = Button(_('Edit'))
            b.setIcon(ThemeResource('icons/edit.png'))
            b.setDescription(_("Edit the dependency link"))
            b.setEnabled(False)
            b.setWidth('100%')
            b.addListener(EditBtHandler(self), IClickListener)
            btbox.addComponent(b)
            self._btedit = b

            b = Button(_('Add'))
            b.setIcon(ThemeResource('icons/add.png'))
            b.setDescription(_("Add a new dependency link"))
            b.setEnabled(False)
            b.setWidth('100%')
            b.addListener(AddBtHandler(self), IClickListener)
            btbox.addComponent(b)
            self._btadd = b

            b = Button(_('Delete'))
            b.setIcon(ThemeResource('icons/delete.png'))
            b.setDescription(_("Remove a dependency link"))
            b.setEnabled(False)
            b.setWidth('100%')
            b.addListener(DeleteBtHandler(self), IClickListener)
            btbox.addComponent(b)
            self._btdel = b

        b = Button(_('Preview'))
        b.setIcon(ThemeResource('icons/graph.png'))
        b.setDescription(_("Preview the dependency tree"))
        b.setEnabled(False)
        b.setWidth('100%')
        b.addListener(PreviewBtHandler(self), IClickListener)
        btbox.addComponent(b)
        self._btpreview = b
        self.addComponent(btbox)

        jobTable.add_listener(self)

    def _newContainer(self):
        jContainer = IndexedContainer()
        jContainer.addContainerProperty('name', str, None)
        jContainer.addContainerProperty('mustbe', str, None)
        jContainer.addContainerProperty('job_id', long, None)
        jContainer.addContainerProperty('link_type', int, None)
        return jContainer

    def getLinkDetails(self, item_id):
        """Get the entry details from a container item ID.

        @param item_id:     the container item ID.
        @return:            a dictionnary containing the item details.
        """
        item = self._container.getItem(item_id)
        r = dict()
        r['name'] = item.getItemProperty('name').getValue()
        r['mustbe'] = item.getItemProperty('mustbe').getValue()
        r['job_id'] = item.getItemProperty('job_id').getValue()
        r['link_type'] = item.getItemProperty('link_type').getValue()
        r['container_id'] = item_id
        return r

    def refresh(self, job, jobsetIds):
        """Show the links in the table for the given job.

        @param job:
                    the job ID for which the links must be shown.
        @param jobsetIds:
                    an array of the IDs the parent jobsets (the last element
                    is the ID of the jobset that contains `job')
        """
#        # Get the details of the selected item first so we will be able
#        # to re-select it at the end of the refresh
#        if job != self._job or self._selected is None:
#            self._selected = None  # A new job
#            filename = host_id = None
#        else:
#            item = self._container.getItem(self._selected)
#            filename = item.getItemProperty('name').getValue()
#            host_id = item.getItemProperty('host_id').getValue()

        self._container = self._newContainer()
        if job:
            session = self._sql_session.open_session()
            if self._workload is None:
                query = session.query(link)
            else:
                query = session.query(link_s)
                query = query.filter(link_s.workload_date == self._workload)
#            new_selection_id = None
            for lnk in query.filter_by(job_id_source=job).all():
                p = path.id2path(session, lnk.job_id_destination,
                                 self._workload)
                destination = p.path.encode('utf-8')
                i = self._container.generateId()
                item = self._container.addItem(i)
                item.getItemProperty('name').setValue(destination)
                item.getItemProperty('mustbe').setValue(
                    status_utils.status2string(lnk.required_status))
                item.getItemProperty('job_id').setValue(
                    long(lnk.job_id_destination))
                item.getItemProperty('link_type').setValue(lnk.required_status)
    #            if f.filename == filename and f.host_id == host_id:
    #                new_selection_id = i
            self._sql_session.close_session(session)
            if self._workload is None or job_main_s.rw_links is True:
                self._btadd.setEnabled(True)
            self._btpreview.setEnabled(True)
        else:
            if self._workload is None or job_main_s.rw_links is True:
                self._btadd.setEnabled(False)
            self._btpreview.setEnabled(False)

        self._table.setContainerDataSource(self._container)
#        if new_selection_id:
#            self._table.select(new_selection_id)
        self._table.setVisibleColumns(['name', 'mustbe'])
        sorted_col = self._table.getSortContainerPropertyId()
        if sorted_col is None:
            self._container.sort(['name'], [True])
        else:
            if self._table.isSortAscending():
                self._container.sort([sorted_col], [True])
            else:
                self._container.sort([sorted_col], [False])
        self._job = job
        self._jobsetIds = jobsetIds[:]


class SelectionHandler(IValueChangeListener):

    """Manage the selection in the link table."""

    def __init__(self, link_obj):
        """Initialize the object.

        @param link_obj:  the associated L{Links} object.
        """
        super(SelectionHandler, self).__init__()
        self._c = link_obj

    def valueChange(self, event):
        """Called when an item has been selected/un-selected.

        @param event:  the listener event.
        """
        AutoRefresh.reset()
        v = event.getProperty().getValue()
        if v is not None:
            self._c._selected = v
            self._c._btedit.setEnabled(True)
            self._c._btdel.setEnabled(True)
        else:
            self._c._selected = None
            self._c._btedit.setEnabled(False)
            self._c._btdel.setEnabled(False)


class AddBtHandler(IClickListener):
    def __init__(self, link_obj):
        super(AddBtHandler, self).__init__()
        self._c = link_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        WindowLink(self._c)


class EditBtHandler(IClickListener):
    def __init__(self, link_obj):
        super(EditBtHandler, self).__init__()
        self._c = link_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        if self._c._selected is not None:
            r = self._c.getLinkDetails(self._c._selected)
            WindowLink(self._c, r)


class PreviewBtHandler(IClickListener):
    def __init__(self, link_obj):
        super(PreviewBtHandler, self).__init__()
        self._c = link_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        WindowPreview(self._c)


class DeleteBtHandler(IClickListener):
    def __init__(self, link_obj):
        super(DeleteBtHandler, self).__init__()
        self._c = link_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        if self._c._selected is not None:
            r = self._c.getLinkDetails(self._c._selected)
            d = DeleteBox(self._c, _("Delete"),
                          _('Are you sure you want to delete this link?'),
                          _('%s (should be %s)') % (r['name'], r['mustbe']))
            d.addListener(DeleteConfirmed(self._c, r['job_id']))


class DeleteConfirmed(BoxCloseListener):

    """Manage link deletion."""

    def __init__(self, link_obj, job_id):
        """Initialize the object.

        @param link_obj:
                    the associated L{Links} object.
        @param job_id:
                    database ID of the linked job/jobset.
        """
        self._c = link_obj
        self._job_id = job_id

    def boxClose(self, event):
        """Called when the user confirmed the deletion."""
        AutoRefresh.reset()
        session = self._c._sql_session.open_session()
        if self._c._workload is None:
            query = session.query(link)
        else:
            query = session.query(link_s)
            query = query.filter(link_s.workload_date == self._c._workload)
        query = query.filter_by(job_id_destination=self._job_id)
        try:
            l = query.filter_by(job_id_source=self._c._job).one()
            session.delete(l)
        except:
            pass
        self._c._sql_session.close_session(session)
        self._c.refresh(self._c._job, self._c._jobsetIds)


class LinkContextMenu(GenericContextMenu):

    """Right button context menu."""

    def __init__(self, link_obj):
        """Initialize the object.

        @param link_obj:
                    the associated L{Links} object.
        """
        super(LinkContextMenu, self).__init__()
        self._c = link_obj

    def getActions(self, target, sender):
        if self._c._job:
            if not target:
                # Background
                return [self._ACTION_ADD]
            return [self._ACTION_EDIT, self._ACTION_ADD, self._ACTION_DELETE]
        else:
            return []

    def handleAction(self, a, sender, target):
        AutoRefresh.reset()
        if a.getCaption() == self._ACTION_EDIT.getCaption():
            r = self._c.getLinkDetails(target)
            WindowLink(self._c, r)

        elif a.getCaption() == self._ACTION_ADD.getCaption():
            WindowLink(self._c)

        elif a.getCaption() == self._ACTION_DELETE.getCaption():
            r = self._c.getLinkDetails(target)
            d = DeleteBox(self._c, _("Delete"),
                          _('Are you sure you want to delete this link?'),
                          _('%s (should be %s)') % (r['name'], r['mustbe']))
            d.addListener(DeleteConfirmed(self._c, r['job_id']))


class TableStyleGenerator(ICellStyleGenerator):

    """ICellStyleGenerator object to style de cells (status of the job)."""

    def __init__(self, c):
        self._c = c

    def getStyle(self, itemId, propertyId):
        if propertyId == 'mustbe':
            item = self._c._container.getItem(itemId)
            s = item.getItemProperty('link_type').getValue()
            return status_utils.status2colorname(s)


class WindowLink(IClickListener):

    """Edit/Add window for links."""

    _bt_captions = [_('Cancel'), _('OK')]

    def __init__(self, link_obj, params=None):
        """Create and display the Edit/Add window.

        @param link_obj:
                    the associated L{Links} object.
        @param params:
                    when the window is used to edit an existing link,
                    this is the link details.  This parameter is
                    a dictionnary as returned by L{Links.getLinksDetails}.
        """
        super(WindowLink, self).__init__()

        self._c = link_obj
        self._params = params

        if params is None:
            title = _('New Link')
        else:
            title = _('Edit Link')
        self._w = Window(title)
        self._w.setWidth('250px')
        self._w.setHeight('405px')

        # VerticalLayout as content by default
        v = self._w.getContent()
        v.setSizeFull()
        v.setMargin(True)
        v.setSpacing(True)

        # Form
        if params and 'job_id' in params:
            t = SelectJob(self._c._jobsetIds,
                          self._c._sql_session, self._c._workload,
                          params['job_id'])
        else:
            t = SelectJob(self._c._jobsetIds,
                          self._c._sql_session, self._c._workload)
        t.setSizeFull()
        self._select_job = t
        v.addComponent(t)
        v.setExpandRatio(t, 1.0)

        s = NativeSelect(_("Must be:"))
        for l in status_utils.status_list():
            s.addItem(status_utils.status2string(l))
        s.setDescription(_("Required status of the linked job/jobset"))
        s.setNullSelectionAllowed(False)
        if params and 'mustbe' in params:
            s.setValue(params['mustbe'])
        else:
            s.setValue(status_utils.status2string(status_utils.COMPLETED))
        self._must = s
        v.addComponent(s)

        # Button box
        h_bt = HorizontalLayout()
        h_bt.setMargin(False)
        h_bt.setSpacing(True)

        for caption in self._bt_captions:
            b = Button(_(caption))
            b.addListener(self, IClickListener)
            h_bt.addComponent(b)

        v.addComponent(h_bt)
        v.setComponentAlignment(h_bt, Alignment.BOTTOM_RIGHT)

        self._c.getWindow().addWindow(self._w)
        self._w.center()

    def buttonClick(self, event):
        AutoRefresh.reset()
        # First button is Cancel
        if event.getButton().getCaption() != _(self._bt_captions[0]):
            # Retrieve the values from the form
            must = status_utils.string2status(self._must.getValue())
            dest_item_ids = self._select_job.getSelected()
            if not dest_item_ids:
                self._c.getWindow().showNotification(
                    _("No job nor jobset selected"),
                    _("<br/>A destination job or jobset must be selected."),
                    Notification.TYPE_ERROR_MESSAGE)
                return

            # Get the job details from the database
            session = self._c._sql_session.open_session()
            try:
                job = sql_get_job(session, self._c._job, self._c._workload)
            except:
                self._c._sql_session.close_session(session)
                self._c.getWindow().showNotification(
                    _("Cannot get the selected job details"),
                    _("<br/>Maybe someone else just removed it."),
                    Notification.TYPE_ERROR_MESSAGE)
                # Close the window
                self._c.getWindow().removeWindow(self._w)
                return

            # Retrieve the link from the database
            for ln in job.links:
                if ln.job_id_destination == dest_item_ids[-1]:
                    # Already in the database
                    # If it was an Add window, print an error message.
                    if self._params is None:
                        self._c._sql_session.close_session(session)
                        self._c.getWindow().showNotification(
                            _("This link already exists"),
                            _("<br/>This new link is already \
                            defined in the database."),
                            Notification.TYPE_ERROR_MESSAGE)
                        return
                    break
            else:
                ln = None
                # Check for loops
                if detect_loop(session, self._c._job, dest_item_ids[-1],
                               self._c._workload):
                    self._c._sql_session.close_session(session)
                    self._c.getWindow().showNotification(
                        _("There is a loop in the links"),
                        _("<br/>A loop has been detected but \
                        loops are not allowed."),
                        Notification.TYPE_ERROR_MESSAGE)
                    return
                if dest_item_ids[-1] == self._c._job:
                    self._c._sql_session.close_session(session)
                    self._c.getWindow().showNotification(
                        _("A job/jobset cannot be linked to itself"),
                        _("<br/>The link points to the job/jobset itself."),
                        Notification.TYPE_ERROR_MESSAGE)
                    return
                if dest_item_ids[-1] in self._c._jobsetIds:
                    self._c._sql_session.close_session(session)
                    self._c.getWindow().showNotification(
                        _("A job/jobset cannot be linked to \
                           one of its parent"),
                        _("<br/>The destination jobset is the parent of the \
                           source job/jobset."),
                        Notification.TYPE_ERROR_MESSAGE)
                    return
                if self._c._job in dest_item_ids:
                    self._c._sql_session.close_session(session)
                    self._c.getWindow().showNotification(
                        _("A jobset cannot be linked to one \
                           of its descendent"),
                        _("<br/>The destination job/jobset is a child of the \
                           source jobset."),
                        Notification.TYPE_ERROR_MESSAGE)
                    return

            # Add a new link
            if self._params is None:
                if self._c._workload is None:
                    l = link(dest_item_ids[-1], must)
                else:
                    l = link_s(dest_item_ids[-1], must, self._c._workload)
                job.links.append(l)

            # Edit a link
            else:
                # Nothing has been changed (maybe the `Must be' flag)
                if dest_item_ids[-1] == self._params['job_id']:
                    if ln:
                        if ln.required_status != must:
                            ln.required_status = must
                    else:
                        # Humm, it should have been in the database...
                        # So recreating the link
                        if self._c._workload is None:
                            l = link(dest_item_ids[-1], must)
                        else:
                            l = link_s(dest_item_ids[-1], must,
                                       self._c._workload)
                        job.links.append(l)

                # The destination job has changed
                else:
                    if ln:
                        self._c._sql_session.close_session(session)
                        self._c.getWindow().showNotification(
                            _("This link already exists"),
                            _("<br/>This new link is already \
                               defined in the database."),
                            Notification.TYPE_ERROR_MESSAGE)
                        return

                    # Find and edit the old record in the database
                    for l in job.links:
                        if l.job_id_destination == self._params['job_id']:
                            l.job_id_destination = dest_item_ids[-1]
                            l.required_status = must
                            break
                    else:
                        # Humm, it should have been in the database...
                        # So recreating the link
                        if self._c._workload is None:
                            l = link(dest_item_ids[-1], must)
                        else:
                            l = link_s(dest_item_ids[-1], must,
                                       self._c._workload)
                        job.links.append(l)
            self._c._sql_session.close_session(session)
            self._c.refresh(self._c._job, self._c._jobsetIds)

        # Close the window
        self._c.getWindow().removeWindow(self._w)


class WindowPreview(IClickListener, IResizeListener):

    """Window to display the dependency tree graph."""

    def __init__(self, link_obj):
        """Create and display the window.

        @param link_obj:
                    the associated L{Links} object.
        """
        super(WindowPreview, self).__init__()
        self._c = link_obj

        window_width = 250
        window_height = 405

        # Whenever the user resizes the preview window (this window),
        # the canvas in the CustomLayout must be resized accordingly.
        # The size of the canvas is computed from the window size.
        # The get the canvas size, self._canvas_width_delta and
        # self._canvas_height_delta are substracted from the window
        # size.
        self._canvas_width_delta = 0
        self._canvas_height_delta = 120
        self._canvas_id = "canvasgraph"

        self._w = Window(_("Link Dependency Graph"))
        self._w.setWidth(window_width, ISizeable.UNITS_PIXELS)
        self._w.setHeight(window_height, ISizeable.UNITS_PIXELS)
        self._w.setImmediate(True)
        self._w.addListener(self, IResizeListener)

        # VerticalLayout as content by default
        v = self._w.getContent()
        v.setSizeFull()
        v.setMargin(False)
        v.setSpacing(False)

        # CustomLayout that contains the canvas which is used to draw
        # the graph (through the Paper.js javascript library)
        l = LinkGraph(self._c._job, self._canvas_id,
                      window_width - self._canvas_width_delta,
                      window_height - self._canvas_height_delta,
                      self._c._sql_session, self._c._workload)
        v.addComponent(l)
        v.setExpandRatio(l, 1.0)

        # Close button
        h_bt = HorizontalLayout()
        h_bt.setMargin(True)
        h_bt.setSpacing(False)
        ok = Button(_('Close'))
        ok.addListener(self, IClickListener)
        h_bt.addComponent(ok)
        v.addComponent(h_bt)
        v.setComponentAlignment(h_bt, Alignment.BOTTOM_RIGHT)

        self._c.getWindow().addWindow(self._w)
        self._w.center()

    def windowResized(self, event):
        """Callback for the resize of the window."""
        h = self._w.getHeight()
        w = self._w.getWidth()
        self._w.getParent().executeJavaScript("""
            var canvas = document.getElementById("%s");
            canvas.paper_obj.view.viewSize = new canvas.paper_obj.Size(%d, %d);
        """ % (self._canvas_id,
               w - self._canvas_width_delta, h - self._canvas_height_delta))
        self._w.requestRepaintAll()

    def buttonClick(self, event):
        """Callback for the close button."""
        AutoRefresh.reset()
        self._c.getWindow().removeWindow(self._w)
