# Schedwi
# Copyright (C) 2012-2015 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Edit/Create job/jobset popup window."""

import time

from muntjac.api import (Window, CheckBox, TextField, TextArea, TabSheet,
                         HorizontalLayout, Button, Alignment, GridLayout,
                         Label, VerticalLayout)
from muntjac.ui.button import IClickListener
from muntjac.ui.window import Notification
from muntjac.terminal.sizeable import ISizeable

import web.jobwidgets
import web.commandargswidget
import web.calendarwidget
import web.envwidget
from simple_queries_job import sql_get_job, sql_get_children
from status_utils import WAITING
from tables.job_main import job_main
from tables.job_main_s import (job_main_s, get_next_id)
from tables.job_status import (job_status, TIME)


(FOCUS_NONE, FOCUS_NAME, FOCUS_DESCR, FOCUS_CMD, FOCUS_ARG,
 FOCUS_OUT, FOCUS_ERR, FOCUS_USER, FOCUS_CGROUP,
 FOCUS_MANUALCMD) = 0, 1, 2, 3, 4, 5, 6, 7, 8, 9


class JobWindow(Window, IClickListener):

    """The job/jobset window."""

    _bt_captions = [_('Cancel'), _('OK')]

    def __init__(self, refresh_obj, jobset_ids, application_window,
                 sql_session, workload=None,
                 job_id=None, is_job=True, x=None, y=None, focus=FOCUS_NAME):
        """Build the job parameters popup.

        @param refresh_obj:
                    the L{web.cutcopypastemenu.CutCopyPasteMenuListener} object
                    to use to refresh the view when the job/jobset has been
                    changed.
        @type refresh_obj: L{web.cutcopypastemenu.CutCopyPasteMenuListener}
        @param jobset_ids:
                    the list of all the jobset IDs from the root jobset up to
                    the parent jobset of the selected object.
        @type jobset_ids: list
        @param application_window:
                    the L{muntjac.api.Application} object (the main window)
        @type application_window: L{muntjac.api.Application}
        @param sql_session:
                    SQLAlchemy session.
        @param workload:
                    workload to consider.
        @param job_id:
                    database ID of the job/jobset to edit.  If None, this is
                    a new job of jobset popup window.
        @param is_job:
                    is the item to create is a job (True) or a jobset (False).
                    Only used if job_id is None.
        @param x:
                the X coordinate of the mouse which is used to position
                the popup.  If None, the popup is centered.
        @param y:
                the Y coordinate of the mouse which is used to position
                the popup.  If None, the popup is centered.
        @param focus:
                widget to focus on.
        @raise sqlalchemy.orm.exc.NoResultFound:
                the given job ID is not in the database.
        """
        super(JobWindow, self).__init__()
        self._refresh_obj = refresh_obj
        self._jobset_ids = jobset_ids
        self._application_window = application_window
        self._sql_session = sql_session
        self._workload = workload
        self._job_id = job_id
        self._is_root = True if not jobset_ids else False

        session = self._sql_session.open_session()
        if self._job_id is not None:
            try:
                job = sql_get_job(session, self._job_id, self._workload)
            except:
                self._sql_session.close_session(session)
                raise
            if job.type:
                self._is_job = True
                self.setCaption(_('Edit job %s (ID: %d)') %
                                (job.name.encode('utf-8'), job_id))
            else:
                self._is_job = False
                if self._is_root:
                    self.setCaption(_('Default parameters'))
                else:
                    self.setCaption(_('Edit jobset %s (ID: %d)') %
                                    (job.name.encode('utf-8'), job_id))
            self._new = False
        else:
            self._is_job = is_job
            if is_job:
                self.setCaption(_('New job'))
            else:
                self.setCaption(_('New jobset'))
            self._new = True

        #
        # Components
        #
        self._enabled = CheckBox()
        self._manual = CheckBox()
        self._name = TextField()
        self._descr = TextArea(_("Description"))
        if self._job_id is not None:
            self._enabled.setValue(bool(job.enabled))
            self._manual.setValue(bool(job.manual))
            self._name.setValue(job.name.encode('utf-8'))
            self._descr.setValue(job.description.encode('utf-8'))
        elif workload is not None:
            # For a job/jobset in a workload set the enabled flag to false
            # by default to prevent the job/jobset to start automatically
            # immediately after been created
            self._enabled.setValue(False)
        else:
            self._enabled.setValue(True)

        #
        # Layout
        #
        if not self._is_root:
            self.setHeight('550px')
            self.setWidth('415px')
        else:
            self.setHeight('535px')
            self.setWidth('356px')
        v = self.getContent()
        v.setSizeFull()
        v.setMargin(False)
        v.setSpacing(False)

        ts = TabSheet()
        ts.setSizeFull()
        v.addComponent(ts)
        v.setExpandRatio(ts, 1.0)

        # Bottom buttons (Cancel, OK)
        h_bt = HorizontalLayout()
        h_bt.setMargin(True)
        h_bt.setSpacing(True)

        for caption in self._bt_captions:
            b = Button(_(caption))
            b.addListener(self, IClickListener)
            h_bt.addComponent(b)

        v.addComponent(h_bt)
        v.setComponentAlignment(h_bt, Alignment.BOTTOM_RIGHT)

        # General
        t1 = GridLayout(3, 6)
        t1.setStyleName("jobdetails")
        t1.setSizeFull()
        t1.setSpacing(True)
        t1.setMargin(True)
        t1.setColumnExpandRatio(1, 1.0)
        t1.setRowExpandRatio(5, 1.0)

        # General/Enabled
        l = Label(_("Enabled:"))
        l.setWidth(None)
        t1.addComponent(l, 0, 0)
        t1.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        self._enabled.setDescription(_("Check for the job to be enabled"))
        t1.addComponent(self._enabled, 1, 0)
        t1.setComponentAlignment(self._enabled, Alignment.MIDDLE_LEFT)
        if workload is not None:
            self._enabled.setEnabled(job_main_s.rw_enabled)

        # General/Name
        if not self._is_root:
            l = Label(_("Name:"))
            l.setWidth(None)
            t1.addComponent(l, 0, 1)
            t1.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
            self._name.setDescription(_("Job/jobset name"))
            self._name.setWidth("100%")
            t1.addComponent(self._name, 1, 1, 2, 1)
            t1.setComponentAlignment(self._name, Alignment.MIDDLE_LEFT)
            if workload is not None:
                self._name.setEnabled(job_main_s.rw_name)

        # General/Host Name
        l = Label(_("Host/Cluster:"))
        l.setWidth(None)
        t1.addComponent(l, 0, 2)
        t1.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        w = web.jobwidgets.Host(self._jobset_ids, session,
                                self._workload, self._job_id)
        t1.addComponent(w.getFieldWidget(), 1, 2)
        w.getFieldWidget().setWidth('100%')
        if workload is not None and job_main_s.rw_job_host is False:
            w.getFieldWidget().setEnabled(False)
        elif not self._is_root:
            t1.addComponent(w.getInheritedCheckbox(), 2, 2)
            t1.setComponentAlignment(w.getInheritedCheckbox(),
                                     Alignment.MIDDLE_RIGHT)
        self._host = w

        # General/Retries
        l = Label(_("Retries:"))
        l.setWidth(None)
        t1.addComponent(l, 0, 3)
        t1.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        w = web.jobwidgets.Retries(self._jobset_ids, session,
                                   self._workload, self._job_id)
        t1.addComponent(w.getFieldWidget(), 1, 3)
        if workload is not None and job_main_s.rw_job_retries is False:
            w.getFieldWidget().setEnabled(False)
        elif not self._is_root:
            t1.addComponent(w.getInheritedCheckbox(), 2, 3)
            t1.setComponentAlignment(w.getInheritedCheckbox(),
                                     Alignment.MIDDLE_RIGHT)
        self._retries = w

        # General/Wait retries
        l = Label(_("Wait retries:"))
        l.setWidth(None)
        t1.addComponent(l, 0, 4)
        t1.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        w = web.jobwidgets.RetryInterval(self._jobset_ids, session,
                                         self._workload, self._job_id)
        t1.addComponent(w.getFieldWidget(), 1, 4)
        if workload is not None and \
           job_main_s.rw_job_retries_interval is False:
            w.getFieldWidget().setEnabled(False)
        elif not self._is_root:
            t1.addComponent(w.getInheritedCheckbox(), 2, 4)
            t1.setComponentAlignment(w.getInheritedCheckbox(),
                                     Alignment.MIDDLE_RIGHT)
        self._retry_interval = w

        # General/Description
        if workload is not None:
            self._descr.setEnabled(job_main_s.rw_description)
        if not self._is_root:
            self._descr.setSizeFull()
            t1.addComponent(self._descr, 0, 5, 2, 5)

        # Triggers
        t2 = GridLayout(3, 6)
        t2.setStyleName("jobdetails")
        t2.setWidth(100, ISizeable.UNITS_PERCENTAGE)
        t2.setSpacing(True)
        t2.setMargin(True)
        t2.setColumnExpandRatio(1, 1.0)

        # Triggers/Calendar
        l = Label(_("Calendar:"))
        l.setWidth(None)
        t2.addComponent(l, 0, 0)
        t2.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        w = web.calendarwidget.Calendar(self._jobset_ids,
                                        application_window,
                                        self._sql_session,
                                        self._workload, self._job_id)
        t2.addComponent(w.getFieldWidget(), 1, 0)
        if workload is not None and job_main_s.rw_cal_id is False:
            w.getFieldWidget().setEnabled(False)
        elif not self._is_root:
            t2.addComponent(w.getInheritedCheckbox(), 2, 0)
            t2.setComponentAlignment(w.getInheritedCheckbox(),
                                     Alignment.MIDDLE_RIGHT)
        self._cal = w

        # Triggers/Start time
        l = Label(_("Start time:"))
        l.setWidth(None)
        t2.addComponent(l, 0, 1)
        t2.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        w = web.jobwidgets.StartTime(self._jobset_ids, session,
                                     self._workload, self._job_id)
        t2.addComponent(w.getFieldWidget(), 1, 1)
        if workload is not None and job_main_s.rw_start_time is False:
            w.getFieldWidget().setEnabled(False)
        elif not self._is_root:
            t2.addComponent(w.getInheritedCheckbox(), 2, 1)
            t2.setComponentAlignment(w.getInheritedCheckbox(),
                                     Alignment.MIDDLE_RIGHT)
        self._start_time = w

        # Triggers/Start limit
        l = Label(_("Start limit:"))
        l.setWidth(None)
        t2.addComponent(l, 0, 2)
        t2.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        w = web.jobwidgets.StartLimit(self._jobset_ids, session,
                                      self._workload, self._job_id)
        t2.addComponent(w.getFieldWidget(), 1, 2)
        if workload is not None and job_main_s.rw_job_start_limit is False:
            w.getFieldWidget().setEnabled(False)
        elif not self._is_root:
            t2.addComponent(w.getInheritedCheckbox(), 2, 2)
            t2.setComponentAlignment(w.getInheritedCheckbox(),
                                     Alignment.MIDDLE_RIGHT)
        self._start_limit = w

        # Triggers/Max duration
        l = Label(_("Max duration:"))
        l.setWidth(None)
        t2.addComponent(l, 0, 3)
        t2.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        w = web.jobwidgets.MaxDuration(self._jobset_ids, session,
                                       self._workload, self._job_id)
        t2.addComponent(w.getFieldWidget(), 1, 3)
        if workload is not None and job_main_s.rw_job_max_duration is False:
            w.getFieldWidget().setEnabled(False)
        elif not self._is_root:
            t2.addComponent(w.getInheritedCheckbox(), 2, 3)
            t2.setComponentAlignment(w.getInheritedCheckbox(),
                                     Alignment.MIDDLE_RIGHT)
        self._max_duration = w

        # Triggers/Manual
        l = Label(_("Manual:"))
        l.setWidth(None)
        t2.addComponent(l, 0, 4)
        t2.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        self._manual.setDescription(_("Whether a confirmation must be asked to\
                                 operators before starting the job"))
        t2.addComponent(self._manual, 1, 4)
        t2.setComponentAlignment(self._manual, Alignment.MIDDLE_LEFT)
        if workload is not None:
            self._manual.setEnabled(job_main_s.rw_manual)

        # Triggers/Manual Command
        l = Label(_("Command:"))
        l.setWidth(None)
        t2.addComponent(l, 0, 5)
        t2.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        w = web.jobwidgets.ManualCommand(
            self._jobset_ids,
            session,
            _("Command to run to alert of a waiting manual job. If empty no \
               command will be run. The following environment variables are \
               set: SCHEDWI_TEMPLATE (the path to a text file that contains \
               the message to send to operators), SCHEDWI_JOBPATH, \
               SCHEDWI_JOBID, SCHEDWI_START_TIME, SCHEDWI_START_TIME_EPOCH, \
               SCHEDWI_TIME_LIMIT, SCHEDWI_TIME_LIMIT_EPOCH, SCHEDWI_URN and \
               SCHEDWI_PASSWORD"),
            self._workload, self._job_id)
        t2.addComponent(w.getFieldWidget(), 1, 5)
        if workload is not None and job_main_s.rw_job_manual_command is False:
            w.getFieldWidget().setEnabled(False)
        elif not self._is_root:
            t2.addComponent(w.getInheritedCheckbox(), 2, 5)
            t2.setComponentAlignment(w.getInheritedCheckbox(),
                                     Alignment.MIDDLE_RIGHT)
        self._manual_command = w

        # Command
        t3 = GridLayout(3, 8)
        t3.setStyleName("jobdetails")
        t3.setSizeFull()
        t3.setSpacing(True)
        t3.setMargin(True)
        t3.setColumnExpandRatio(1, 1.0)
        t3.setRowExpandRatio(7, 1.0)

        # Command/Command
        l = Label(_("Command:"))
        l.setWidth(None)
        t3.addComponent(l, 0, 0)
        t3.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        w = web.jobwidgets.Command(self._jobset_ids,
                                   session,
                                   _("Command to run (without its arguments) \
                                      If empty no command will be run"),
                                   self._workload,
                                   self._job_id)
        t3.addComponent(w.getFieldWidget(), 1, 0)
        if workload is not None and job_main_s.rw_job_command is False:
            w.getFieldWidget().setEnabled(False)
        elif not self._is_root:
            t3.addComponent(w.getInheritedCheckbox(), 2, 0)
            t3.setComponentAlignment(w.getInheritedCheckbox(),
                                     Alignment.MIDDLE_RIGHT)
        self._command = w

        # Command/Username
        l = Label(_("Username:"))
        l.setWidth(None)
        t3.addComponent(l, 0, 1)
        t3.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        w = web.jobwidgets.Username(
            self._jobset_ids,
            session,
            _("System user name used to run the command"),
            self._workload,
            self._job_id)
        t3.addComponent(w.getFieldWidget(), 1, 1)
        if workload is not None and job_main_s.rw_job_username is False:
            w.getFieldWidget().setEnabled(False)
        elif not self._is_root:
            t3.addComponent(w.getInheritedCheckbox(), 2, 1)
            t3.setComponentAlignment(w.getInheritedCheckbox(),
                                     Alignment.MIDDLE_RIGHT)
        self._username = w

        # Command/stdout
        l = Label(_("Standard output file:"))
        l.setWidth(None)
        t3.addComponent(l, 0, 2)
        t3.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        w = web.jobwidgets.Stdout(self._jobset_ids,
                                  session,
                                  _("Output file name (empty means no file)"),
                                  self._workload,
                                  self._job_id)
        t3.addComponent(w.getFieldWidget(), 1, 2)
        if workload is not None and job_main_s.rw_job_file_out is False:
            w.getFieldWidget().setEnabled(False)
        elif not self._is_root:
            t3.addComponent(w.getInheritedCheckbox(), 2, 2)
            t3.setComponentAlignment(w.getInheritedCheckbox(),
                                     Alignment.MIDDLE_RIGHT)
        self._stdout = w

        # Command/stderr
        l = Label(_("Standard error file:"))
        l.setWidth(None)
        t3.addComponent(l, 0, 3)
        t3.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        w = web.jobwidgets.Stderr(self._jobset_ids,
                                  session,
                                  _("Error file name (empty means no file)"),
                                  self._workload,
                                  self._job_id)
        t3.addComponent(w.getFieldWidget(), 1, 3)
        if workload is not None and job_main_s.rw_job_file_err is False:
            w.getFieldWidget().setEnabled(False)
        elif not self._is_root:
            t3.addComponent(w.getInheritedCheckbox(), 2, 3)
            t3.setComponentAlignment(w.getInheritedCheckbox(),
                                     Alignment.MIDDLE_RIGHT)
        self._stderr = w

        # Command/Error code
        l = Label(_("Success return code:"))
        l.setWidth(None)
        t3.addComponent(l, 0, 4)
        t3.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        w = web.jobwidgets.ReturnCode(self._jobset_ids, session,
                                      self._workload, self._job_id)
        t3.addComponent(w.getFieldWidget(), 1, 4)
        if workload is not None and \
           job_main_s.rw_job_success_return_code is False:
            w.getFieldWidget().setEnabled(False)
        elif not self._is_root:
            t3.addComponent(w.getInheritedCheckbox(), 2, 4)
            t3.setComponentAlignment(w.getInheritedCheckbox(),
                                     Alignment.MIDDLE_RIGHT)
        self._return_code = w

        # Command/Detach
        l = Label(_("Start in detach mode:"))
        l.setWidth(None)
        t3.addComponent(l, 0, 5)
        t3.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        w = web.jobwidgets.Detach(self._jobset_ids, session,
                                  self._workload, self._job_id)
        t3.addComponent(w.getFieldWidget(), 1, 5)
        if workload is not None and job_main_s.rw_job_detach is False:
            w.getFieldWidget().setEnabled(False)
        elif not self._is_root:
            t3.addComponent(w.getInheritedCheckbox(), 2, 5)
            t3.setComponentAlignment(w.getInheritedCheckbox(),
                                     Alignment.MIDDLE_RIGHT)
        self._detach = w

        # Command/cgroup
        l = Label(_("Linux Control Group:"))
        l.setWidth(None)
        t3.addComponent(l, 0, 6)
        t3.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        w = web.jobwidgets.CGroup(self._jobset_ids,
                                  session,
                                  _("Control Group name"),
                                  self._workload,
                                  self._job_id)
        t3.addComponent(w.getFieldWidget(), 1, 6)
        if workload is not None and job_main_s.rw_job_control_group is False:
            w.getFieldWidget().setEnabled(False)
        elif not self._is_root:
            t3.addComponent(w.getInheritedCheckbox(), 2, 6)
            t3.setComponentAlignment(w.getInheritedCheckbox(),
                                     Alignment.MIDDLE_RIGHT)
        self._cgroup = w

        # Command/Arguments
        w = web.commandargswidget.CommandArgs(self._jobset_ids,
                                              application_window, session,
                                              self._workload, self._job_id)
        w.getFieldWidget().setWidth("100%")
        t3.addComponent(w.getFieldWidget(), 0, 7, 1, 7)
        if workload is not None and job_main_s.rw_job_arguments is False:
            w.getFieldWidget().setEnabled(False)
        elif not self._is_root:
            t3.addComponent(w.getInheritedCheckbox(), 2, 7)
            t3.setComponentAlignment(w.getInheritedCheckbox(),
                                     Alignment.MIDDLE_RIGHT)
        self._args = w

        # Environments
        t4 = VerticalLayout()
        t4.setSizeFull()
        t4.setSpacing(True)
        t4.setMargin(True)

        # Environments/Load user environment (.profile, ...)
        le = HorizontalLayout()
        le.setWidth('100%')
        le.setSpacing(True)
        le.setMargin(False)
        t4.addComponent(le)

        w = web.jobwidgets.LoadEnv(self._jobset_ids, session,
                                   self._workload, self._job_id)
        le.addComponent(w.getFieldWidget())
        le.setExpandRatio(w.getFieldWidget(), 1.0)
        if workload is not None and job_main_s.rw_job_loadenv is False:
            w.getFieldWidget().setEnabled(False)
        elif not self._is_root:
            le.addComponent(w.getInheritedCheckbox())
        self._loadenv = w

        self._sql_session.close_session(session)

        # Environments/Environment groups
        e = web.envwidget.EnvWidget(self._jobset_ids, application_window,
                                    self._host,
                                    self._sql_session, self._workload,
                                    self._job_id)
        t4.addComponent(e)
        t4.setExpandRatio(e, 1.0)
        if workload is not None and job_main_s.rw_job_environment is False:
            e.setEnabled(False)
        self._env = e

        # Add the components to the tab sheet
        ts.addTab(t1, _('General'))
        ts.addTab(t2, _('Triggers'))
        ts.addTab(t3, _('Command'))
        ts.addTab(t4, _('Environments'))
        self._ts = ts
        self._t1 = t1
        self._t2 = t2
        self._t3 = t3
        self._t4 = t4

        application_window.addWindow(self)
        if x is None or y is None:
            self.center()
        else:
            self.setPositionX(x)
            self.setPositionY(y)

        if focus == FOCUS_NAME:
            self._ts.setSelectedTab(self._t1)
            self._name.focus()
        elif focus == FOCUS_DESCR:
            self._ts.setSelectedTab(self._t1)
            self._descr.focus()
        elif focus == FOCUS_CMD:
            self._ts.setSelectedTab(self._t3)
            self._command.setFocus()
        elif focus == FOCUS_USER:
            self._ts.setSelectedTab(self._t3)
            self._username.setFocus()
        elif focus == FOCUS_OUT:
            self._ts.setSelectedTab(self._t3)
            self._stdout.setFocus()
        elif focus == FOCUS_ERR:
            self._ts.setSelectedTab(self._t3)
            self._stderr.setFocus()
        elif focus == FOCUS_CGROUP:
            self._ts.setSelectedTab(self._t3)
            self._cgroup.setFocus()
        elif focus == FOCUS_MANUALCMD:
            self._ts.setSelectedTab(self._t2)
            self._manual_command.setFocus()
        elif focus == FOCUS_ARG:
            self._ts.setSelectedTab(self._t3)
            self._args.setFocus()

    def buttonClick(self, event):
        # First button is Cancel
        if event.getButton().getCaption() != _(self._bt_captions[0]):
            # Name sanity checks
            name = self._name.getValue().strip()
            if not name:
                self._application_window.showNotification(
                    _("Name is empty"),
                    _("<br/>The job or jobset name cannot \
                       be empty or contain just spaces."),
                    Notification.TYPE_ERROR_MESSAGE)
                self._ts.setSelectedTab(self._t1)
                self._name.focus()
                return
            session = self._sql_session.open_session()
            if not self._is_root:
                jobs = filter(lambda i: i.name.encode('utf-8') == name,
                              sql_get_children(session, self._jobset_ids[-1],
                                               workload=self._workload))
                if jobs and (self._job_id is None or
                             jobs[0].id != self._job_id):
                    self._sql_session.cancel_session(session)
                    self._application_window.showNotification(
                        _("The name is already used"),
                        _("<br/>The specified name is already taken."),
                        Notification.TYPE_ERROR_MESSAGE)
                    self._ts.setSelectedTab(self._t1)
                    self._name.focus()
                    return
            enabled = self._enabled.getValue()
            manual = self._manual.getValue()
            description = self._descr.getValue()
            cal_id = self._cal.get_value()
            try:
                start_time = self._start_time.get_value()
            except web.jobwidgets.StartTimeError as e:
                self._sql_session.cancel_session(session)
                self._application_window.showNotification(
                    _("Wrong value for the start time"),
                    '<br/>' + str(e),
                    Notification.TYPE_ERROR_MESSAGE)
                self._ts.setSelectedTab(self._t1)
                return
            if self._job_id is None:
                if self._workload is None:
                    job = job_main(self._jobset_ids[-1], name,
                                   1 if self._is_job else 0, int(enabled), 1,
                                   int(manual),
                                   description, 0, 0, cal_id, start_time)
                else:
                    job = job_main_s(get_next_id(session, self._workload),
                                     self._jobset_ids[-1], name,
                                     1 if self._is_job else 0, int(enabled), 1,
                                     int(manual),
                                     description, 0, 0, cal_id, start_time,
                                     self._workload)
                    job.job_status = job_status(status=WAITING,
                                                time_status_set=int(
                                                    time.time()),
                                                retry_num=0,
                                                error_msg=_("Manually added"),
                                                duration=0,
                                                host_id=None,
                                                wait_reason=TIME,
                                                workload_date=self._workload)
            else:
                try:
                    job = sql_get_job(session, self._job_id, self._workload)
                except:
                    self._sql_session.cancel_session(session)
                    self._application_window.showNotification(
                        _("Cannot get the edited job/jobset details"),
                        _("<br/>Maybe someone else just removed it."),
                        Notification.TYPE_ERROR_MESSAGE)
                    # Close the window
                    self._application_window.removeWindow(self)
                    return
                job.name = name.decode('utf-8')
                job.enabled = int(enabled)
                job.manual = int(manual)
                job.description = description.decode('utf-8')
                job.cal_id = cal_id
                job.start_time = start_time
            msg = self._start_limit.updateJob(job)
            if msg:
                self._sql_session.cancel_session(session)
                self._application_window.showNotification(
                    _("Wrong value for the start limit"),
                    '<br/>' + msg,
                    Notification.TYPE_ERROR_MESSAGE)
                self._ts.setSelectedTab(self._t1)
                return
            msg = self._max_duration.updateJob(job)
            if msg:
                self._sql_session.cancel_session(session)
                self._application_window.showNotification(
                    _("Wrong value for the max duration"),
                    '<br/>' + msg,
                    Notification.TYPE_ERROR_MESSAGE)
                self._ts.setSelectedTab(self._t1)
                return
            msg = self._retries.updateJob(job)
            if msg:
                self._sql_session.cancel_session(session)
                self._application_window.showNotification(
                    _("Wrong value for the number of retries"),
                    '<br/>' + msg,
                    Notification.TYPE_ERROR_MESSAGE)
                self._ts.setSelectedTab(self._t1)
                return
            msg = self._retry_interval.updateJob(job)
            if msg:
                self._sql_session.cancel_session(session)
                self._application_window.showNotification(
                    _("Wrong value for the retry interval"),
                    '<br/>' + msg,
                    Notification.TYPE_ERROR_MESSAGE)
                self._ts.setSelectedTab(self._t1)
                return
            self._host.updateJob(job)
            self._command.updateJob(job)
            self._username.updateJob(job)
            self._stdout.updateJob(job)
            self._stderr.updateJob(job)
            self._cgroup.updateJob(job)
            self._args.updateJob(job)
            self._env.updateJob(job)
            self._loadenv.updateJob(job)
            self._detach.updateJob(job)
            self._manual_command.updateJob(job)
            msg = self._return_code.updateJob(job)
            if msg:
                self._sql_session.cancel_session(session)
                self._application_window.showNotification(
                    _("Wrong value for the return code"),
                    '<br/>' + msg,
                    Notification.TYPE_ERROR_MESSAGE)
                self._ts.setSelectedTab(self._t3)
                return
            if self._job_id is None:
                session.add(job)
            self._sql_session.close_session(session)
            self._refresh_obj.repaint()
        # Close the window
        self._application_window.removeWindow(self)
