# Schedwi
# Copyright (C) 2012-2015 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Find an object."""

from muntjac.api import (HorizontalLayout, Window, Button, Table,
                         Alignment, Label, TextField)
from muntjac.ui.button import IClickListener
from muntjac.event.item_click_event import IItemClickListener
from muntjac.data.util.indexed_container import IndexedContainer
from muntjac.terminal.theme_resource import ThemeResource
from muntjac.event.shortcut_listener import ShortcutListener
from muntjac.event.shortcut_action import KeyCode

import path
import path_cal
import simple_queries_job
import environments_utils
from tables.job_main import (job_main, ROOT_JOBSET_ID)
from tables.job_main_s import job_main_s
from tables.calendars import calendars
from tables.calendars_s import calendars_s
from tables.constraint_file import constraint_file
from tables.constraint_file_s import constraint_file_s
from tables.environments import environments
from tables.environments_s import environments_s
from tables.environment_var import environment_var
from tables.environment_var_s import environment_var_s
from tables.hosts import hosts
from tables.clusters import clusters
from tables.job_arguments import job_arguments
from tables.job_arguments_s import job_arguments_s
from tables.job_command import job_command
from tables.job_command_s import job_command_s
from tables.job_file_err import job_file_err
from tables.job_file_err_s import job_file_err_s
from tables.job_control_group import job_control_group
from tables.job_control_group_s import job_control_group_s
from tables.job_file_out import job_file_out
from tables.job_file_out_s import job_file_out_s
from tables.job_username import job_username
from tables.job_username_s import job_username_s
from tables.job_manual_command import job_manual_command
from tables.job_manual_command_s import job_manual_command_s
import web.jobwindow
from web.hostwindow import HostWindow
from web.envwindow import EnvWindow
from web.jobwindow import JobWindow
from web.calendarwindow import CalendarWindow


class Find(HorizontalLayout, IClickListener):

    """Find text entry widget."""

    _LIMIT = 20
    _NONE, _JOB, _ENV, _HOST, _CAL, _CLUSTER = 0, 1, 2, 3, 4, 5
    (_JOB_FILE,
     _ENV_NAME, _ENV_DESCR, _ENV_VAR_NAME, _ENV_VAR_VAL,
     _CAL_NAME, _CAL_DESCR,
     _HOST_NAME, _HOST_PORT, _HOST_DESCR,
     _CLUSTER_NAME, _CLUSTER_DESCR) = (1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12)

    def __init__(self, parent_widget, application, sql_session, workload=None):
        """Create the widget.

        @param parent_widget:
                    a parent widget used to add the popup result window.
        @param application:
                    the L{web.main.Main} object.
        @type application:
                    L{muntjac.api.Application}
        @param sql_session:
                    SQLAlchemy session.
        @param workload:
                    workload to consider.
        """
        super(Find, self).__init__()
        self._application = application
        self._parent_widget = parent_widget
        self._sql_session = sql_session
        self._workload = workload

        self.setSpacing(True)
        t = TextField()
        t.setInputPrompt(_("Search"))
        t.addShortcutListener(EnterShortcutLaunchSearch(self))
        t.setWidth('100%')
        self.addComponent(t)
        self.setExpandRatio(t, 1.0)
        self._text = t

        b = Button()
        b.setIcon(ThemeResource('icons/search.png'))
        b.setDescription(_("Search"))
        b.addListener(self, IClickListener)
        self.addComponent(b)

    def _new_container(self):
        """Create and return a new container data source.

        @return:    the new empty container data source.
        @rtype:
            L{muntjac.data.util.indexed_container.IndexedContainer}
        """
        jContainer = IndexedContainer()
        jContainer.addContainerProperty('label', str, None)
        jContainer.addContainerProperty('object', str, None)
        jContainer.addContainerProperty('value', str, None)
        jContainer.addContainerProperty('type', int, None)
        jContainer.addContainerProperty('id', long, None)
        jContainer.addContainerProperty('attr', int, None)
        return jContainer

    def _add_result(self, container, label, obj_name,
                    obj_type, obj_id,
                    attr,
                    value=None):
        """Callback to store results in the container.

        @param container:
                    the container to add item to.
        @type container:
                    L{muntjac.data.util.indexed_container.IndexedContainer}
        @param label:
                    the label (type name of the found item)
        @param obj_name:
                    the name of the found item.
        @param obj_type:
                    type of object (_NONE, _JOB, _ENV, _HOST, _CAL or _CLUSTER)
        @param attr:
                    in the object, field name in which the item has been found
                    (_ENV_NAME, _ENV_DESCR, _ENV_VAR_NAME, _ENV_VAR_VAL,
                    _CAL_NAME, ...)
        @param value:
                    value of the field
        """
        item = container.addItem(container.generateId())
        item.getItemProperty('label').setValue(label.encode('utf-8'))
        item.getItemProperty('object').setValue(obj_name.encode('utf-8'))
        item.getItemProperty('type').setValue(obj_type)
        item.getItemProperty('id').setValue(long(obj_id))
        item.getItemProperty('attr').setValue(attr)
        if value is not None:
            item.getItemProperty('value').setValue(value.encode('utf-8'))

    def buttonClick(self, event):
        self.do_search(self._text.getValue())

    def do_search(self, search_string):
        """Search in the database."""
        v = str(search_string)
        if not v.strip():
            return
        tofind = "%" + \
                 v.replace("*", "%").replace("?", "_").decode('utf-8') + "%"

        c = self._new_container()
        session = self._sql_session.open_session()
        if self._workload is None:
            # job_main
            q = session.query(job_main).filter(job_main.name.ilike(tofind))
            i = 0
            for j in q.limit(self._LIMIT):
                self._add_result(
                    container=c,
                    label=_("Job name") if j.type else _("Jobset name"),
                    obj_name=unicode(path.id2path(session, j.id,
                                     self._workload)),
                    obj_type=self._JOB,
                    obj_id=j.id,
                    attr=web.jobwindow.FOCUS_NAME)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more jobs/jobsets"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=web.jobwindow.FOCUS_NAME)
            q = session.query(job_main).filter(
                job_main.description.ilike(tofind))
            i = 0
            for j in q.limit(self._LIMIT):
                self._add_result(
                    container=c,
                    label=_("Job description") if j.type
                    else _("Jobset description"),
                    obj_name=unicode(path.id2path(session, j.id,
                                     self._workload)),
                    obj_type=self._JOB,
                    obj_id=j.id,
                    attr=web.jobwindow.FOCUS_DESCR)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more job/jobset description"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=web.jobwindow.FOCUS_DESCR)
            # job_command
            q = session.query(job_command).filter(
                job_command.command.ilike(tofind))
            i = 0
            for j in q.limit(self._LIMIT):
                job = simple_queries_job.sql_get_job(session, j.job_id,
                                                     self._workload)
                self._add_result(
                    container=c,
                    label=_("Job command") if job.type
                    else _("Jobset command"),
                    obj_name=unicode(path.id2path(session, j.job_id,
                                                  self._workload)),
                    obj_type=self._JOB,
                    obj_id=j.job_id,
                    attr=web.jobwindow.FOCUS_CMD,
                    value=j.command)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more job/jobset commands"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=web.jobwindow.FOCUS_CMD)
            # job_manual_command
            q = session.query(job_manual_command).filter(
                job_manual_command.manual_command.ilike(tofind))
            i = 0
            for j in q.limit(self._LIMIT):
                job = simple_queries_job.sql_get_job(session, j.job_id,
                                                     self._workload)
                self._add_result(
                    container=c,
                    label=_("Alert command"),
                    obj_name=unicode(path.id2path(session, j.job_id,
                                                  self._workload)),
                    obj_type=self._JOB,
                    obj_id=j.job_id,
                    attr=web.jobwindow.FOCUS_MANUALCMD,
                    value=j.manual_command)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more job/jobset alert commands"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=web.jobwindow.FOCUS_MANUALCMD)
            # job_arguments
            q = session.query(job_arguments).filter(
                job_arguments.argument.ilike(tofind))
            list_ids = list()
            i = 0
            for j in q.limit(self._LIMIT):
                if j.job_id not in list_ids:
                    job = simple_queries_job.sql_get_job(session, j.job_id,
                                                         self._workload)
                    self._add_result(
                        container=c,
                        label=_("Job command arg") if job.type
                        else _("Jobset command arg"),
                        obj_name=unicode(path.id2path(session, j.job_id,
                                                      self._workload)),
                        obj_type=self._JOB,
                        obj_id=j.job_id,
                        attr=web.jobwindow.FOCUS_ARG,
                        value=j.argument)
                    list_ids.append(j.job_id)
                    i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more job/jobset command args"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=web.jobwindow.FOCUS_ARG)
            # job_file_out
            q = session.query(job_file_out).filter(
                job_file_out.file_out.ilike(tofind))
            i = 0
            for j in q.limit(self._LIMIT):
                job = simple_queries_job.sql_get_job(session, j.job_id,
                                                     self._workload)
                self._add_result(
                    container=c,
                    label=_("Job output file") if job.type
                    else _("Jobset output file"),
                    obj_name=unicode(path.id2path(session, j.job_id,
                                                  self._workload)),
                    obj_type=self._JOB,
                    obj_id=j.job_id,
                    attr=web.jobwindow.FOCUS_OUT,
                    value=j.file_out)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more job/jobset output file"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=web.jobwindow.FOCUS_OUT)
            # job_file_err
            q = session.query(job_file_err).filter(
                job_file_err.file_err.ilike(tofind))
            i = 0
            for j in q.limit(self._LIMIT):
                job = simple_queries_job.sql_get_job(session, j.job_id,
                                                     self._workload)
                self._add_result(
                    container=c,
                    label=_("Job error file") if job.type
                    else _("Jobset error file"),
                    obj_name=unicode(path.id2path(session, j.job_id,
                                                  self._workload)),
                    obj_type=self._JOB,
                    obj_id=j.job_id,
                    attr=web.jobwindow.FOCUS_ERR,
                    value=j.file_err)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more job/jobset error file"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=web.jobwindow.FOCUS_ERR)
            # job_control_group
            q = session.query(job_control_group).filter(
                job_control_group.control_group.ilike(tofind))
            i = 0
            for j in q.limit(self._LIMIT):
                job = simple_queries_job.sql_get_job(session, j.job_id,
                                                     self._workload)
                self._add_result(
                    container=c,
                    label=_("Job Control Group") if job.type
                    else _("Jobset Control Group"),
                    obj_name=unicode(path.id2path(session, j.job_id,
                                                  self._workload)),
                    obj_type=self._JOB,
                    obj_id=j.job_id,
                    attr=web.jobwindow.FOCUS_CGROUP,
                    value=j.control_group)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more job/jobset control groups"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=web.jobwindow.FOCUS_CGROUP)
            # job_username
            q = session.query(job_username).filter(
                job_username.username.ilike(tofind))
            i = 0
            for j in q.limit(self._LIMIT):
                job = simple_queries_job.sql_get_job(session, j.job_id,
                                                     self._workload)
                self._add_result(
                    container=c,
                    label=_("Job user") if job.type else _("Jobset user"),
                    obj_name=unicode(path.id2path(session, j.job_id,
                                                  self._workload)),
                    obj_type=self._JOB,
                    obj_id=j.job_id,
                    attr=web.jobwindow.FOCUS_USER,
                    value=j.username)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more job/jobset users"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=web.jobwindow.FOCUS_USER)
            # constraint_file
            q = session.query(constraint_file).filter(
                constraint_file.filename.ilike(tofind))
            i = 0
            for j in q.limit(self._LIMIT):
                job = simple_queries_job.sql_get_job(session, j.job_id,
                                                     self._workload)
                self._add_result(
                    container=c,
                    label=_("Job constraint file") if job.type
                    else _("Jobset constraint file"),
                    obj_name=unicode(path.id2path(session, j.job_id,
                                                  self._workload)),
                    obj_type=self._JOB,
                    obj_id=j.job_id,
                    attr=self._JOB_FILE,
                    value=j.filename)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more job/jobset constraint files"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=self._JOB_FILE)
            # environments
            q = session.query(environments).filter(
                environments.name.ilike(tofind))
            i = 0
            for j in q.limit(self._LIMIT):
                self._add_result(
                    container=c,
                    label=_("Environment name"),
                    obj_name=j.name,
                    obj_type=self._ENV,
                    obj_id=j.id,
                    attr=self._ENV_NAME)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more environments"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=self._ENV_NAME)
            q = session.query(environments).filter(
                environments.description.ilike(tofind))
            i = 0
            for j in q.limit(self._LIMIT):
                self._add_result(
                    container=c,
                    label=_("Environment description"),
                    obj_name=j.name,
                    obj_type=self._ENV,
                    obj_id=j.id,
                    attr=self._ENV_DESCR)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more environment descriptions"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=self._ENV_DESCR)
            # environment_var
            q = session.query(environment_var).filter(
                environment_var.env_key.ilike(tofind))
            i = 0
            for j in q.limit(self._LIMIT):
                e = environments_utils.get_env(session, j.env_id,
                                               self._workload)
                self._add_result(
                    container=c,
                    label=_("Environment variable name"),
                    obj_name=e.name,
                    obj_type=self._ENV,
                    obj_id=j.env_id,
                    attr=self._ENV_VAR_NAME,
                    value=j.env_key + '=' + j.env_value)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more environment variables"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=self._ENV_VAR_NAME)
            q = session.query(environment_var).filter(
                environment_var.env_value.ilike(tofind))
            i = 0
            for j in q.limit(self._LIMIT):
                e = environments_utils.get_env(session, j.env_id,
                                               self._workload)
                self._add_result(
                    container=c,
                    label=_("Environment variable value"),
                    obj_name=e.name,
                    obj_type=self._ENV,
                    obj_id=j.env_id,
                    attr=self._ENV_VAR_VAL,
                    value=j.env_key + '=' + j.env_value)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more environment variables"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=self._ENV_VAR_NAME)
            # calendars
            q = session.query(calendars).filter(calendars.name.ilike(tofind))
            i = 0
            for j in q.limit(self._LIMIT):
                self._add_result(
                    container=c,
                    label=_("Calendar directory name") if j.entry_type
                    else _("Calendar name"),
                    obj_name=unicode(
                        path_cal.PathCal(session,
                                         id=j.id,
                                         workload=self._workload)),
                    obj_type=self._CAL,
                    obj_id=j.id,
                    attr=self._CAL_NAME)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more calendars"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=self._CAL_NAME)
            q = session.query(calendars).filter(
                calendars.description.ilike(tofind))
            i = 0
            for j in q.limit(self._LIMIT):
                self._add_result(
                    container=c,
                    label=_("Calendar directory description") if j.entry_type
                    else _("Calendar description"),
                    obj_name=unicode(
                        path_cal.PathCal(session,
                                         id=j.id,
                                         workload=self._workload)),
                    obj_type=self._CAL,
                    obj_id=j.id,
                    attr=self._CAL_DESCR)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more calendar descriptions"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=self._CAL_DESCR)
        else:
            # job_main
            q = session.query(job_main_s).filter(job_main_s.name.ilike(tofind))
            q = q.filter(job_main_s.workload_date == self._workload)
            i = 0
            for j in q.limit(self._LIMIT):
                self._add_result(
                    container=c,
                    label=_("Job name") if j.type else _("Jobset name"),
                    obj_name=unicode(path.id2path(session, j.id,
                                     self._workload)),
                    obj_type=self._JOB,
                    obj_id=j.id,
                    attr=web.jobwindow.FOCUS_NAME)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more jobs/jobsets"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=web.jobwindow.FOCUS_NAME)
            q = session.query(job_main_s).filter(
                job_main_s.description.ilike(tofind))
            q = q.filter(job_main_s.workload_date == self._workload)
            i = 0
            for j in q.limit(self._LIMIT):
                self._add_result(
                    container=c,
                    label=_("Job description") if j.type
                    else _("Jobset description"),
                    obj_name=unicode(path.id2path(session, j.id,
                                     self._workload)),
                    obj_type=self._JOB,
                    obj_id=j.id,
                    attr=web.jobwindow.FOCUS_DESCR)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more job/jobset description"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=web.jobwindow.FOCUS_DESCR)
            # job_command
            q = session.query(job_command_s).filter(
                job_command_s.command.ilike(tofind))
            q = q.filter(job_command_s.workload_date == self._workload)
            i = 0
            for j in q.limit(self._LIMIT):
                job = simple_queries_job.sql_get_job(session, j.job_id,
                                                     self._workload)
                self._add_result(
                    container=c,
                    label=_("Job command") if job.type
                    else _("Jobset command"),
                    obj_name=unicode(path.id2path(session, j.job_id,
                                                  self._workload)),
                    obj_type=self._JOB,
                    obj_id=j.job_id,
                    attr=web.jobwindow.FOCUS_CMD,
                    value=j.command)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more job/jobset commands"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=web.jobwindow.FOCUS_CMD)
            # job_manual_command
            q = session.query(job_manual_command_s).filter(
                job_manual_command_s.manual_command.ilike(tofind))
            q = q.filter(job_manual_command_s.workload_date == self._workload)
            i = 0
            for j in q.limit(self._LIMIT):
                job = simple_queries_job.sql_get_job(session, j.job_id,
                                                     self._workload)
                self._add_result(
                    container=c,
                    label=_("Alert command"),
                    obj_name=unicode(path.id2path(session, j.job_id,
                                                  self._workload)),
                    obj_type=self._JOB,
                    obj_id=j.job_id,
                    attr=web.jobwindow.FOCUS_MANUALCMD,
                    value=j.manual_command)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more job/jobset alert commands"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=web.jobwindow.FOCUS_MANUALCMD)
            # job_arguments
            q = session.query(job_arguments_s).filter(
                job_arguments_s.argument.ilike(tofind))
            q = q.filter(job_arguments_s.workload_date == self._workload)
            list_ids = list()
            i = 0
            for j in q.limit(self._LIMIT):
                if j.job_id not in list_ids:
                    job = simple_queries_job.sql_get_job(session, j.job_id,
                                                         self._workload)
                    self._add_result(
                        container=c,
                        label=_("Job command arg") if job.type
                        else _("Jobset command arg"),
                        obj_name=unicode(path.id2path(session, j.job_id,
                                                      self._workload)),
                        obj_type=self._JOB,
                        obj_id=j.job_id,
                        attr=web.jobwindow.FOCUS_ARG,
                        value=j.argument)
                    list_ids.append(j.job_id)
                    i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more job/jobset command args"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=web.jobwindow.FOCUS_ARG)
            # job_file_out
            q = session.query(job_file_out_s).filter(
                job_file_out_s.file_out.ilike(tofind))
            q = q.filter(job_file_out_s.workload_date == self._workload)
            i = 0
            for j in q.limit(self._LIMIT):
                job = simple_queries_job.sql_get_job(session, j.job_id,
                                                     self._workload)
                self._add_result(
                    container=c,
                    label=_("Job output file") if job.type
                    else _("Jobset output file"),
                    obj_name=unicode(path.id2path(session, j.job_id,
                                                  self._workload)),
                    obj_type=self._JOB,
                    obj_id=j.job_id,
                    attr=web.jobwindow.FOCUS_OUT,
                    value=j.file_out)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more job/jobset output file"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=web.jobwindow.FOCUS_OUT)
            # job_file_err
            q = session.query(job_file_err_s).filter(
                job_file_err_s.file_err.ilike(tofind))
            q = q.filter(job_file_err_s.workload_date == self._workload)
            i = 0
            for j in q.limit(self._LIMIT):
                job = simple_queries_job.sql_get_job(session, j.job_id,
                                                     self._workload)
                self._add_result(
                    container=c,
                    label=_("Job error file") if job.type
                    else _("Jobset error file"),
                    obj_name=unicode(path.id2path(session, j.job_id,
                                                  self._workload)),
                    obj_type=self._JOB,
                    obj_id=j.job_id,
                    attr=web.jobwindow.FOCUS_ERR,
                    value=j.file_err)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more job/jobset error file"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=web.jobwindow.FOCUS_ERR)
            # job_control_group
            q = session.query(job_control_group_s).filter(
                job_control_group_s.control_group.ilike(tofind))
            q = q.filter(job_control_group_s.workload_date == self._workload)
            i = 0
            for j in q.limit(self._LIMIT):
                job = simple_queries_job.sql_get_job(session, j.job_id,
                                                     self._workload)
                self._add_result(
                    container=c,
                    label=_("Job Control Group") if job.type
                    else _("Jobset Control Group"),
                    obj_name=unicode(path.id2path(session, j.job_id,
                                                  self._workload)),
                    obj_type=self._JOB,
                    obj_id=j.job_id,
                    attr=web.jobwindow.FOCUS_CGROUP,
                    value=j.control_group)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more job/jobset Control Groups"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=web.jobwindow.FOCUS_CGROUP)
            # job_username
            q = session.query(job_username_s).filter(
                job_username_s.username.ilike(tofind))
            q = q.filter(job_username_s.workload_date == self._workload)
            i = 0
            for j in q.limit(self._LIMIT):
                job = simple_queries_job.sql_get_job(session, j.job_id,
                                                     self._workload)
                self._add_result(
                    container=c,
                    label=_("Job user") if job.type else _("Jobset user"),
                    obj_name=unicode(path.id2path(session, j.job_id,
                                                  self._workload)),
                    obj_type=self._JOB,
                    obj_id=j.job_id,
                    attr=web.jobwindow.FOCUS_USER,
                    value=j.username)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more job/jobset users"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=web.jobwindow.FOCUS_USER)
            # constraint_file
            q = session.query(constraint_file_s).filter(
                constraint_file_s.filename.ilike(tofind))
            q = q.filter(constraint_file_s.workload_date == self._workload)
            i = 0
            for j in q.limit(self._LIMIT):
                job = simple_queries_job.sql_get_job(session, j.job_id,
                                                     self._workload)
                self._add_result(
                    container=c,
                    label=_("Job constraint file") if job.type
                    else _("Jobset constraint file"),
                    obj_name=unicode(path.id2path(session, j.job_id,
                                                  self._workload)),
                    obj_type=self._JOB,
                    obj_id=j.job_id,
                    attr=self._JOB_FILE,
                    value=j.filename)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more job/jobset constraint files"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=self._JOB_FILE)
            # environments
            q = session.query(environments_s).filter(
                environments_s.name.ilike(tofind))
            q = q.filter(environments_s.workload_date == self._workload)
            i = 0
            for j in q.limit(self._LIMIT):
                self._add_result(
                    container=c,
                    label=_("Environment name"),
                    obj_name=j.name,
                    obj_type=self._ENV,
                    obj_id=j.id,
                    attr=self._ENV_NAME)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more environments"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=self._ENV_NAME)
            q = session.query(environments_s).filter(
                environments_s.description.ilike(tofind))
            q = q.filter(environments_s.workload_date == self._workload)
            i = 0
            for j in q.limit(self._LIMIT):
                self._add_result(
                    container=c,
                    label=_("Environment description"),
                    obj_name=j.name,
                    obj_type=self._ENV,
                    obj_id=j.id,
                    attr=self._ENV_DESCR)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more environment descriptions"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=self._ENV_DESCR)
            # environment_var
            q = session.query(environment_var_s).filter(
                environment_var_s.env_key.ilike(tofind))
            q = q.filter(environment_var_s.workload_date == self._workload)
            i = 0
            for j in q.limit(self._LIMIT):
                e = environments_utils.get_env(session, j.env_id,
                                               self._workload)
                self._add_result(
                    container=c,
                    label=_("Environment variable name"),
                    obj_name=e.name,
                    obj_type=self._ENV,
                    obj_id=j.env_id,
                    attr=self._ENV_VAR_NAME,
                    value=j.env_key + '=' + j.env_value)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more environment variables"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=self._ENV_VAR_NAME)
            q = session.query(environment_var_s).filter(
                environment_var_s.env_value.ilike(tofind))
            q = q.filter(environment_var_s.workload_date == self._workload)
            i = 0
            for j in q.limit(self._LIMIT):
                e = environments_utils.get_env(session, j.env_id,
                                               self._workload)
                self._add_result(
                    container=c,
                    label=_("Environment variable value"),
                    obj_name=e.name,
                    obj_type=self._ENV,
                    obj_id=j.env_id,
                    attr=self._ENV_VAR_VAL,
                    value=j.env_key + '=' + j.env_value)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more environment variables"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=self._ENV_VAR_NAME)
            # calendars
            q = session.query(calendars_s).filter(
                calendars_s.name.ilike(tofind))
            q = q.filter(calendars_s.workload_date == self._workload)
            i = 0
            for j in q.limit(self._LIMIT):
                self._add_result(
                    container=c,
                    label=_("Calendar directory name") if j.entry_type
                    else _("Calendar name"),
                    obj_name=unicode(
                        path_cal.PathCal(session,
                                         id=j.id,
                                         workload=self._workload)),
                    obj_type=self._CAL,
                    obj_id=j.id,
                    attr=self._CAL_NAME)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more calendars"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=self._CAL_NAME)
            q = session.query(calendars_s).filter(
                calendars_s.description.ilike(tofind))
            q = q.filter(calendars_s.workload_date == self._workload)
            i = 0
            for j in q.limit(self._LIMIT):
                self._add_result(
                    container=c,
                    label=_("Calendar directory description") if j.entry_type
                    else _("Calendar description"),
                    obj_name=unicode(
                        path_cal.PathCal(session,
                                         id=j.id,
                                         workload=self._workload)),
                    obj_type=self._CAL,
                    obj_id=j.id,
                    attr=self._CAL_DESCR)
                i += 1
            if i >= self._LIMIT:
                self._add_result(
                    container=c,
                    label=_("..."),
                    obj_name=_("more calendar descriptions"),
                    obj_type=self._NONE,
                    obj_id=0,
                    attr=self._CAL_DESCR)

        # clusters
        q = session.query(clusters).filter(clusters.name.ilike(tofind))
        i = 0
        for j in q.limit(self._LIMIT):
            self._add_result(container=c,
                             label=_("Cluster name"),
                             obj_name=j.name,
                             obj_type=self._CLUSTER,
                             obj_id=j.id,
                             attr=self._CLUSTER_NAME)
            i += 1
        if i >= self._LIMIT:
            self._add_result(container=c,
                             label=_("..."),
                             obj_name=_("more cluster names"),
                             obj_type=self._NONE,
                             obj_id=0,
                             attr=self._CLUSTER_NAME)
        q = session.query(clusters).filter(clusters.description.ilike(tofind))
        i = 0
        for j in q.limit(self._LIMIT):
            self._add_result(container=c,
                             label=_("Cluster description"),
                             obj_name=j.name,
                             obj_type=self._CLUSTER,
                             obj_id=j.id,
                             attr=self._CLUSTER_DESCR)
            i += 1
        if i >= self._LIMIT:
            self._add_result(container=c,
                             label=_("..."),
                             obj_name=_("more cluster descriptions"),
                             obj_type=self._NONE,
                             obj_id=0,
                             attr=self._CLUSTER_NAME)

        # hosts
        q = session.query(hosts).filter(hosts.hostname.ilike(tofind))
        i = 0
        for j in q.limit(self._LIMIT):
            self._add_result(container=c,
                             label=_("Host name"),
                             obj_name=unicode(j),
                             obj_type=self._HOST,
                             obj_id=j.id,
                             attr=self._HOST_NAME)
            i += 1
        if i >= self._LIMIT:
            self._add_result(container=c,
                             label=_("..."),
                             obj_name=_("more host names"),
                             obj_type=self._NONE,
                             obj_id=0,
                             attr=self._HOST_NAME)
        q = session.query(hosts).filter(hosts.portnum.ilike(tofind))
        i = 0
        for j in q.limit(self._LIMIT):
            self._add_result(
                container=c,
                label=_("Host port"),
                obj_name="%s(%s)" % (j.hostname, j.portnum),
                obj_type=self._HOST,
                obj_id=j.id,
                attr=self._HOST_PORT)
            i += 1
        if i >= self._LIMIT:
            self._add_result(container=c,
                             label=_("..."),
                             obj_name=_("more host ports"),
                             obj_type=self._NONE,
                             obj_id=0,
                             attr=self._HOST_PORT)
        self._sql_session.close_session(session)
        q = session.query(hosts).filter(hosts.description.ilike(tofind))
        i = 0
        for j in q.limit(self._LIMIT):
            self._add_result(container=c,
                             label=_("Host description"),
                             obj_name=unicode(j),
                             obj_type=self._HOST,
                             obj_id=j.id,
                             attr=self._HOST_DESCR)
            i += 1
        if i >= self._LIMIT:
            self._add_result(container=c,
                             label=_("..."),
                             obj_name=_("more host descriptions"),
                             obj_type=self._NONE,
                             obj_id=0,
                             attr=self._HOST_DESCR)
        ResultPopup(self, c)


class EnterShortcutLaunchSearch(ShortcutListener):

    def __init__(self, obj):
        """Initialize the object.

        @param obj:     L{Find} object.
        """
        super(EnterShortcutLaunchSearch, self).__init__(None, KeyCode.ENTER,
                                                        None)
        self._c = obj

    def handleAction(self, sender, target):
        self._c.do_search(self._c._text.getValue())


class ResultPopup(IClickListener):

    """Popup to display the find results."""

    def __init__(self, find_obj, container):
        """Create the popup.

        @param find_obj:
                the assosiated L{Find} object.
        @param container:
                the data container
        @type container:
                L{muntjac.data.util.indexed_container.IndexedContainer}
        """
        super(ResultPopup, self).__init__()
        self._c = find_obj
        self._container = container

        self._w = Window(_("Result"))
        self._w.setWidth('700px')
        self._w.setHeight('433px')

        # VerticalLayout as content by default
        v = self._w.getContent()
        v.setSizeFull()
        v.setMargin(True)
        v.setSpacing(True)

        t = Table('', container)
        t.setSelectable(True)
        t.setMultiSelect(False)
        t.setImmediate(True)
        t.setColumnReorderingAllowed(False)
        t.setColumnCollapsingAllowed(False)
        t.setVisibleColumns(['label', 'object', 'value'])
        t.setColumnHeaders([_('Object'), _('Name'), _('Value')])
        t.setColumnWidth('label', 165)
        t.setColumnExpandRatio('object', 1.0)
        t.setColumnExpandRatio('value', 1.0)
        t.setSizeFull()
        t.addListener(DoubleClick(self), IItemClickListener)
        t.addShortcutListener(EnterShortcut(self))
        self._table = t
        v.addComponent(t)
        v.setExpandRatio(t, 1.0)

        v.addComponent(Label(_("Displayed: ") + str(container.size())))

        # Close button
        ok = Button(_('Close'))
        ok.addListener(self, IClickListener)
        v.addComponent(ok)
        v.setComponentAlignment(ok, Alignment.BOTTOM_RIGHT)

        self._c._parent_widget.getWindow().addWindow(self._w)
        self._w.center()

    def get_datasource(self):
        """Return the container data source used by the table.

        @return:    the container data source.
        @rtype:
            L{muntjac.data.util.indexed_container.IndexedContainer}
        """
        return self._table.getContainerDataSource()

    def get_selected(self):
        """Return the container ID of the selected item."""
        return self._table.getValue()

    def launch_detail(self, item_id):
        """Display a sub-window on the provided item ID (host, job, ...)

        @param item_id:
                    container ID of the item.
        """
        if item_id is None:
            return
        c = self.get_datasource()
        item = c.getItem(item_id)
        if item is None:
            return
        obj_type = item.getItemProperty('type').getValue()
        obj_id = item.getItemProperty('id').getValue()
        obj_attr = item.getItemProperty('attr').getValue()
        if obj_type == Find._JOB:
            if obj_id == ROOT_JOBSET_ID:
                ids = []
            else:
                try:
                    jobset_ids = path.id2path(self._c._sql_session, obj_id,
                                              self._c._workload)
                    ids = jobset_ids.id[:-1]
                except:
                    return
            JobWindow(self._c._parent_widget, ids,
                      self._c._parent_widget.getWindow(),
                      self._c._sql_session,
                      self._c._workload,
                      obj_id,
                      focus=obj_attr)
        elif obj_type == Find._ENV:
            EnvWindow(self._c._parent_widget.getWindow(),
                      self._c._sql_session,
                      self._c._workload,
                      obj_id)
        elif obj_type == Find._CLUSTER:
            HostWindow(self._c._parent_widget,
                       self._c._parent_widget.getWindow(),
                       self._c._sql_session,
                       self._c._workload,
                       cluster_id_to_select=obj_id)
        elif obj_type == Find._HOST:
            HostWindow(self._c._parent_widget,
                       self._c._parent_widget.getWindow(),
                       self._c._sql_session,
                       self._c._workload,
                       host_id_to_select=obj_id)
        elif obj_type == Find._CAL:
            CalendarWindow(self._c._parent_widget,
                           self._c._parent_widget.getWindow(),
                           self._c._sql_session,
                           self._c._workload,
                           obj_id)

    def buttonClick(self, event):
        # Close the window
        self._c._parent_widget.getWindow().removeWindow(self._w)


class EnterShortcut(ShortcutListener):

    def __init__(self, obj):
        """Initialize the object.

        @param obj:     L{Find} object.
        """
        super(EnterShortcut, self).__init__(None, KeyCode.ENTER, None)
        self._c = obj

    def handleAction(self, sender, target):
        self._c.launch_detail(self._c.get_selected())


class DoubleClick(IItemClickListener):

    """Double-click callback."""

    def __init__(self, result_popup_obj):
        """Initialize the callback.

        @param result_popup_obj:
                    the associated L{ResultPopup} object.
        @type result_popup_obj:    L{ResultPopup}
        """
        super(DoubleClick, self).__init__()
        self._c = result_popup_obj

    def itemClick(self, event):
        if event.isDoubleClick():
            self._c.launch_detail(event.getItemId())
