# Schedwi
# Copyright (C) 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Hosts window."""


from muntjac.api import (Window, HorizontalSplitPanel, Button, Alignment,
                         VerticalLayout, HorizontalLayout, ListSelect,
                         TextField, TextArea, Table, Label, TwinColSelect)
from muntjac.data.property import IValueChangeListener
from muntjac.ui.button import IClickListener
from muntjac.ui.window import Notification
from muntjac.terminal.theme_resource import ThemeResource
from muntjac.terminal.sizeable import ISizeable
from muntjac.data.util.indexed_container import IndexedContainer

from web.boxes import DeleteBox, BoxCloseListener
import cmd_clusters.whatuses
import host_utils
import cluster_utils
from tables.clusters import clusters
from tables.host_clusters import host_clusters


class ClustersTab(HorizontalSplitPanel):

    """Cluster tab."""

    def __init__(self, host_window_obj, refresh_obj, main_window,
                 sql_session, workload=None, cluster_id_to_select=None):
        """Build the Host window.

        @param host_window_obj:
                    the parent L{web.hostwindow.HostWindow} window object.
        @param refresh_obj:
                    the object to use to refresh the view when a cluster has
                    been changed.
        @param main_window:
                    the L{web.main.SchedwiWindow} object (the main window)
        @param sql_session:
                    SQLAlchemy session.
        @param workload:
                    workload to consider.
        @param cluster_id_to_select:
                    database ID of the cluster to select by default.
        """
        super(ClustersTab, self).__init__()
        self._host_window_obj = host_window_obj
        self._refresh_obj = refresh_obj
        self._main_window = main_window
        self._sql_session = sql_session
        self._workload = workload

        # Layout
        self.setSplitPosition(220, ISizeable.UNITS_PIXELS, False)
        self.setSizeFull()

        vert_left = VerticalLayout()
        vert_left.setSizeFull()
        vert_left.setMargin(True)
        self.addComponent(vert_left)

        cluster_list = ListSelect(_("Clusters:"))
        cluster_list.setSizeFull()
        cluster_list.setNullSelectionAllowed(False)
        cluster_list.addListener(ClusterSelected(self), IValueChangeListener)
        cluster_list.setImmediate(True)
        vert_left.addComponent(cluster_list)
        vert_left.setExpandRatio(cluster_list, 1.0)
        self._cluster_list = cluster_list

        if workload is None:
            bt_box_group = HorizontalLayout()
            bt_box_group.setWidth('100%')
            bt_box_group.setMargin(False)
            vert_left.addComponent(bt_box_group)

            edit_group = Button()
            edit_group.setIcon(ThemeResource('icons/edit14.png'))
            edit_group.setStyleName('small')
            edit_group.setDescription(_("Edit the selected cluster"))
            edit_group.setEnabled(False)
            edit_group.addListener(ClusterEdit(self), IClickListener)
            bt_box_group.addComponent(edit_group)
            self._edit_group = edit_group

            add_group = Button()
            add_group.setIcon(ThemeResource('icons/add14.png'))
            add_group.setStyleName('small')
            add_group.setDescription(_("Create a new cluster"))
            add_group.addListener(ClusterCreate(self), IClickListener)
            bt_box_group.addComponent(add_group)

            del_group = Button()
            del_group.setIcon(ThemeResource('icons/delete14.png'))
            del_group.setStyleName('small')
            del_group.setDescription(_("Delete the selected cluster"))
            del_group.setEnabled(False)
            del_group.addListener(ClusterDel(self), IClickListener)
            bt_box_group.addComponent(del_group)
            self._del_group = del_group

            where_group = Button()
            where_group.setIcon(ThemeResource('icons/search14.png'))
            where_group.setStyleName('small')
            where_group.setDescription(_("Cluster references"))
            where_group.setEnabled(False)
            where_group.addListener(ClusterWhere(self), IClickListener)
            bt_box_group.addComponent(where_group)
            self._where_group = where_group

        vert_right = VerticalLayout()
        vert_right.setSizeFull()
        vert_right.setMargin(True)
        vert_right.setSpacing(True)
        self.addComponent(vert_right)

        name = Label()
        name.setStyleName('envname')
        name.setContentMode(Label.CONTENT_TEXT)
        vert_right.addComponent(name)
        self._name = name

        description = Label()
        description.setContentMode(Label.CONTENT_TEXT)
        vert_right.addComponent(description)
        self._descr = description

        # TwinCol to show and select the hosts in the cluster
        # There is no listener at this stage.  The listener is added by the
        # _set_twin_container() method and it will be fired every time the
        # user add (>>) or remove (<<) a host in the list.  It will trigger
        # an update of the database (write_database_twin()).
        # The reason why the listener can not be set here is that it will be
        # trigger not only by the user but also whenever the list is refreshed
        # by the user selecting an other cluster.
        tw = TwinColSelect()
        tw.setEnabled(True if workload is None else False)
        tw.setSizeFull()
        tw.setNullSelectionAllowed(True)
        tw.setMultiSelect(True)
        tw.setImmediate(True)
        tw.setLeftColumnCaption('Available hosts')
        tw.setRightColumnCaption('Hosts in the cluster')
        vert_right.addComponent(tw)
        vert_right.setExpandRatio(tw, 1.0)
        self._twin = tw
        self._twin_listener = None

        # Fill the cluster list
        container = self._set_cluster_container()

        # Initialize the host list
        self._set_twin_container()

        if cluster_id_to_select is not None:
            self._cluster_list.setValue(cluster_id_to_select)
        else:
            item_id = container.firstItemId()
            if item_id:
                self._cluster_list.setValue(item_id)

    def refresh_main(self):
        """Refresh the parent view."""
        self._refresh_obj.refresh()

    def _set_twin_container(self, selected_cluster_id=None):
        """Build and set the container for the host list.

        @param selected_cluster_id:
                        the database ID of the cluster for which the host
                        list must be displayed.
        """
        if self._twin_listener:
            self._twin.removeListener(self._twin_listener)
        c = IndexedContainer()
        c.addContainerProperty('name', str, None)
        session = self._sql_session.open_session()
        for h in host_utils.name2host_list(session, '*'):
                item = c.addItem(h.id)
                item.getItemProperty('name').setValue(str(h))
        self._twin.setContainerDataSource(c)
        self._twin.setItemCaptionPropertyId('name')
        if selected_cluster_id is not None:
            self._twin.setValue(
                [h.id for h in
                 cluster_utils.get_hosts_in_cluster(session,
                                                    selected_cluster_id)])
        self._sql_session.close_session(session)
        if self._twin_listener is None:
            self._twin_listener = TwinSelected(self)
        self._twin.addListener(self._twin_listener, IValueChangeListener)

    def _set_cluster_container(self):
        """Build and set the container for the cluster list."""
        c = IndexedContainer()
        c.addContainerProperty('name', str, None)
        c.addContainerProperty('description', str, None)
        for cl in cluster_utils.name2cluster_list(self._sql_session, '*'):
            item = c.addItem(cl.id)
            item.getItemProperty('name').setValue(cl.name.encode('utf-8'))
            item.getItemProperty('description').setValue(
                cl.description.encode('utf-8'))
        self._cluster_list.setContainerDataSource(c)
        self._cluster_list.setItemCaptionPropertyId('name')
        return c

    def get_selected_cluster(self):
        """Return the currently selected host ID.

        @return:    the database ID of the selected host.
        """
        return self._cluster_list.getValue()

    def refresh_cluster_list(self, cluster_id_to_select=None):
        """Rebuild the cluster list.

        @param cluster_id_to_select:
                        database ID of the cluster to select after
                        the refresh.  If None (default), the previously
                        selected cluster will be re-selected.
        """
        # Save the selection so we can restore it
        if cluster_id_to_select is None:
            cluster_id_to_select = self.get_selected_cluster()
        self._set_cluster_container()
        self._cluster_list.setValue(cluster_id_to_select)

    def set_state_cluster_buttons(self):
        """Enable or disable (greyed out) action buttons whether a
        cluster is selected or not.
        """
        if self._workload is not None:
            return
        selected_item_id = self.get_selected_cluster()
        if selected_item_id is None:
            self._edit_group.setEnabled(False)
            self._del_group.setEnabled(False)
            self._where_group.setEnabled(False)
        else:
            self._del_group.setEnabled(True)
            self._edit_group.setEnabled(True)
            self._where_group.setEnabled(True)

    def propagate_select_cluster(self):
        """Display the selected cluster details (name, description and
        host members) in the right pane of the window.
        """
        selected_item_id = self.get_selected_cluster()
        if selected_item_id is None:
            self._name.setValue('')
            self._descr.setValue('')
            self._descr.removeStyleName('envdescription')
        else:
            c = self._cluster_list.getContainerDataSource()
            item = c.getItem(selected_item_id)
            self._name.setValue(item.getItemProperty('name').getValue())
            d = item.getItemProperty('description').getValue().strip()
            self._descr.setValue(d)
            if d:
                self._descr.addStyleName('envdescription')
            else:
                self._descr.removeStyleName('envdescription')
        self._set_twin_container(selected_item_id)

    def get_previous_cluster_id(self, cluster_id):
        """Return the database ID of the previous cluster in the list.

        @param cluster_id:
                    the database ID of the cluster.
        @return:    the ID of the previous cluster in the list.
        """
        c = self._cluster_list.getContainerDataSource()
        i = c.prevItemId(cluster_id)
        return i if i is not None else c.nextItemId(cluster_id)

    def write_database_twin(self):
        """Write in the database the hosts associated with the cluster.
        """
        cluster_id = self.get_selected_cluster()
        if cluster_id is not None:
            # Build the list of hosts from the selected items
            new_list = list()
            for item in self._twin.getValue():
                new_list.append(host_clusters(item))
            # Retrieve the cluster from the database and update the host list
            session = self._sql_session.open_session()
            try:
                c = cluster_utils.get_cluster_by_id(session, cluster_id)
            except:
                self._sql_session.close_session(session)
                self._main_window.showNotification(
                    _("Cannot get the selected cluster details"),
                    _("<br/>Maybe someone else just removed it."),
                    Notification.TYPE_ERROR_MESSAGE)
                self.refresh_cluster_list()
                return
            c.host_clusters = new_list
            self._sql_session.close_session(session)


class ClusterSelected(IValueChangeListener):

    """Callback for when a cluster is selected in the list."""

    def __init__(self, clusters_tab_obj):
        """Initialize the callback.

        @param clusters_tab_obj:
                    the associated L{ClustersTab} object.
        """
        super(ClusterSelected, self).__init__()
        self._c = clusters_tab_obj

    def valueChange(self, event):
        # Enable/disable action buttons
        self._c.set_state_cluster_buttons()
        # Display the selected host details
        self._c.propagate_select_cluster()


class ClusterEdit(IClickListener):

    """Callback for the Edit host button."""

    def __init__(self, clusters_tab_obj):
        """Initialize the callback.

        @param clusters_tab_obj:
                    the associated L{ClustersTab} object.
        """
        super(ClusterEdit, self).__init__()
        self._c = clusters_tab_obj

    def buttonClick(self, event):
        cluster_id = self._c.get_selected_cluster()
        if cluster_id is not None:
            NewCluster(self._c, cluster_id)


class ClusterCreate(IClickListener):

    """Callback for the Create a new cluster button."""

    def __init__(self, clusters_tab_obj):
        """Initialize the callback.

        @param clusters_tab_obj:
                    the associated L{ClustersTab} object.
        """
        super(ClusterCreate, self).__init__()
        self._c = clusters_tab_obj

    def buttonClick(self, event):
        NewCluster(self._c)


class DeleteConfirmed(BoxCloseListener):

    """Delete confirmation callback."""

    def __init__(self, clusters_tab_obj, cluster_id):
        """Initialize the object.

        @param clusters_tab_obj
                    the associated L{ClustersTab} object.
        @param cluster_id:
                    database ID of the cluster to remove.
        """
        super(DeleteConfirmed, self).__init__()
        self._c = clusters_tab_obj
        self._cluster_id = cluster_id

    def boxClose(self, event):
        """Called when the user confirmes the deletion."""
        session = self._c._sql_session.open_session()
        try:
            c = cluster_utils.get_cluster_by_id(session, self._cluster_id)
        except:
            pass
        else:
            session.delete(c)
        self._c._sql_session.close_session(session)
        self._c.refresh_cluster_list(
            self._c.get_previous_cluster_id(self._cluster_id))


class ClusterDel(IClickListener):

    """Callback for the Remove cluster button."""

    def __init__(self, clusters_tab_obj):
        """Initialize the callback.

        @param clusters_tab_obj:
                    the associated L{ClustersTab} object.
        """
        super(ClusterDel, self).__init__()
        self._c = clusters_tab_obj

    def buttonClick(self, event):
        cluster_id = self._c.get_selected_cluster()
        if cluster_id is not None:
            session = self._c._sql_session.open_session()
            try:
                c = cluster_utils.get_cluster_by_id(session, cluster_id)
            except:
                self._c._sql_session.close_session(session)
                self._c.refresh_cluster_list()
                return
            if cmd_clusters.whatuses.is_used(session, c):
                name = c.name.encode('utf-8')
                self._c._sql_session.close_session(session)
                self._c._main_window.showNotification(
                    _("This cluster is still in used"),
                    _("<br/>%s is in used somewhere else and cannot be \
                       deleted from the database.") % str(name),
                    Notification.TYPE_ERROR_MESSAGE)
            else:
                self._c._sql_session.close_session(session)
                d = DeleteBox(self._c._main_window,
                              _("Delete"),
                              _('Are you sure you want to delete \
                                 this cluster?'))
                d.addListener(DeleteConfirmed(self._c, cluster_id))


class ClusterWhere(IClickListener):

    """Callback for the Where used button."""

    def __init__(self, clusters_tab_obj):
        """Initialize the callback.

        @param clusters_tab_obj:
                    the associated L{ClustersTab} object.
        """
        super(ClusterWhere, self).__init__()
        self._c = clusters_tab_obj

    def buttonClick(self, event):
        cluster_id = self._c.get_selected_cluster()
        if cluster_id is not None:
            session = self._c._sql_session.open_session()
            try:
                c = cluster_utils.get_cluster_by_id(session, cluster_id)
            except:
                self._c._sql_session.close_session(session)
                self._c._main_window.showNotification(
                    _("Cannot get the selected cluster details"),
                    _("<br/>Maybe someone else just removed it."),
                    Notification.TYPE_ERROR_MESSAGE)
                self._c.refresh_cluster_list()
                return
            # The items (job, jobset, hosts) that are using this
            # cluster are stored in a list.  Later on, they will be displayed
            # in a popup (WhereWindow).
            self._gathered = list()
            cmd_clusters.whatuses.print_whatuses(session, c,
                                                 self._gather, self)
            self._c._sql_session.close_session(session)
            if not self._gathered:
                self._c._main_window.showNotification(
                    _("Not used"),
                    _("<br/>%s is not used.") % str(c.name.encode('utf-8')))
            else:
                WhereWindow(self._c, self._gathered)

    @staticmethod
    def _gather(self, title, val):
        """Callback method for the L{cmd_clusters.whatuses.print_whatuses}
        function.  This method is used to collect the items that are using
        a cluster.
        """
        self._gathered.append([title, val])


class NewCluster(IClickListener):

    """Edit/Create window for a cluster."""

    _bt_captions = [_('Cancel'), _('OK')]

    def __init__(self, clusters_tab_obj, cluster_id=None):
        """Create and display the Edit/Create window.

        @param clusters_tab_obj:
                    the associated L{ClustersTab} object.
        @param cluster_id:
                    when the window is used to edit an existing cluster, this
                    is the database cluster ID.
        """
        super(NewCluster, self).__init__()

        self._c = clusters_tab_obj
        self._cluster_id = cluster_id

        if cluster_id is None:
            title = _('New cluster')
            cluster_obj = None
        else:
            title = _('Edit cluster')
            try:
                cluster_obj = cluster_utils.get_cluster_by_id(
                    self._c._sql_session, cluster_id)
            except:
                self._c._main_window.showNotification(
                    _("Cannot get the selected cluster details"),
                    _("<br/>Maybe someone else just removed it."),
                    Notification.TYPE_ERROR_MESSAGE)
                self._c.refresh_cluster_list()
                return
        self._w = Window(title)
        self._w.setWidth("360px")

        # VerticalLayout as content by default
        v = self._w.getContent()
        v.setSizeFull()
        v.setMargin(True)
        v.setSpacing(True)

        t = TextField(_('Name:'))
        t.setWidth('100%')
        t.setDescription(_('Cluster name'))
        if cluster_obj:
            t.setValue(cluster_obj.name.encode('utf-8'))
        self._name = t
        v.addComponent(t)

        t = TextArea(_('Description:'))
        t.setSizeFull()
        t.setDescription(_('Free text as a comment or description'))
        if cluster_obj:
            t.setValue(cluster_obj.description.encode('utf-8'))
        self._descr = t
        v.addComponent(t)
        v.setExpandRatio(t, 1.0)

        # Button box
        h_bt = HorizontalLayout()
        h_bt.setMargin(False)
        h_bt.setSpacing(True)

        for caption in self._bt_captions:
            b = Button(_(caption))
            b.addListener(self, IClickListener)
            h_bt.addComponent(b)

        v.addComponent(h_bt)
        v.setComponentAlignment(h_bt, Alignment.BOTTOM_RIGHT)

        self._c._main_window.addWindow(self._w)
        self._w.center()

    def buttonClick(self, event):
        # First button is Cancel
        if event.getButton().getCaption() != _(self._bt_captions[0]):
            # Retrieve the values from the form
            name = self._name.getValue().strip()
            if not name:
                self._c._main_window.showNotification(
                    _("Name is empty"),
                    _("<br/>The cluster name cannot \
                       be empty or contain just spaces."),
                    Notification.TYPE_ERROR_MESSAGE)
                self._name.focus()
                return
            descr = self._descr.getValue().strip()

            # Try to retrieve the cluster from the database
            session = self._c._sql_session.open_session()
            try:
                c = cluster_utils.name2cluster(session, name)
            except:
                # Not found.
                c = None
            else:
                # Already in the database
                # If it was an Add Window, print an error message.
                if self._cluster_id is None or self._cluster_id != c.id:
                    self._c._sql_session.close_session(session)
                    self._c._main_window.showNotification(
                        _("This name is already used"),
                        _("<br/>The cluster %s is already \
                        defined in the database.") % name,
                        Notification.TYPE_ERROR_MESSAGE)
                    self._name.focus()
                    return

            # Add the new cluster in the database
            if self._cluster_id is None:
                c = clusters(name, descr)
                session.add(c)

            # Edit the cluster
            else:
                if c is None:
                    # Retrieve the old cluster
                    try:
                        c = cluster_utils.get_cluster_by_id(session,
                                                            self._cluster_id)
                    except:
                        # Humm, it should have been in the database...
                        # So recreating the cluster
                        c = clusters(name, descr)
                        session.add(c)
                c.name = name.decode('utf-8')
                c.description = descr.decode('utf-8')
            self._c._sql_session.close_session(session)
            self._c.refresh_cluster_list(c.id)

        # Close the window
        self._c._main_window.removeWindow(self._w)
        # Refresh the main view
        self._c.refresh_main()


class WhereWindow(IClickListener):

    """Window to display the cluster associations."""

    def __init__(self, clusters_tab_obj, obj_list):
        """Create and display the window.

        @param clusters_tab_obj:
                    the associated L{ClustersTab} object.
        @param obj_list:
                    list of objects.  Each object is an array of two
                    strings.  The first one is the type of object (job,
                    jobset, host) and the second one is its name.
        """
        super(WhereWindow, self).__init__()

        self._c = clusters_tab_obj

        self._w = Window(_("Cluster references"))
        self._w.setWidth("320px")
        self._w.setHeight("320px")

        # VerticalLayout as content by default
        v = self._w.getContent()
        v.setSizeFull()
        v.setMargin(True)
        v.setSpacing(True)

        t = Table()
        t.setSizeFull()
        t.setColumnReorderingAllowed(False)
        t.setColumnCollapsingAllowed(False)
        t.setSortDisabled(False)
        t.setSelectable(False)
        t.setMultiSelect(False)
        c = IndexedContainer()
        c.addContainerProperty('type', str, None)
        c.addContainerProperty('value', str, None)
        for k, val in obj_list:
            val = val.encode('utf-8')
            item = c.addItem(c.generateId())
            item.getItemProperty('type').setValue(k)
            item.getItemProperty('value').setValue(val)
        t.setContainerDataSource(c)
        t.setColumnExpandRatio('value', 1.0)
        v.addComponent(t)
        v.setExpandRatio(t, 1.0)

        # Close button
        ok = Button(_('Close'))
        ok.addListener(self, IClickListener)
        v.addComponent(ok)
        v.setComponentAlignment(ok, Alignment.BOTTOM_RIGHT)

        self._c._main_window.addWindow(self._w)
        self._w.center()

    def buttonClick(self, event):
        self._c._main_window.removeWindow(self._w)


class TwinSelected(IValueChangeListener):

    """Callback for when a host is selected (moved from the left list
    to the right list) in the twin list."""

    def __init__(self, clusters_tab_obj):
        """Initialize the callback.

        @param clusters_tab_obj:
                    the associated L{ClustersTab} object.
        """
        super(TwinSelected, self).__init__()
        self._c = clusters_tab_obj

    def valueChange(self, event):
        # Save the new host list in the database
        self._c.write_database_twin()
