# Schedwi
# Copyright (C) 2013-2015 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Calendar window."""

from muntjac.api import (Window, HorizontalSplitPanel, VerticalSplitPanel,
                         Button, Alignment, Tree, HorizontalLayout, TextField,
                         VerticalLayout, Table, NativeButton, Label)
from muntjac.ui.button import IClickListener
from muntjac.event.item_click_event import IItemClickListener
from muntjac.data.property import IValueChangeListener
from muntjac.ui.tree import IExpandListener, ICollapseListener
from muntjac.ui.table import IHeaderClickListener
from muntjac.terminal.sizeable import ISizeable
from muntjac.data.util.hierarchical_container import HierarchicalContainer
from muntjac.data.util.indexed_container import IndexedContainer
from muntjac.ui.embedded import Embedded
from muntjac.ui.abstract_select import ItemDescriptionGenerator
from muntjac.terminal.theme_resource import ThemeResource
from muntjac.ui.window import Notification
from muntjac.event.action import Action
from muntjac.event.shortcut_action import KeyCode

from web.cutcopypastemenu import GenericContextMenu
from web.boxes import DeleteBox, BoxCloseListener
from web.calendarview import CalendarView
from web.calendareditwindow import CalendarEditWindow
from path_cal import id2list, PathCal
from simple_queries_cal import (sql_get_cal_children, sql_count_cal_children,
                                sql_count_cal_children2, sql_get_cal)
from cmd_cal.rm import rm_recursive_force
from cmd_cal.cp import cp_recursive
import cmd_cal.whatuses
from tables.calendars import (calendars, ROOT_CAL_DIR_ID)
from tables.calendars_s import calendars_s


# Memorize the last selected item to show the same containing dir next time
_LAST_SELECTED = None

# Cut/Copy/Paste
_SELECTED_ID = None
_IS_CUT = False


class CutCopyPasteMenuListener(object):

    """Interface for providing details required for popup menus."""

    def get_parent_id(self):
        """Return the database ID of the parent jobset."""
        raise NotImplementedError

    def get_application(self):
        """Return the L{muntjac.api.Application} object (the main window)."""
        raise NotImplementedError

    def repaint(self):
        """Refresh the view."""
        raise NotImplementedError

    def repaint_tools(self):
        """Repaint the cut/copy/paste button."""
        raise NotImplementedError

    def is_directory(self, item_id):
        """Tell if the provided object is a directory or a calendar.

        @return:    True if the item is a calendar, else False.
        """
        raise NotImplementedError


class CalendarWindow(Window, IClickListener, IValueChangeListener,
                     IExpandListener, ICollapseListener,
                     CutCopyPasteMenuListener):

    def __init__(self, refresh_obj, main_window,
                 sql_session, workload=None, cal_id_to_select=None):
        """Build the Calendar window.

        @param refresh_obj:
                    the object to use to refresh the view when the host has
                    been changed.
        @param main_window:
                    the L{web.main.SchedwiWindow} object (the main window)
        @param sql_session:
                    SQLAlchemy session.
        @param workload:
                    workload to consider.
        @param cal_id_to_select:
                    database ID of the calendar to select by default.
        """
        global _LAST_SELECTED
        super(CalendarWindow, self).__init__()
        self._refresh_obj = refresh_obj
        self._main_window = main_window
        self._sql_session = sql_session
        self._workload = workload
        self._selected_dir = None
        self._selected_cal = None

        # Layout
        self.setCaption(_('Calendars'))
        self.setHeight('580px')
        self.setWidth('760px')

        v = self.getContent()
        v.setSizeFull()
        v.setMargin(False)
        v.setSpacing(False)

        horiz = HorizontalSplitPanel()
        horiz.setSplitPosition(190, ISizeable.UNITS_PIXELS)
        horiz.setSizeFull()
        v.addComponent(horiz)
        v.setExpandRatio(horiz, 1.0)

        h = HorizontalLayout()
        h.setWidth('100%')
        h.setMargin(True)
        h.setSpacing(False)
        h.setStyleName('popupt')
        v.addComponent(h)

        ok = Button(_('Close'))
        ok.addListener(self, IClickListener)
        h.addComponent(ok)
        h.setComponentAlignment(ok, Alignment.BOTTOM_RIGHT)

        # Left site: calendar tree
        self._tree = Tree()
        self._tree.setImmediate(True)
        self._tree.setNullSelectionAllowed(False)
        self._tree.setItemDescriptionGenerator(Tooltips(sql_session, workload))
        if workload is None:
            self._popup_menu = CutCopyPasteMenuDirectoryTree(self)
            self._tree.addActionHandler(self._popup_menu)
        self._new_container_dir()
        horiz.addComponent(self._tree)

        # Right site: toolbar and detailed view
        r = VerticalLayout()
        r.setSizeFull()
        horiz.addComponent(r)

        # Breadcrumb and toolbar horizontal layout
        h = HorizontalLayout()
        h.setWidth("100%")
        h.setSpacing(True)
        h.setMargin(False)
        h.addStyleName("toolbar")
        r.addComponent(h)

        # Toolbar
        t = HorizontalLayout()
        t.setSpacing(False)
        t.setMargin(False)
        h.addComponent(t)

        # Breadcrumb
        b = BreadCrumb(self)
        b.setMargin(False)
        h.addComponent(b)
        h.setExpandRatio(b, 1.0)
        self._bread_crumb = b

        if workload is None:
            b = NativeButton()
            b.setIcon(ThemeResource('icons/edit.png'))
            b.setDescription(_("Edit the selected calendar or directory"))
            b.setStyleName('toolbutton')
            b.setEnabled(False)
            b.addListener(self, IClickListener)
            t.addComponent(b)
            self._btedit = b

            b = Label()  # Separator
            b.setIcon(ThemeResource('icons/sep.png'))
            t.addComponent(b)
            t.setComponentAlignment(b, Alignment.MIDDLE_CENTER)

            b = NativeButton()
            b.setIcon(ThemeResource('icons/dir-new.png'))
            b.setDescription(_("Create a new directory"))
            b.setStyleName('toolbutton')
            b.setEnabled(True)
            b.addListener(self, IClickListener)
            t.addComponent(b)
            self._btadddir = b

            b = NativeButton()
            b.setIcon(ThemeResource('icons/calendar-new.png'))
            b.setDescription(_("Create a new calendar"))
            b.setStyleName('toolbutton')
            b.setEnabled(True)
            b.addListener(self, IClickListener)
            t.addComponent(b)
            self._btaddcal = b

            b = NativeButton()
            b.setIcon(ThemeResource('icons/delete.png'))
            b.setDescription(_("Delete the selected calendar or directory"))
            b.setStyleName('toolbutton')
            b.setEnabled(False)
            b.addListener(self, IClickListener)
            t.addComponent(b)
            self._btdel = b

            b = NativeButton()
            b.setIcon(ThemeResource('icons/search14.png'))
            b.setDescription(_("List associated jobs and jobsets"))
            b.setStyleName('toolbutton')
            b.setEnabled(False)
            b.addListener(self, IClickListener)
            t.addComponent(b)
            self._btwhat = b

            b = Label()  # Separator
            b.setIcon(ThemeResource('icons/sep.png'))
            t.addComponent(b)
            t.setComponentAlignment(b, Alignment.MIDDLE_CENTER)

            b = NativeButton()
            b.setIcon(ThemeResource('icons/cut.png'))
            b.setDescription(_("Cut the selected calendar or directory"))
            b.setStyleName('toolbutton')
            b.setEnabled(False)
            b.addListener(self, IClickListener)
            t.addComponent(b)
            self._btcut = b

            b = NativeButton()
            b.setIcon(ThemeResource('icons/copy.png'))
            b.setDescription(
                _("Mark the selected calendar or directory for copy"))
            b.setStyleName('toolbutton')
            b.setEnabled(False)
            b.addListener(self, IClickListener)
            t.addComponent(b)
            self._btcopy = b

            b = NativeButton()
            b.setIcon(ThemeResource('icons/paste.png'))
            b.setDescription(_("Paste the previously copied or cut item"))
            b.setStyleName('toolbutton')
            b.setEnabled(False)
            b.addListener(self, IClickListener)
            t.addComponent(b)
            self._btpaste = b

        vert = VerticalSplitPanel()
        vert.setSplitPosition(240, ISizeable.UNITS_PIXELS)
        vert.setSizeFull()
        r.addComponent(vert)
        r.setExpandRatio(vert, 1.0)

        # Table
        self._table = CalendarTable(main_window, self, sql_session,
                                    workload)
        vert.addComponent(self._table)

        # Calendar preview
        self._calendar_view = CalendarView()
        self._table.set_preview(self._calendar_view)
        vert.addComponent(self._calendar_view)

        # Callbacks
        self._tree.addListener(self, IValueChangeListener)
        self._tree.addListener(self, IExpandListener)
        self._tree.addListener(self, ICollapseListener)

        # Show the window
        main_window.addWindow(self)
        self.center()

        # Select a calendar
        if cal_id_to_select is not None:
            p = id2list(sql_session, cal_id_to_select, workload)
            for i in p[:-1]:
                self._tree.expandItem(i)
            if len(p) > 2:
                self._tree.select(p[-2])
                self._table.select(cal_id_to_select)
                self._table.setCurrentPageFirstItemId(cal_id_to_select)
            else:
                self._tree.select(ROOT_CAL_DIR_ID)
        elif _LAST_SELECTED is not None:
            try:
                p = id2list(sql_session, _LAST_SELECTED, workload)
            except:
                self._tree.expandItem(ROOT_CAL_DIR_ID)
                self._tree.select(ROOT_CAL_DIR_ID)
            else:
                if p:
                    for i in p:
                        self._tree.expandItem(i)
                self._tree.select(_LAST_SELECTED)
        else:
            self._tree.expandItem(ROOT_CAL_DIR_ID)
            self._tree.select(ROOT_CAL_DIR_ID)

    def buttonClick(self, event):
        """Callback for the Close button and all the toolbar buttons."""
        if self._workload is not None:
            self._main_window.removeWindow(self)
            return
        b = event.getButton()
        if b == self._btedit:
            if self._selected_cal is not None:
                self._popup_menu.do_edit(self._selected_cal)
        elif b == self._btadddir:
            self._popup_menu.do_new_directory(self._selected_dir)
        elif b == self._btaddcal:
            self._popup_menu.do_new_cal(self._selected_dir)
        elif b == self._btdel:
            if self._selected_cal is not None:
                self._popup_menu.do_delete(self._selected_cal)
        elif b == self._btwhat:
            if self._selected_cal is not None:
                self._popup_menu.do_whatuses(self._selected_cal)
        elif b == self._btcut:
            if self._selected_cal is not None:
                self._popup_menu.do_cut(self._selected_cal)
        elif b == self._btcopy:
            if self._selected_cal is not None:
                self._popup_menu.do_copy(self._selected_cal)
        elif b == self._btpaste:
            if self._selected_dir is not None:
                self._popup_menu.do_paste(self._selected_dir)
        else:
            self._main_window.removeWindow(self)

    def do_select(self, dir_id):
        """Mark the provided directory as selected.

        @param dir_id:   the database ID of the directory to select.
        """
        if dir_id is not None:
            c = self.get_datasource_tree()
            if dir_id in c.getItemIds():
                self._tree.select(dir_id)
            else:
                p = PathCal(self._sql_session, id=dir_id,
                            workload=self._workload)
                for i in p.id[:-1]:
                    self._tree.expandItem(i)
                self._tree.select(dir_id)

    def refresh_main(self):
        """Refresh the parent view."""
        self._refresh_obj.refresh()

    def get_datasource_tree(self):
        """Return the container data source used by the calendar tree.

        @return:    the container data source.
        @rtype:
            L{muntjac.data.util.hierarchical_container.HierarchicalContainer}
        """
        return self._tree.getContainerDataSource()

    def nodeExpand(self, event):
        """Callback when a directory has been expanded."""
        cal_id = event.getItemId()
        c = self.get_datasource_tree()
        session = self._sql_session.open_session()
        for child in sql_get_cal_children(session, cal_id,
                                          True, self._workload):
            child_id = child.id
            item = c.addItem(child_id)
            item.getItemProperty('name').setValue(child.name.encode('utf-8'))
            c.setParent(child_id, cal_id)
            # Only show the expand arrow if the dir has children
            if sql_count_cal_children(session, child_id, True, self._workload):
                c.setChildrenAllowed(child_id, True)
            else:
                c.setChildrenAllowed(child_id, False)
        self._sql_session.close_session(session)

    def nodeCollapse(self, event):
        """Callback when a directory has been collapsed."""
        cal_id = event.getItemId()
        c = self.get_datasource_tree()
        # Remove all the existing children first
        if c.hasChildren(cal_id):
            for childId in c.getChildren(cal_id):
                self._tree.collapseItem(childId)
                c.removeItemRecursively(childId)

    def valueChange(self, event):
        """Callback when a directory has been selected or de-selected.
        """
        global _LAST_SELECTED
        self._selected_dir = event.getProperty().getValue()
        if self._selected_dir:
            _LAST_SELECTED = self._selected_dir
        self.propagate_dir_select(self._selected_dir)

    def repaint(self):
        """Refresh the whole window."""
        global _LAST_SELECTED, _SELECTED_ID
        # Is the last selected item still in the database?
        if _LAST_SELECTED is not None:
            try:
                sql_get_cal(self._sql_session, _LAST_SELECTED, self._workload)
            except:
                _LAST_SELECTED = None
        # Save the ID of the selected item and the expanded dirs in the tree
        selected = self._selected_dir
        selected_cal = self._selected_cal
        c = self.get_datasource_tree()
        expandedIds = list()
        for i in c.getItemIds():
            if self._tree.isExpanded(i):
                expandedIds.append(i)

        # Create a new container
        self._tree.collapseItemsRecursively(ROOT_CAL_DIR_ID)
        c = self._new_container_dir()

        # Re-expand the previously expanded items
        while True:
            expand_done = False
            for i in expandedIds:
                if i in c.getItemIds() and not self._tree.isExpanded(i):
                    self._tree.expandItem(i)
                    if self._tree.hasChildren(i):
                        expand_done = True
            if expand_done is False:
                break

        # Re-select the previously selected item
        self._selected_dir = selected
        if selected in c.getItemIds():
            self._tree.select(selected)
            self._table.select(selected_cal)
        else:
            # If the previously selected directory does not exist anymore
            # in the database, select the root directory
            try:
                sql_get_cal(self._sql_session, selected, self._workload)
            except:
                self._tree.select(ROOT_CAL_DIR_ID)

        # Is the previously cut/copied item still in the database?
        if self._workload is None and _SELECTED_ID is not None:
            try:
                sql_get_cal(self._sql_session, _SELECTED_ID, self._workload)
            except:
                _SELECTED_ID = None
                self._btpaste.setEnabled(False)

    def _new_container_dir(self):
        """Create a new container for the directory tree."""
        jContainer = HierarchicalContainer()
        jContainer.addContainerProperty('name', str, None)
        # Root calendar
        item = jContainer.addItem(ROOT_CAL_DIR_ID)
        item.getItemProperty('name').setValue('/')
        self._tree.setContainerDataSource(jContainer)
        self._tree.setItemCaptionPropertyId('name')
        return jContainer

    def propagate_dir_select(self, dir_id):
        """Display the content of the selected directory on the table on the
        right.

        @param dir_id:
                    database ID of the selected directory.
        """
        self._bread_crumb.update(dir_id)
        self._table.update(dir_id)

    def set_state_toolbar(self, cal_id):
        """Set the toobar button state (active or greyed out) on whether a
        calendar/directory is selected in the table on the right.

        @param cal_id:
                    database ID of the selected calendar or None if nothing
                    is selected.
        """
        global _SELECTED_ID
        self._selected_cal = cal_id
        if self._workload is not None:
            return
        if cal_id is None:
            self._btedit.setEnabled(False)
            self._btdel.setEnabled(False)
            self._btwhat.setEnabled(False)
            self._btcut.setEnabled(False)
            self._btcopy.setEnabled(False)
        else:

            self._btedit.setEnabled(True)
            self._btdel.setEnabled(True)
            self._btwhat.setEnabled(True)
            self._btcut.setEnabled(True)
            self._btcopy.setEnabled(True)
        if _SELECTED_ID is None:
            self._btpaste.setEnabled(False)
        else:
            self._btpaste.setEnabled(True)

    def _build_directory_hierarchy(self, directory_id):
        """Build and return the directory ID list for the provided dir ID.

        @param directory_id:
                    the ID of the directory for which the list must be built.
        @return:    the list of the directory IDs from the root directory up to
                    the provided directory.
        """
        c = self.get_datasource_tree()
        ids = [directory_id]
        while True:
            directory_id = c.getParent(directory_id)
            if directory_id is not None:
                ids.insert(0, directory_id)
            else:
                break
        return ids

    def is_directory(self, item_id):
        """From L{CutCopyPasteMenuListener}"""
        try:
            cal = sql_get_cal(self._sql_session, item_id, self._workload)
        except:
            return False
        return True if cal.entry_type else False

    def get_parent_id(self):
        """From L{CutCopyPasteMenuListener}"""
        if self._selected_dir is None:
            return None
        ids = self._build_directory_hierarchy(self._selected_dir)
        return ids[-1]

    def get_application(self):
        """From L{CutCopyPasteMenuListener}"""
        return self._main_window

    def repaint_tools(self):
        """From L{CutCopyPasteMenuListener}"""
        global _SELECTED_ID
        if _SELECTED_ID is None:
            self._btpaste.setEnabled(False)
        else:
            self._btpaste.setEnabled(True)


class DeleteConfirmed(BoxCloseListener):

    """Confirmation popup callback."""

    def __init__(self, menu_obj, obj):
        """Initialize the object.

        @param menu_obj:
                    the associated L{CutCopyPasteMenuListener} object.
        @param obj:
                    L{tables.calendars.calendars} object to remove.
        """
        self._c = menu_obj
        self._dir_or_cal = obj

    def boxClose(self, event):
        """Deletion confirmed."""
        session = self._c._sql_session.open_session()
        p = PathCal(session, id=self._dir_or_cal.id,
                    workload=self._c._workload)
        ret = rm_recursive_force(session, p, self._c._workload)
        self._c._sql_session.close_session(session)
        self._c.repaint()
        if ret:
            self._c.get_application().showNotification(
                _("A calendar is in used and cannot be deleted"),
                _("<br/>%s is still used by a job or jobset.") % ret,
                Notification.TYPE_ERROR_MESSAGE)


class Tooltips(ItemDescriptionGenerator):

    """Calendar tree item tooltips."""

    def __init__(self, sql_session, workload=None):
        """Initialize the callback.

        @param sql_session:
                    SQLAlchemy session.
        @param workload:
                    workload to consider.
        """
        self._sql_session = sql_session
        self._workload = workload

    def generateDescription(self, source, itemId, propertyId):
        try:
            nb_cal, nb_dir = sql_count_cal_children2(self._sql_session,
                                                     itemId, self._workload)
        except:
            nb_cal = nb_dir = 0
        if nb_cal == 1:
            s = _("1 calendar")
        else:
            s = _("%d calendars") % nb_cal
        if nb_dir == 1:
            s += '<br/>' + _("1 sub-directory")
        else:
            s += '<br/>' + _("%d sub-directories") % nb_dir
        return s


class CalendarTable(Table, IValueChangeListener, IHeaderClickListener,
                    CutCopyPasteMenuListener):

    """Calendar list view."""

    _ICON_DIR = ThemeResource('icons/dir.png')
    _ICON_CAL = ThemeResource('icons/calendar.png')

    def __init__(self, main_window, calendar_window_obj,
                 sql_session, workload=None):
        """Create and initialize the table.

        @param main_window:
                    top application window.
        @type main_window:
                    L{web.main.SchedwiWindow}
        @param calendar_window_obj:
                    the parent L{CalendarWindow} object.
        @param sql_session:
                    SQLAlchemy session.
        @param workload:
                    workload to consider.
        """
        super(CalendarTable, self).__init__()
        self._main_window = main_window
        self._sql_session = sql_session
        self._workload = workload
        self._c = calendar_window_obj
        self._preview = None

        self._sortCol = None
        self._sortOrd = True  # Ascending
        self._dir_id = None
        self._selected_cal = None
        self.setSizeFull()
        self.setSelectable(True)
        self.setMultiSelect(False)
        self.setImmediate(True)
        self.setColumnReorderingAllowed(False)
        self.setColumnCollapsingAllowed(False)
        self.addStyleName("callisttable")
        self._new_container_table()
        self.addListener(self, IValueChangeListener)
        self.addListener(self, IHeaderClickListener)
        self.addListener(DoubleClick(self), IItemClickListener)
        if workload is None:
            self._popup_menu = CutCopyPasteMenuCalendarTable(self)
            self.addActionHandler(self._popup_menu)

    def _new_container_table(self):
        """Create and set a new calendar table container."""
        jContainer = IndexedContainer()
        jContainer.addContainerProperty('type', Embedded, None)
        jContainer.addContainerProperty('name', str, None)
        jContainer.addContainerProperty('formula', str, None)
        # type_cal: 1 --> dir  and   0 --> calendar
        jContainer.addContainerProperty('type_cal', int, None)
        self.setContainerDataSource(jContainer)
        self.setVisibleColumns(['type', 'name', 'formula'])
        self.setColumnHeaders([_('Type'), _('Name'), _('Formula')])
        self.setColumnWidth('type', 30)
        self.setColumnAlignment('type', Table.ALIGN_CENTER)

    def set_preview(self, preview):
        """Set the bottom calendar preview widget.

        @param preview:
                    the preview widget.
        @type preview:
                    L{web.calendarview.CalendarView}
        """
        self._preview = preview

    def get_datasource_table(self):
        """Return the container data source used by the calendar table.

        @return:    the container data source.
        @rtype:
            L{muntjac.data.util.indexed_container.IndexedContainer}
        """
        return self.getContainerDataSource()

    def update(self, dir_id):
        """Display the content of the provided directory.

        @param dir_id:
                    database ID of the directory for which the content must
                    be displayed.
        """
        self._new_container_table()
        if dir_id is None:
            return
        session = self._sql_session.open_session()
        try:
            lst = sql_get_cal_children(session, dir_id, False,
                                       self._workload, True)
        except:
            self._sql_session.close_session(session)
            return
        self._sql_session.close_session(session)
        c = self.get_datasource_table()
        for cal in lst:
            item = c.addItem(cal.id)
            item.getItemProperty('name').setValue(cal.name.encode('utf-8'))
            item.getItemProperty('type_cal').setValue(cal.entry_type)
            # entry_type = 0 --> calendar   1 --> directory
            if cal.entry_type == 1:
                e = Embedded(source=self._ICON_DIR)
                item.getItemProperty('type').setValue(e)
            else:
                e = Embedded(source=self._ICON_CAL)
                item.getItemProperty('type').setValue(e)
                item.getItemProperty('formula').setValue(
                    cal.formula.encode('utf-8'))
        self._dir_id = dir_id
        # Sort
        if self._sortCol == 'type':
            self.sort(['type_cal'], [not self._sortOrd])
        elif self._sortCol:
            self.sort([self._sortCol], [self._sortOrd])

        self.setVisibleColumns(['type', 'name', 'formula'])

        # Re-select the previously selected item
        if self._selected_cal in c.getItemIds():
            self.select(self._selected_cal)
        else:
            self._selected_cal = None
            self._c.set_state_toolbar(None)

    def valueChange(self, event):
        """Callback when a calendar has been selected or de-selected."""
        self._selected_cal = event.getProperty().getValue()
        self._c.set_state_toolbar(self._selected_cal)
        if self._preview is not None:
            if self._selected_cal is None:
                self._preview.set_formula(None)
            else:
                c = self.get_datasource_table()
                item = c.getItem(self._selected_cal)
                if item.getItemProperty('type_cal').getValue():
                    # Directory
                    self._preview.set_formula(None)
                else:
                    formula = item.getItemProperty('formula').getValue()
                    self._preview.set_formula(formula)

    def headerClick(self, event):
        """Callback when the hearder of a column is clicked."""
        col = event.getPropertyId()
        if self._sortCol == col:
            self._sortOrd = not self._sortOrd
        else:
            self._sortCol = col
        if self._sortCol == 'type':
            self.sort(['type_cal'], [not self._sortOrd])

    def is_directory(self, item_id):
        """From L{CutCopyPasteMenuListener}"""
        c = self.get_datasource_table()
        if c.getItem(item_id).getItemProperty('type_cal').getValue():
            return True
        return False

    def get_parent_id(self):
        """From L{CutCopyPasteMenuListener}"""
        return self._dir_id

    def get_application(self):
        """From L{CutCopyPasteMenuListener}"""
        return self._main_window

    def repaint(self):
        """From L{CutCopyPasteMenuListener}"""
        self._c.repaint()

    def repaint_tools(self):
        """From L{CutCopyPasteMenuListener}"""
        self._c.repaint_tools()


class BreadCrumb(HorizontalLayout):

    """Bread crumb (path of the selected directory) UI."""

    def __init__(self, calendar_window_obj):
        """Create the object.

        @param calendar_window__obj:
                    the associated L{CalendarWindow} object.
        @type calendar_window_obj:
                    L{CalendarWindow}
        """
        super(BreadCrumb, self).__init__()
        self._c = calendar_window_obj
        self._hierarchy_buttons = dict()
        self.addStyleName("breadcrumb")

    def update(self, dir_id):
        """Rebuild the bread crumb.

        @param dir_id:
                    database ID of the selected directory.
        """
        if dir_id is None:
            return
        if dir_id not in self._hierarchy_buttons:
            self.removeAllComponents()
            self._hierarchy_buttons = dict()
            for i in id2list(self._c._sql_session, dir_id, self._c._workload):
                if i == ROOT_CAL_DIR_ID:
                    name = '/'
                else:
                    c = sql_get_cal(self._c._sql_session, i, self._c._workload)
                    name = c.name.encode('utf-8')
                b = NativeButton(name)
                b.addStyleName('breadcrumbbt')
                if i == dir_id:
                    b.addStyleName("displayed")
                else:
                    b.addStyleName("notdisplayed")
                b.addListener(BreadCrumbDirHandler(self._c, i), IClickListener)
                self.addComponent(b)
                self._hierarchy_buttons[i] = b
        else:
            to_remove = list()
            session = self._c._sql_session.open_session()
            for i, b in self._hierarchy_buttons.iteritems():
                if i == dir_id:
                    b.removeStyleName("notdisplayed")
                    b.addStyleName("displayed")
                else:
                    try:
                        sql_get_cal(session, i, self._c._workload)
                    except:
                        to_remove.append(i)
                    else:
                        b.removeStyleName("displayed")
                        b.addStyleName("notdisplayed")
            self._c._sql_session.close_session(session)
            for dir_or_cal_id in to_remove:
                self.removeComponent(self._hierarchy_buttons[dir_or_cal_id])
                del self._hierarchy_buttons[dir_or_cal_id]


class BreadCrumbDirHandler(IClickListener):

    """Callback for when a directory in the bread crumb is clicked."""

    def __init__(self, calendar_window_obj, dir_id):
        """Initialize the callback.

        @param calendar_window__obj:
                    the associated L{CalendarWindow} object.
        @type calendar_window_obj:
                    L{CalendarWindow}
        @param dir_id:
                    the database ID of the directory.
        """
        super(BreadCrumbDirHandler, self).__init__()
        self._c = calendar_window_obj
        self._dir_id = dir_id

    def buttonClick(self, event):
        self._c.do_select(self._dir_id)


class DoubleClick(IItemClickListener):

    """Double-click callback."""

    def __init__(self, calendar_table_obj):
        """Initialize the callback.

        @param calendar_table_obj:
                    the associated L{CalendarTable} object.
        @type calendar_table_obj:    L{CalendarTable}
        """
        super(DoubleClick, self).__init__()
        self._c = calendar_table_obj

    def itemClick(self, event):
        if event.isDoubleClick():
            item_id = event.getItemId()
            # Edit for calendars and cd for directories
            if self._c.is_directory(item_id):
                self._c._c.do_select(item_id)
            elif self._c._workload is None:
                self._c._popup_menu.do_edit(item_id)


class CutCopyPasteMenu(GenericContextMenu):

    # Actions for the context menu
    _ACTION_PASTE = Action(_('Paste Into Directory'),
                           ThemeResource('icons/paste.png'))
    _ACTION_NEWCAL = Action(_('Create New Calendar...'),
                            ThemeResource('icons/calendar-new.png'))
    _ACTION_NEWDIR = Action(_('Create New Directory...'),
                            ThemeResource('icons/dir-new.png'))
    _ACTION_WHAT = Action(_('List associated Jobs/Jobsets...'),
                          ThemeResource('icons/search14.png'))

    def __init__(self, obj, sql_session, workload=None):
        """Initialize the object.

        @param obj:
                    L{CutCopyPasteMenuListener} object to retrieve the
                    details required by the menu.
        @param sql_session:
                    SQLAlchemy session.
        @param workload:
                    workload to consider.
        """
        super(CutCopyPasteMenu, self).__init__()
        self._sql_session = sql_session
        self._workload = workload
        self._c = obj

    def handleAction(self, a, sender, target):
        global _SELECTED_ID, _IS_CUT

        if not target:
            # Table background
            target = self._c.get_parent_id()

        # Edit
        if a.getCaption() == self._ACTION_EDIT.getCaption():
            self.do_edit(target)

        # Cut
        elif a.getCaption() == self._ACTION_CUT.getCaption():
            self.do_cut(target)

        # Copy
        elif a.getCaption() == self._ACTION_COPY.getCaption():
            self.do_copy(target)

        # Paste
        elif a.getCaption() == self._ACTION_PASTE.getCaption():
            self.do_paste(target)

        # Delete
        elif a.getCaption() == self._ACTION_DELETE.getCaption():
            self.do_delete(target)

        # What uses
        elif a.getCaption() == self._ACTION_WHAT.getCaption():
            self.do_whatuses(target)

        # New Calendar
        elif a.getCaption() == self._ACTION_NEWCAL.getCaption():
            self.do_new_cal(target)

        # New Directory
        elif a.getCaption() == self._ACTION_NEWDIR.getCaption():
            self.do_new_directory(target)

    def _get_name(self, target):
        """Compute a new name for the copied/pasted object.

        @param target:
                    database ID of the destination directory.
        @return:    the new name or None in case of error.
        """
        global _SELECTED_ID, _IS_CUT

        if not _SELECTED_ID:
            return None
        if _SELECTED_ID in id2list(self._sql_session, target, self._workload):
            self._c.getWindow().showNotification(
                _("Cannot move a directory into itself") if _IS_CUT else
                _("Cannot copy a directory into itself"),
                _("<br/>The destination directory is inside the source one."),
                Notification.TYPE_ERROR_MESSAGE)
            return None

        # Try to find a unique name for the pasted calendar
        # First, get the selected calendar/directory name
        session = self._sql_session.open_session()
        try:
            cal = sql_get_cal(session, _SELECTED_ID, self._workload)
        except:
            self._sql_session.close_session(session)
            self._c.getWindow().showNotification(
                _("Cannot get the source calendar details"),
                _("<br/>Maybe someone else just removed it."),
                Notification.TYPE_ERROR_MESSAGE)
            return None
        name = cal.name

        # Then, get the list of the calendar/directory names in the
        # destination directory
        dest_names = list()
        for cal in sql_get_cal_children(session, target, False,
                                        self._workload):
            dest_names.append(cal.name)
        self._sql_session.close_session(session)

        i = 1
        n = name
        while n in dest_names:
            n = name + '.%d' % i
            i += 1
        return n

    def do_edit(self, target):
        """Launch the edit popup.

        @param target:
                database ID of the calendar or directory to edit.
        """
        if self._c.is_directory(target):
            DirPopup(self._c, self._c.get_parent_id(), target)
        else:
            CalendarEditWindow(self._c, self._c.get_parent_id(), target)

    def do_cut(self, target):
        """Cut a calendar or directory.

        @param target:
                database ID of the calendar or directory to cut.
        """
        global _SELECTED_ID, _IS_CUT
        _SELECTED_ID = target
        _IS_CUT = True
        self._c.repaint_tools()

    def do_copy(self, target):
        """Copy a calendar or directory.

        @param target:
                database ID of the calendar or directory to copy.
        """
        global _SELECTED_ID, _IS_CUT
        _SELECTED_ID = target
        _IS_CUT = False
        self._c.repaint_tools()

    def do_paste(self, target):
        """Paste a calendar or directory in the provided directory.

        @param target:
                database ID of the parent directory in which to paste
                the previously copied or cut calendar or directory.
        """
        global _SELECTED_ID, _IS_CUT
        name = self._get_name(target)
        if not name:
            return
        session = self._sql_session.open_session()
        # Get the source calendar/directory
        try:
            src_cal = sql_get_cal(session, _SELECTED_ID, self._workload)
        except:
            self._sql_session.close_session(session)
            self._c.get_application().showNotification(
                _("Cannot get the source calendar details"),
                _("<br/>Maybe someone else just removed it."),
                Notification.TYPE_ERROR_MESSAGE)
            return
        if _IS_CUT:
            src_cal.parent = target
            src_cal.name = name
        else:
            try:
                dst_cal = sql_get_cal(session, target, self._workload)
            except:
                self._sql_session.close_session(session)
                self._c.get_application().showNotification(
                    _("Cannot get the destination directory details"),
                    _("<br/>Maybe someone else just removed it."),
                    Notification.TYPE_ERROR_MESSAGE)
                return
            cp_recursive(session, src_cal, dst_cal, name.encode('utf-8'))
        self._sql_session.close_session(session)
        self._c.repaint()

    def do_delete(self, target):
        """Delete a calendar or a directory.

        @param target:
                    database ID of the calendar or directory to delete.
        """
        try:
            obj = sql_get_cal(self._sql_session, target, self._workload)
        except:
            return
        d = DeleteBox(
            self._c.get_application(), _("Delete"),
            _('Are you sure you want to permanently delete "%s"?') %
            obj.name.encode('utf-8'),
            _('If you delete a directory, all its content will be \
            permanently lost.') if obj.entry_type != 0 else
            _('If you delete a calendar, it will be permanently lost.'))
        d.addListener(DeleteConfirmed(self._c, obj))

    def do_whatuses(self, target):
        """List the objects (jobs or jobsets) that are using the provided
        calendar or directory.

        @param target:
                    database ID of the calendar or directory to delete.
        """
        session = self._sql_session.open_session()
        try:
            cal = sql_get_cal(session, target, self._workload)
        except:
            self._sql_session.close_session(session)
            self._c.get_application().showNotification(
                _("Cannot get the calendar or directory details"),
                _("<br/>Maybe someone else just removed it."),
                Notification.TYPE_ERROR_MESSAGE)
            return
        p = PathCal(session, id=cal.id, workload=self._workload)
        # The jobs/jobsets that are using this calendar/directory are stored
        # in a list which is later displayed in a popup window.
        self._gathered = list()
        if cal.entry_type:
            cmd_cal.whatuses.print_whatuses_dir(session, cal, p,
                                                self._workload,
                                                self._gather, self)
        else:
            cmd_cal.whatuses.print_whatuses(session, cal, p,
                                            self._workload, self._gather, self)
        self._sql_session.close_session(session)
        if not self._gathered:
            self._c.get_application().showNotification(
                _("Not used"),
                _("<br/>%s is not used.") % cal.name.encode('utf-8'))
        else:
            WhereWindow(self._c, self._gathered, cal.entry_type)

    @staticmethod
    def _gather(self, cal_name, title, val):
        """Callback method for the L{cmd_cal.whatuses.print_whatuses_dir} and
        L{cmd_cal.whatuses.print_whatuses} functions.  This method is used to
        collect the jobs/jobsets that are using a calendar or directory.
        """
        self._gathered.append([cal_name, title, val])

    def do_new_cal(self, target):
        CalendarEditWindow(self._c, target)

    def do_new_directory(self, target):
        DirPopup(self._c, target)


class CutCopyPasteMenuDirectoryTree(CutCopyPasteMenu):

    def __init__(self, calendar_window_obj):
        super(CutCopyPasteMenuDirectoryTree, self).__init__(
            calendar_window_obj,
            calendar_window_obj._sql_session,
            calendar_window_obj._workload)

    def getActions(self, target, sender):
        if target is not None:
            if target == ROOT_CAL_DIR_ID:
                # Root Jobset
                return [self._ACTION_PASTE]
            else:
                return [self._ACTION_EDIT, self._ACTION_CUT,
                        self._ACTION_COPY, self._ACTION_PASTE,
                        self._ACTION_DELETE, self._ACTION_WHAT]


class CutCopyPasteMenuCalendarTable(CutCopyPasteMenu):
    def __init__(self, calendar_table_obj):
        super(CutCopyPasteMenuCalendarTable, self).__init__(
            calendar_table_obj,
            calendar_table_obj._sql_session,
            calendar_table_obj._workload)

    def getActions(self, target, sender):
        if not target:
            # Background
            return [self._ACTION_NEWDIR, self._ACTION_NEWCAL,
                    self._ACTION_PASTE]

        if self._c.is_directory(target):
            # Jobset
            return [self._ACTION_EDIT, self._ACTION_CUT,
                    self._ACTION_COPY, self._ACTION_PASTE,
                    self._ACTION_DELETE, self._ACTION_WHAT]
        else:
            return [self._ACTION_EDIT, self._ACTION_CUT, self._ACTION_COPY,
                    self._ACTION_DELETE, self._ACTION_WHAT]


class DirPopup(IClickListener):

    """Display the add/edit directory popup."""

    def __init__(self, obj, parent_id, dir_id=None, x=None, y=None):
        """Create the popup.

        @param obj:
                    L{CutCopyPasteMenuListener} object to retrieve the
                    details required by the menu.
        @param parent_id:
                database ID of the parent directory.
        @param dir_id:
                when the window is used to edit an existing directory,
                this is the database ID of this directory.
        @param x:
                the X coordinate of the mouse which is used to position
                the popup.  By default, the popup is centered.
        @param y:
                the Y coordinate of the mouse which is used to position
                the popup.  By default, the popup is centered.
        """
        super(DirPopup, self).__init__()
        self._c = obj
        self._parent_id = parent_id
        self._dir_id = dir_id

        if dir_id is None:
            title = _('New directory')
        else:
            title = _('Edit directory')
        h = HorizontalLayout()
        h.setWidth('100%')
        h.setMargin(True)
        h.setSpacing(True)

        self._name = TextField(_('Name:'))
        self._name.focus()
        self._name.setWidth('100%')
        h.addComponent(self._name)
        h.setExpandRatio(self._name, 1.0)

        if dir_id is not None:
            try:
                c = sql_get_cal(self._c._sql_session, dir_id,
                                self._c._workload)
            except:
                self._c._main_window.showNotification(
                    _("Cannot get the directory details"),
                    _("<br/>Maybe someone else just removed it."),
                    Notification.TYPE_ERROR_MESSAGE)
                return
            self._old_name = c.name
            self._name.setValue(c.name.encode('utf-8'))
        else:
            self._old_name = None

        ok = Button(_('OK'))
        ok.addListener(self, IClickListener)
        ok.setClickShortcut(KeyCode.ENTER)
        h.addComponent(ok)
        h.setComponentAlignment(ok, Alignment.BOTTOM_RIGHT)

        self._w = Window(title, h)
        self._w.setWidth('250px')
        self._c._main_window.addWindow(self._w)
        if x and y:
            self._w.setPositionX(x)
            self._w.setPositionY(y)
        else:
            self._w.center()

    def buttonClick(self, event):
        # Retrieve the entered values
        dir_name = self._name.getValue().strip()
        if not dir_name:
            self._c._main_window.showNotification(
                _("Invalid directory name"),
                _("<br/>The directory name cannot be empty or \
                just contain spaces"),
                Notification.TYPE_ERROR_MESSAGE)
            self._name.focus()
            return

        # Try to find if a directory with the same name already exists
        session = self._c._sql_session.open_session()
        if self._c._workload is None:
            query = session.query(calendars)
        else:
            query = session.query(calendars_s)
            query = query.filter(calendars_s.workload_date ==
                                 self._c._workload)
        query = query.filter_by(parent=self._parent_id)
        query = query.filter_by(name=dir_name.decode('utf-8'))
        try:
            query.one()
        except:
            pass
        else:
            if self._dir_id is None or \
               self._old_name != dir_name.decode('utf-8'):
                self._c._sql_session.close_session(session)
                self._c._main_window.showNotification(
                    _("The name is already used"),
                    _("<br/><br/>The specified name is already taken."),
                    Notification.TYPE_ERROR_MESSAGE)
                self._name.focus()
                return

        # New directory
        if self._dir_id is None:
            if self._c._workload is None:
                cal = calendars(self._parent_id, dir_name, 1, '', '')
            else:
                cal = calendars_s(self._parent_id, dir_name, 1, '', '',
                                  self._c._workload)
            session.add(cal)
        # Update
        elif self._old_name != dir_name.decode('utf-8'):
            try:
                cal = sql_get_cal(session, self._dir_id, self._c._workload)
            except:
                self._c._main_window.showNotification(
                    _("Cannot get the directory details"),
                    _("<br/>Maybe someone else just removed it."),
                    Notification.TYPE_ERROR_MESSAGE)
            else:
                cal.name = dir_name.decode('utf-8')
        self._c._sql_session.close_session(session)
        self._c.repaint()
        self._c._main_window.removeWindow(self._w)


class WhereWindow(IClickListener):

    """Window to display the list og jobs/jobsets using a calendar."""

    def __init__(self, obj, obj_list, cal_type):
        """Create and display the window.

        @param obj:
                    L{CutCopyPasteMenuListener} object to retrieve the
                    details required by the popup.
        @param obj_list:
                    list of jobs/jobsets.  Each object is an array of three
                    strings (cal_name, type, job_name).  The first one is the
                    calendar/directory name, the second is the type (job or
                    jobset) and the last one is the job/jobset
                    L{path.Path} object.
        @param cal_type:
                    type of the searched object: 0 for calendar and 1 for
                    directory
        """
        super(WhereWindow, self).__init__()

        self._c = obj

        self._w = Window(_("Jobs and jobsets"))
        self._w.setWidth("472px")
        self._w.setHeight("292px")

        # VerticalLayout as content by default
        v = self._w.getContent()
        v.setSizeFull()
        v.setMargin(True)
        v.setSpacing(True)

        t = Table()
        t.setSizeFull()
        t.setColumnReorderingAllowed(False)
        t.setColumnCollapsingAllowed(False)
        t.setSortDisabled(False)
        t.setSelectable(False)
        t.setMultiSelect(False)
        c = IndexedContainer()
        c.addContainerProperty('cal_name', str, None)
        c.addContainerProperty('type', str, None)
        c.addContainerProperty('value', str, None)
        for cal_name, k, val in obj_list:
            val = unicode(val)
            item = c.addItem(c.generateId())
            item.getItemProperty('cal_name').setValue(cal_name)
            item.getItemProperty('type').setValue(k)
            item.getItemProperty('value').setValue(val.encode('utf-8'))
        t.setContainerDataSource(c)
        t.setColumnExpandRatio('value', 1.0)
        if cal_type:
            t.setColumnExpandRatio('cal_name', 1.0)
            t.setColumnHeaders(['', _('Type'), _('Job/Jobset Name')])
        else:
            t.setVisibleColumns(['type', 'value'])
            t.setColumnHeaders([_('Type'), _('Job/Jobset Name')])
        v.addComponent(t)
        v.setExpandRatio(t, 1.0)

        # Close button
        ok = Button(_('Close'))
        ok.addListener(self, IClickListener)
        v.addComponent(ok)
        v.setComponentAlignment(ok, Alignment.BOTTOM_RIGHT)

        self._c.get_application().addWindow(self._w)
        self._w.center()

    def buttonClick(self, event):
        self._c.get_application().removeWindow(self._w)
