# Schedwi
# Copyright (C) 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Calendar view widget."""

import calendar
from datetime import datetime
from muntjac.api import (VerticalLayout, InlineDateField, Panel, Alignment)
from muntjac.ui.custom_layout import CustomLayout
from muntjac.data.property import IValueChangeListener

from cmd_cal import calcomp
import locale_utils


class CalendarView(VerticalLayout, IValueChangeListener):

    """Calendar view widget.

    The view show a year.  In the read/only mode, this calendar can
    visually represent a formula.  In edit mode, a formula can be built
    by clicking on days, month names or weekday names.

    """

    # List of days names in formula (for tooltips)
    _fomula_day_names = ["Monday", "Tuesday", "Wednesday", "Thursday",
                         "Friday", "Saturday", "Sunday"]

    def __init__(self, year=None, month_by_row=4, text_area_id=None):
        """Initialize the view.

        @param year:
                    year to initially display (by default the current year)
        @param month_by_row:
                    number of months to display on a row (4 by default - this
                    means 3 rows)
        @param text_area_id:
                    HTML tag ID of the text area to update when the user
                    click on a day, a month name or a weekday name. If not
                    set (the default), then the calendar view is read only.
        """
        super(CalendarView, self).__init__()
        self._year = year
        self._month_by_row = month_by_row
        self._text_area_id = text_area_id
        self._rows = 12 / month_by_row
        self._formula = None
        self._locale = locale_utils.get_locale()
        self._firstweekday = self._locale.first_week_day  # 0: Monday, ...

        d = datetime.today()
        if self._year is None:
            self._year = d.year
        else:
            d = datetime.datetime(self._year, d.month, 15)

        self.setSizeFull()
        self.setMargin(False)
        self.setSpacing(False)

        self._yearbt = InlineDateField()
        self._yearbt.setResolution(InlineDateField.RESOLUTION_YEAR)
        self._yearbt.setValue(d)
        self._yearbt.setImmediate(True)
        self._yearbt.addListener(self, IValueChangeListener)
        self.addComponent(self._yearbt)
        self.setComponentAlignment(self._yearbt, Alignment.MIDDLE_CENTER)

        p = Panel()
        p.setSizeFull()
        self.addComponent(p)
        self.setExpandRatio(p, 1.0)
        self._panel = p
        self.draw_calendar()

    def draw_calendar(self, year=None):
        """Display the calendar for the provided year.

        @param year:
                    year to display (by default the current year)
        """
        if year is not None:
            self._year = year
        c = calendar.Calendar(self._firstweekday)
        y = c.yeardatescalendar(self._year, self._month_by_row)
        calendar.setfirstweekday(self._firstweekday)

        # day_labels: list of local day names.
        # day_labels[0]: local name of the first day of the week
        # day_labels[1]: local name of the second day of the week ...
        day_labels = list()
        for i in range(self._firstweekday, 7 + self._firstweekday):
            # day_labels[0]: local name of the first day of the week, ...
            day_labels.append(self._locale.days['format']['narrow'][i % 7])

        fc = self._get_compiled_formula()

        # html: the HTML code is built in this html list
        html = list()

        # day_html_ids: list of all the days of the year.  Each item is
        # an array [month_number, day_number, is_selected, tag_id]
        # For instance [11, 23, false, 'd11/13']
        day_html_ids = list()

        # month_status: list of status (all days selected or not) for
        # each month.  This is used in the javascript code to determine
        # which kind of formula to buid (`Not <month>/*' or `<month>/*')
        # -1: unknown (initial state), 0: not all days are selected,
        # 1: all days are selected
        month_status = [-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1]

        # day_week_status: list of status (all days selected or not) for
        # the week day for a year (mondays, tuesdays, ...)  This is used in
        # the javascript code to determine which kind of formula to buid
        # (`Not */<weekday>' or `*/<weekday>')  -1: unknown (initial state),
        # 0: not all weekdays are selected, 1: all weekdays are selected
        day_week_status = [-1, -1, -1, -1, -1, -1, -1]

        for season in range(self._rows):
            # Title
            html.append('<tr>')
            for month in range(self._month_by_row):
                html.append('<th colspan="7" class="calmonthname">')
                month_number = season * self._month_by_row + month + 1
                if self._text_area_id is not None:
                    html.append('<a href="#" id="m%d" title="%d/*">' %
                                        (month_number, month_number))
                html.append(
                    self._locale.months['format']['wide'][month_number])
                if self._text_area_id is not None:
                    html.append('</a>')
                html.append('</th><th/>')
            html.append('</tr><tr>')
            for month in range(self._month_by_row):
                for weekday in range(7):
                    html.append('<th class="calweekdayname">')
                    if self._text_area_id is not None:
                        day_number = (weekday + self._firstweekday) % 7
                        html.append('<a href="#" class="w%d" title="*/%s">' %
                            (day_number, self._fomula_day_names[day_number]))
                    html.append(day_labels[weekday])
                    if self._text_area_id is not None:
                        html.append('</a>')
                    html.append('</th>')
                html.append('<th/>')
            html.append('</tr>')

            # Content
            for week in range(6):
                html.append('<tr>')
                for month in range(self._month_by_row):
                    for day in range(7):
                        try:
                            dt = y[season][month][week][day]
                        except:
                            # Week number may be too high (not all month have
                            # 6 weeks)
                            html.append('<td/>')
                            continue
                        m = season * self._month_by_row + month + 1
                        if dt.year == self._year and dt.month == m:
                            status_day = [dt.month, dt.day]
                            if (fc is not None and
                                fc.calmatch(dt.day, dt.month,
                                         dt.year) != calcomp.CAL_NOMATCH):
                                html.append('<td class="calday selected">')
                                status_day.append('true')
                                if month_status[m - 1] != 0:
                                    month_status[m - 1] = 1
                                wd = dt.weekday()
                                if day_week_status[wd] != 0:
                                    day_week_status[wd] = 1
                            else:
                                html.append('<td class="calday">')
                                status_day.append('false')
                                month_status[m - 1] = 0
                                day_week_status[dt.weekday()] = 0
                            if self._text_area_id is not None:
                                did = 'd%d/%d' % (dt.month, dt.day)
                                html.append(
                                        '<a href="#" id="%s" title="%d/%d">' %
                                                    (did, dt.month, dt.day))
                                status_day.append(did)
                                day_html_ids.append(status_day)
                            html.append(str(dt.day))
                            if self._text_area_id is not None:
                                html.append('</a>')
                            html.append('</td>')
                        else:
                            html.append('<td class="calday"/>')
                    html.append('<td/>')
                html.append('</tr>')
            html.append('<tr><td/></tr>')

        # HTML code
        js = list()
        if self._text_area_id is not None:
            js.append("""<script type="text/javascript">
/* Day names */
var week_days_list = new Array("Monday", "Tuesday", "Wednesday", "Thursday",
                               "Friday", "Saturday", "Sunday");
/* Already selected week days (none by default) */
var week_days = new Array(%s);

/* Already selected months (none by default) */
var months = new Array(%s);

/* Already selected days */
var days = new Array(12)
for (i = 0; i < 12; i++) {
    days[i] = new Array(31);
}


/*
 * Append the provided formula chunck to the Muntjac TextArea.
 *
 * @param formula:
 *         the formula to append at the end of the TextArea.
 */
function muntjac_append_formula(formula)
{
    var elem = document.getElementById("%s");
    elem.focus();
    elem.value += "\\n" + formula;
    elem.blur();
}


/*
 * Build the formula for a day in the year (month/day) and update Muntjac
 *
 * @param month:
 *         month number (between 1 and 12)
 * @param day:
 *         day number (between 1 and 31)
 */
function day_formula(month, day)
{
    var formula;

    if (days[month - 1][day - 1] == true) {
        formula = "Not " +  month + "/" + day;
        /*
         * No need to change the day state as Muntjac will
         * completly recreate the calendar view.
         */
        // days[month - 1][day - 1] = false;
    }
    else {
        formula = month + "/" + day;
    }
    muntjac_append_formula(formula);
}


/*
 * Build the formula for a weekday and update Muntjac.
 *
 * @param week_day_number:
 *         week day number (0 for Monday, 1 for Tuesday..., 6 for Sunday)
 * @param day:
 *         day number (between 1 and 31)
 */
function week_day_formula(week_day_number)
{
    var formula;

    if (week_days[week_day_number] == true) {
        formula = "Not */" + week_days_list[week_day_number];
    }
    else {
        formula = "*/" + week_days_list[week_day_number];
    }
    muntjac_append_formula(formula);
}


/*
 * Build the formula for all the day of a month and update Muntjac.
 *
 * @param month_number:
 *         month number (1 to 12)
 */
function month_formula(month_number)
{
    var formula;

    if (months[month_number - 1] == true) {
        formula = "Not " + month_number + "/*";
    }
    else {
        formula = month_number + "/*";
    }
    muntjac_append_formula(formula);
}
""" % (", ".join(map(lambda i: "true" if i else "false", day_week_status)),
       ", ".join(map(lambda i: "true" if i else "false", month_status)),
       self._text_area_id))
            for d in day_html_ids:
                # d[0] --> Month
                # d[1] --> Day number in the month
                # d[2] --> Is day selected (true/false)?
                # d[3] --> HTML id
                js.append('days[' + str(d[0] - 1) + '][' + str(d[1] - 1) +
                          '] = ' + d[2] + ';')
                js.append('document.getElementById("' + d[3] + '").onclick =' +
                          ' function () { day_formula(' + str(d[0]) + ', ' +
                          str(d[1]) + '); return false; };')
            for m in range(1, 13):
                js.append('document.getElementById("m' + str(m) +
                          '").onclick = function () { month_formula(' +
                          str(m) + '); return false; };')
            for wd in range(7):
                js.append('var a = document.getElementsByClassName("w' +
                          str(wd) + '");')
                js.append('for (var i = 0; i < a.length; i++) {')
                js.append('a[i].onclick = function () { week_day_formula(' +
                          str(wd) + '); return false; }; }')
            js.append('</script>')

        c = CustomLayout()
        c.setTemplateContents('<table class="caltable">' +
                              ''.join(html).encode('utf-8') +
                              '</table>' +
                              '\n'.join(js))
        self._panel.setContent(c)

    def _get_compiled_formula(self):
        """Get the calendar formula in its compiled form.

        @return:    the compiled formula or None if no formula has been
                    set (see L{set_formula})
        @rtype:     L{cmd_cal.calcomp.cal_t}
        """
        if self._formula is None:
            return None
        else:
            err, idx, c = calcomp.str2cal(self._formula, self._year)
            return c

    def set_formula(self, formula):
        """Set the calendar formula to preview.

        @param formula:
                    the formula to display.  If None, no formula is showned
        @return:
                    None on success or an error message (string) if
                    the formula contains errors.
        """
        # Sanity check: compile the formula to catch errors
        if formula is not None:
            err, idx, c = calcomp.str2cal(formula, self._year)
            if err == calcomp.CAL_EMPTYFIELD:
                return _("%s: syntax error: empty field\n") % formula
            elif err == calcomp.CAL_BADMONTHNAME:
                return _("%s: syntax error: invalid month name\n") % formula
            elif err == calcomp.CAL_BADDAYNAME:
                return _("%s: syntax error: invalid day name\n") % formula
            elif err == calcomp.CAL_MONTHOOR:
                return _("%s: syntax error: month number out of range\n") % \
                                                        formula
            elif err == calcomp.CAL_DAYOOR:
                return _("%s: syntax error: day number out of range\n") % \
                                                        formula
            elif err == calcomp.CAL_BADOPTION:
                return _("%s: syntax error: invalid option flag\n") % formula
            elif err != calcomp.CAL_NOERROR:
                return _("%s: syntax error\n") % formula
        self._formula = formula
        self.draw_calendar()
        return None

    def valueChange(self, event):
        """Callback for year button."""
        self.draw_calendar(event.getProperty().getValue().year)
