# Schedwi
# Copyright (C) 2011-2015 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import sqlalchemy.orm.session
from sqlalchemy.schema import Column
from sqlalchemy.types import Integer, Text, String
from sqlalchemy.orm import relationship
from sqlalchemy.sql import func

from base import Base, convert2unicode

from job_start_limit_s import job_start_limit_s
from job_max_duration_s import job_max_duration_s
from job_retries_s import job_retries_s
from job_retries_interval_s import job_retries_interval_s
from job_username_s import job_username_s
from job_file_out_s import job_file_out_s
from job_file_err_s import job_file_err_s
from job_control_group_s import job_control_group_s
from job_loadenv_s import job_loadenv_s
from job_detach_s import job_detach_s
from job_manual_command_s import job_manual_command_s
from job_success_return_code_s import job_success_return_code_s
from job_command_s import job_command_s
from job_arguments_s import job_arguments_s
from job_environment_s import job_environment_s
from job_host_s import job_host_s
from job_cluster_s import job_cluster_s
from constraint_file_s import constraint_file_s
from link_s import link_s
from job_status import job_status


class job_main_s(Base):
    __tablename__ = 'job_main_s'

    id = Column(Integer, primary_key=True)
    parent = Column(Integer)
    name = Column(String(255))
    rw_name = True
    type = Column(Integer)   # 0 (jobset) or 1 (job)
    rw_type = False
    enabled = Column(Integer)
    rw_enabled = True
    has_env = Column(Integer)
    rw_has_env = True
    manual = Column(Integer)
    rw_manual = True
    description = Column(Text)
    rw_description = True
    x = Column(Integer)
    rw_x = True
    y = Column(Integer)
    rw_y = True
    cal_id = Column(Integer)
    rw_cal_id = True
    start_time = Column(Integer)  # In seconds since epoch (time(2))
    rw_start_time = True
    workload_date = Column(Integer, primary_key=True)

    job_start_limit = relationship(job_start_limit_s,
                                   uselist=False,
                                   backref="job_main_s",
                                   cascade="all, delete, delete-orphan")
    rw_job_start_limit = True
    job_max_duration = relationship(job_max_duration_s,
                                    uselist=False,
                                    backref="job_main_s",
                                    cascade="all, delete, delete-orphan")
    rw_job_max_duration = True
    job_retries = relationship(job_retries_s,
                               uselist=False,
                               backref="job_main_s",
                               cascade="all, delete, delete-orphan")
    rw_job_retries = True
    job_retries_interval = relationship(job_retries_interval_s,
                                        uselist=False,
                                        backref="job_main_s",
                                        cascade="all, delete, delete-orphan")
    rw_job_retries_interval = True
    job_username = relationship(job_username_s,
                                uselist=False,
                                backref="job_main_s",
                                cascade="all, delete, delete-orphan")
    rw_job_username = True
    job_file_out = relationship(job_file_out_s,
                                uselist=False,
                                backref="job_main_s",
                                cascade="all, delete, delete-orphan")
    rw_job_file_out = True
    job_file_err = relationship(job_file_err_s,
                                uselist=False,
                                backref="job_main_s",
                                cascade="all, delete, delete-orphan")
    rw_job_file_err = True
    job_control_group = relationship(job_control_group_s,
                                     uselist=False,
                                     backref="job_main_s",
                                     cascade="all, delete, delete-orphan")
    rw_job_control_group = True
    job_loadenv = relationship(job_loadenv_s, uselist=False,
                               backref="job_main_s",
                               cascade="all, delete, delete-orphan")
    rw_job_loadenv = True
    job_detach = relationship(job_detach_s, uselist=False,
                              backref="job_main_s",
                              cascade="all, delete, delete-orphan")
    rw_job_detach = True
    job_manual_command = relationship(job_manual_command_s, uselist=False,
                                      backref="job_main_s",
                                      cascade="all, delete, delete-orphan")
    rw_job_manual_command = True
    job_success_return_code = relationship(
        job_success_return_code_s,
        uselist=False,
        backref="job_main_s",
        cascade="all, delete, delete-orphan")
    rw_job_success_return_code = True
    job_command = relationship(job_command_s,
                               uselist=False,
                               backref="job_main_s",
                               cascade="all, delete, delete-orphan")
    rw_job_command = True
    job_arguments = relationship(job_arguments_s,
                                 order_by=job_arguments_s.position,
                                 backref="job_main_s",
                                 cascade="all, delete, delete-orphan")
    rw_job_arguments = True
    job_environment = relationship(job_environment_s,
                                   order_by=job_environment_s.position,
                                   backref="job_main_s",
                                   cascade="all, delete, delete-orphan")
    rw_job_environment = True
    job_host = relationship(job_host_s, uselist=False, backref="job_main_s",
                            cascade="all, delete, delete-orphan")
    rw_job_host = True
    job_cluster = relationship(job_cluster_s, uselist=False,
                               backref="job_main_s",
                               cascade="all, delete, delete-orphan")
    rw_job_cluster = True
    constraint_file = relationship(constraint_file_s,
                                   order_by=constraint_file_s.exist,
                                   backref="job_main_s",
                                   cascade="all, delete, delete-orphan")
    rw_constraint_file = True
    links = relationship(
        link_s, backref="job_main_s",
        primaryjoin="and_(link_s.job_id_source==job_main_s.id, "
                    "link_s.workload_date==job_main_s.workload_date)",
        cascade="all, delete, delete-orphan")
    rw_links = True

    link_targets = relationship(
        link_s,
        primaryjoin="and_(link_s.job_id_destination == job_main_s.id, "
                    "link_s.workload_date == job_main_s.workload_date)",
        cascade="all, delete, delete-orphan")
    rw_link_targets = True

    job_status = relationship(job_status,
                              uselist=False,
                              backref="job_main_s",
                              cascade="all, delete, delete-orphan")
    rw_job_status = False

    def __init__(self, id, parent, name, type, enabled, has_env, manual,
                 description, x, y, cal_id, start_time, workload_date):
        self.id = id
        self.parent = parent
        self.name = convert2unicode(name)
        self.type = type
        self.enabled = enabled
        self.has_env = has_env
        self.manual = manual
        self.description = convert2unicode(description)
        self.x = x
        self.y = y
        self.cal_id = cal_id
        self.start_time = start_time
        self.workload_date = workload_date

    def copy(self, new_id):
        j = job_main_s(new_id, self.parent, self.name, self.type, self.enabled,
                       self.has_env, self.manual, self.description,
                       self.x, self.y, self.cal_id, self.start_time,
                       self.workload_date)
        if self.job_start_limit:
            j.job_start_limit = self.job_start_limit.copy()
        else:
            j.job_start_limit = self.job_start_limit
        if self.job_max_duration:
            j.job_max_duration = self.job_max_duration.copy()
        else:
            j.job_max_duration = self.job_max_duration
        if self.job_retries:
            j.job_retries = self.job_retries.copy()
        else:
            j.job_retries = self.job_retries
        if self.job_retries_interval:
            j.job_retries_interval = self.job_retries_interval.copy()
        else:
            j.job_retries_interval = self.job_retries_interval
        if self.job_username:
            j.job_username = self.job_username.copy()
        else:
            j.job_username = self.job_username
        if self.job_file_out:
            j.job_file_out = self.job_file_out.copy()
        else:
            j.job_file_out = self.job_file_out
        if self.job_file_err:
            j.job_file_err = self.job_file_err.copy()
        else:
            j.job_file_err = self.job_file_err
        if self.job_control_group:
            j.job_control_group = self.job_control_group.copy()
        else:
            j.job_control_group = self.job_control_group
        if self.job_loadenv:
            j.job_loadenv = self.job_loadenv.copy()
        else:
            j.job_loadenv = self.job_loadenv
        if self.job_detach:
            j.job_detach = self.job_detach.copy()
        else:
            j.job_detach = self.job_detach
        if self.job_manual_command:
            j.job_manual_command = self.job_manual_command.copy()
        else:
            j.job_manual_command = self.job_manual_command
        if self.job_success_return_code:
            j.job_success_return_code = self.job_success_return_code.copy()
        else:
            j.job_success_return_code = self.job_success_return_code
        if self.job_command:
            j.job_command = self.job_command.copy()
        else:
            j.job_command = self.job_command
        j.job_arguments = list()
        for arg in self.job_arguments:
            j.job_arguments.append(arg.copy())
        j.job_environment = list()
        for env in self.job_environment:
            j.job_environment.append(env.copy())
        if self.job_host:
            j.job_host = self.job_host.copy()
        else:
            j.job_host = self.job_host
        if self.job_cluster:
            j.job_cluster = self.job_cluster.copy()
        else:
            j.job_cluster = self.job_cluster
        j.constraint_file = list()
        for f in self.constraint_file:
            j.constraint_file.append(f.copy())
        j.links = list()
        for link in self.links:
            j.links.append(link.copy())
        if self.job_status:
            j.job_status = self.job_status.copy()
        else:
            j.job_status = self.job_status
        return j


def get_next_id(sql_session, workload):
    """Return the next available ID from the job_main_s database table.

    @param sql_session:
                SQLAlchemy session (it can be an opened session)
    @param workload:
                the workload to look for (YYYYMMDD)
    @return:
                The next available ID.
    """
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
    else:
        session = sql_session

    query = session.query(func.max(job_main_s.id).label("max_id"))
    try:
        res = query.filter(job_main_s.workload_date == workload).one()
    except:
        new_id = 1
    else:
        new_id = res.max_id + 1

    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return new_id
