# Schedwi
# Copyright (C) 2013-2015 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Simple database queries for workloads."""

import datetime
import babel.dates

import sqlalchemy.orm.session
from sqlalchemy.orm.exc import NoResultFound
from sqlalchemy import desc

from tables.job_status import job_status
from tables.job_main import ROOT_JOBSET_ID
from tables.job_main_s import job_main_s
import locale_utils
import status_utils


def sql_get_workload_list(sql_session):
    """Return the list of workload objects.

    @param sql_session:
                SQLAlchemy session (it can be an opened session)
    @return:
                the workload list.  Each item is of type
                L{tables.job_status.job_status}
    @raise sqlalchemy.orm.exc.NoResultFound:
                no workloads in the database.
    """
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
    else:
        session = sql_session

    query = session.query(job_status).filter(job_status.job_id ==
                                             ROOT_JOBSET_ID)
    try:
        lst = query.order_by(desc(job_status.workload_date)).all()
    except:
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)
        raise

    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return lst


def sql_get_workload_list_text(sql_session):
    """Return the list of workload in a more human readable format.

    @param sql_session:
                SQLAlchemy session (it can be an opened session)
    @return:
                the workload list.  Each item is a dictionnary with the
                following keys:

                  - date: the workload date from the database (YYYYMMDD)
                  - date_str: the locale representation of the date
                  - datetime: the associated L{datetime.date} object
                  - status: the workload status code from the database
                  - status_str: the status name

    @raise sqlalchemy.orm.exc.NoResultFound:
                no workloads in the database.
    """
    lst = list()
    for js in sql_get_workload_list(sql_session):
        d = datetime.date(js.workload_date / 10000,
                          js.workload_date % 10000 / 100,
                          js.workload_date % 100)
        date_str = babel.dates.format_date(d, format="full",
                                           locale=locale_utils.get_locale())
        lst.append({"date": js.workload_date,
                    "date_str": date_str.encode('utf-8'),
                    "datetime": d,
                    "status": js.status,
                    "status_str": status_utils.status2string(js.status)})
    return lst


def does_workload_exist(sql_session, workload):
    """Check if a workload exists.

    @param sql_session:
                SQLAlchemy session (it can be an opened session)
    @param workload:
                the workload to look for (YYYYMMDD)
    @return:
                True if the workload has been found in the database,
                False otherwise
    """
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
    else:
        session = sql_session
    query = session.query(job_status).filter(job_status.job_id ==
                                             ROOT_JOBSET_ID)
    try:
        query.filter(job_status.workload_date == workload).one()
    except NoResultFound:
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)
        return False

    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return True


def num_jobs_by_status(sql_session, workload):
    """Return the number of job/jobset by status.

    @param sql_session:
                SQLAlchemy session (it can be an opened session)
    @param workload:
                the workload to look for (YYYYMMDD)
    @return:
                the tuple (num_waiting, num_running, num_completed, num_failed)
    """
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
    else:
        session = sql_session

    query = session.query(job_status)
    query = query.filter(job_status.workload_date == workload)
    query = query.filter(job_status.status == status_utils.WAITING)
    # Do not count disabled jobs and jobsets
    query = query.filter(job_status.job_id == job_main_s.id)
    query = query.filter(job_main_s.enabled != 0)
    num_waiting = query.count()

    query = session.query(job_status)
    query = query.filter(job_status.workload_date == workload)
    query = query.filter(job_status.status == status_utils.RUNNING)
    num_running = query.count()

    query = session.query(job_status)
    query = query.filter(job_status.workload_date == workload)
    query = query.filter(job_status.status == status_utils.COMPLETED)
    num_completed = query.count()

    query = session.query(job_status)
    query = query.filter(job_status.workload_date == workload)
    query = query.filter(job_status.status == status_utils.FAILED)
    num_failed = query.count()

    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return (num_waiting, num_running, num_completed, num_failed)
