# Schedwi
# Copyright (C) 2011-2015 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to list workloads and to change the current workload."""

import sys
import getopt
import datetime

from help import print_trim
import simple_queries_wl


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: wl [WORKLOAD]
    List all workloads, or, if WORKLOAD is given, enter the specified workload
    (all subsequent commands will apply to this workload objects - type `exit'
    to leave this mode).
    WORKLOAD can be:
      - a workload date YYYYMMDD
      - 0 or a negative number.  0 means today's workload, -1 means
        yesterdays, ...
      - the keywords `Today' and `Yesterday'

    Option:
      -h, --help  display this help
    """))


def parse(sql_session, w):
    """Parse the given workload string

    Returns the tuple (workload, error_message) with workload the parsed
    workload or None in case of error.  In this later case, error_message
    contains an error message.

    """
    try:
        delta = int(w)
    except:
        s = w.lower()
        if s == "today":
            delta = 0
        elif s == "yesterday":
            delta = -1
        else:
            return (None, _("`%s': wrong workload format") % w.decode('utf-8'))
    if delta <= 0:
        workload_date = datetime.date.today() + datetime.timedelta(delta)
        workload = workload_date.strftime("%Y%m%d")
    else:
        # A proper workload date has been provided (ie. 20101224)
        workload = str(delta)
    # Check that the workload exists
    if not simple_queries_wl.does_workload_exist(sql_session, workload):
        return (None, _("`%s': no such workload") % w)
    return (workload, "")


def _print_workload(lst):
    """Print workloads."""
    for item in lst:
        print "%s - %s - %s" % (item["date"],
                                item["date_str"],
                                item["status_str"])


def wl(sql_session, arguments):
    """List workloads and change the current workload.

    Arguments:
    sql_session -- SQLAlchemy session
    arguments -- list of arguments given to the wl command.  Only the first
                 one is used and specifies the workload.  Is no argument is
                 provided, simply list all the available workloads.

    """
    try:
        optlist, args = getopt.getopt(arguments, 'h', ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write(_("wl: ") + str(err) + "\n")
        return None
    for o, a in optlist:
        if o in ("-h", "--help"):
            usage()
            return None
    if not args:
        # List current workloads
        try:
            lst = simple_queries_wl.sql_get_workload_list_text(sql_session)
        except:
            return None
        _print_workload(lst)
        return None
    workload, err_msg = parse(sql_session, args[0])
    if workload is None:
        sys.stderr.write(_("wl: ") + err_msg + "\n")
        return None
    return workload
