# Schedwi
# Copyright (C) 2013-2015 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to trigger/acknowledge manual jobs and jobsets."""

import sys
import getopt

from datetime import datetime
import babel.dates
import locale_utils
import path
import config
from simple_queries_ack import (sql_get_acknowledge_request_by_job_id,
                                sql_get_acknowledge_request_by_workload)
from help import print_trim


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: trigger [OPTION]... JOB|JOBSET...
        List and trigger the start of pending manual JOBs and JOBSETs.
        JOBs and JOBSETs can be expressed as a path but also by a workload
        and a job/jobset ID according to the followin format:
           workload_jobID
        For instance 20131201_101

        Options:
          -l, --list  list all the pending and acknowledged jobs and jobsets
          -h, --help  display this help.
    """))


def _print_ack(session, path, ack=None):
    """Print the status of the provided acknowledge request.

    @param session:
                An opened SQLAlchemy session.
    @param path:
                the L{path.Path} object matching the ack parameter.
    @param ack:
                The L{tables.acknowledge_manual} object to display.  If None,
                the entry will be retrieved automatically.
    @return:
                0 on success or 1 if the job/jobset cannot be retrieved from
                the acknowledge_manual table.
    """
    if ack is None:
        # Check that the job/jobset is actually in the database
        try:
            ack = sql_get_acknowledge_request_by_job_id(session, path.id[-1],
                                                        path.workload)
        except:
            sys.stderr.write(
                _("%s: no such pending or acknowledged job/jobset\n") % path)
            return 1

    if ack.status == 0:
        print _("%(name)s (%(wordkload)d_%(jobid)d) waiting for \
acknowledgement since %(datetime)s") % \
            {
                "name": path,
                "wordkload": ack.workload_date,
                "jobid": ack.job_id,
                "datetime": babel.dates.format_datetime(
                    ack.ack_time,
                    format="full",
                    locale=locale_utils.get_locale())}
    else:
        print _("%(name)s (%(wordkload)d_%(jobid)d) acknowledged by \
%(username)s at %(datetime)s") % \
            {
                "name": path,
                "wordkload": ack.workload_date,
                "jobid": ack.job_id,
                "username": ack.username,
                "datetime": babel.dates.format_datetime(
                    ack.ack_time,
                    format="full",
                    locale=locale_utils.get_locale())}
    return 0


def trigger_path(session, path):
    """Change the status of the pending job/jobset in the database.

    @param session:
                An opened SQLAlchemy session.
    @param path:
                the L{path.Path} object.
    @return:
                0 on success and 1 otherwise.
    """
    if path.workload is None:
        sys.stderr.write(_("trigger: `%s': workload not set (see `help wl')\n")
                         % path)
        return 1
    # Check that the job/jobset is actually in the database
    try:
        ack = sql_get_acknowledge_request_by_job_id(session, path.id[-1],
                                                    path.workload)
    except:
        sys.stderr.write(_("trigger: `%s' is not pending\n")
                         % path)
        return 1
    # Check that the job/jobset is still pending
    if ack.status != 0:
        sys.stderr.write(_("trigger: `%s': already acknowledged by %s at %s\n")
                         % (path, ack.username,
                            babel.dates.format_datetime(
                                ack.ack_time,
                                format="full",
                                locale=locale_utils.get_locale())))
        return 1
    ack.status = 1
    ack.ack_time = datetime.now()
    ack.username = "tui admin"   # .decode('utf-8')
    return 0


def trigger(sql_session, current_cwd, arguments, workload=None):
    """Trigger the start of pending manual jobs and jobsets.

    @param sql_session:
                The SQLAlchemy session object.
    @param current_cwd:
                Current working jobset (a L{path.Path} object)
    @param arguments:
                List of arguments given to the trigger command (list
                of jobs/jobsets)
    @param workload:
                The associated workload. If None, all the jobs/jobsets must
                have the <workload>_<jobid> format.
    """
    try:
        optlist, args = getopt.getopt(arguments, "lh", ["list", "help"])
    except getopt.GetoptError, err:
        sys.stderr.write(_("trigger: ") + str(err) + "\n")
        return 1
    listing_mode = False
    for o, a in optlist:
        if o in ("-l", "--list"):
            listing_mode = True
        if o in ("-h", "--help"):
            usage()
            return 0
    paths = list()
    if args:
        for arg in args:
            p = path.get_paths(sql_session, arg, current_cwd,
                               workload=workload)
            # An argument is an unknow path
            if not p:
                sys.stderr.write(_("trigger: `%s': no such job or jobset\n")
                                 % arg.decode('utf-8'))
                return 1
            paths.extend(p)
    elif not listing_mode:
        sys.stderr.write(_("trigger: missing job or jobset operand\n"))
        return 1
    ret = 0
    session = sql_session.open_session()
    if listing_mode:
        if paths:
            for p in paths:
                if p.workload is None:
                    sys.stderr.write(
                        _("trigger: `%s': workload not set (see `help wl')\n")
                        % p)
                    ret += 1
                else:
                    ret += _print_ack(session, p)
        else:
            if workload is None:
                sql_session.close_session(session)
                sys.stderr.write(
                    _("trigger: workload not set (see `help wl')\n"))
                return 1
            for a in sql_get_acknowledge_request_by_workload(session,
                                                             workload):
                p = path.id2path(session, a.job_id, workload)
                ret += _print_ack(session, p, a)
    else:
        for p in paths:
            ret += trigger_path(session, p)
        if ret == 0 and config.ISATTY_STDOUT:
            print _("This may take up to a minute to be taken into account")
    sql_session.close_session(session)
    return 1 if ret else 0
