# Schedwi
# Copyright (C) 2011-2015 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to list job and jobset status."""

import sys
import getopt
import time

from sqlalchemy import desc
from sqlalchemy.orm.exc import NoResultFound, MultipleResultsFound

import path
import simple_queries_job
from tables.job_status import job_status
import status_utils
import host_utils
import colors
import wl
from help import print_trim


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: status [OPTIONS]...
Get the status of jobs and jobsets.

Options:
  -d                      list jobsets only.
  -f, --date-fmt=FORMAT   date/time format to use to display the time of the
                          status change. See strftime(3) for supported formats.
                          By default `%c'
  -F                      append  a `/' for jobsets.
  --host=HOSTNAME[:PORT]  filter by the given host.
  -n                      do not display job/jobset current status.
  -N                      do not display job/jobset status message.
  --noreport              omits printing of the job and jobset report at the
                          end of the listing.
  -s, --status=STATUS,... only list jobs and jobsets with the specified STATUS
                          (`running', `completed', `failed', `waiting' or
                          `all').  By default only failed jobs and jobsets are
                          displayed.
  -t                      do not display the time of the status change.
  -w, --workload=WORKLOAD only show job and jobset status for the specified
                          workload.  By default only the current workload is
                          displayed if a workload has been selected by the
                          `wl' command.  Otherwise, all the workloads are
                          displayed.
  -h, --help              display this help.
    """))


def status(sql_session, arguments, workload):
    """Start jobs and jobsets.

    Arguments:
    sql_session -- SQLAlchemy session
    arguments -- list of arguments given to the status command
    workload -- workload to use

    """
    try:
        optlist, args = getopt.getopt(arguments,
                                      "df:FnNs:tw:h",
                                      [
                                          "help",
                                          "date-fmt=",
                                          "host=",
                                          "noreport",
                                          "status=",
                                          "workload="])
    except getopt.GetoptError, err:
        sys.stderr.write(_("status: ") + str(err) + "\n")
        return 1
    options = {'only_dir': False,
               'host': None,
               'print_report': True,
               'append_type': False,
               'print_message': True,
               'status': [status_utils.FAILED],
               'workload': workload,
               'show_status': True,
               'time_format': "%c",
               'print_time': True
               }
    for o, a in optlist:
        if o == "-d":
            options['only_dir'] = True

        elif o in ("-f", "--date-fmt"):
            options['time_format'] = a

        elif o == "-F":
            options['append_type'] = True

        elif o == "--host":
            try:
                host = host_utils.name2host(sql_session, a)
            except NoResultFound:
                sys.stderr.write(_("status: no such host (--host=%s)\n") % a)
                return 1
            except MultipleResultsFound:
                sys.stderr.write(_("status: more than one host match the \
provided name (--host=%s)\n") % a)
                return 1
            options['host'] = host.id

        elif o == "--noreport":
            options['print_report'] = False

        elif o == "-n":
            options['show_status'] = False

        elif o == "-N":
            options['print_message'] = False

        elif o in ("-s", "--status"):
            options['status'] = list()
            for s in a.split(','):
                if "all" in s.lower():
                    options['status'] = status_utils.status_list()
                    break
                status_code = status_utils.string2status(s)
                if not status_code:
                    sys.stderr.write(_("status: unknown status %s (%s)\n") %
                                     (s.decode('utf-8'), o))
                    return 1
                options['status'].append(status_code)

        elif o == "-t":
            options['print_time'] = False

        elif o in ("-w", "--workload"):
            w, err_msg = wl.parse(sql_session, a)
            if w is None:
                sys.stderr.write("status: " + err_msg + "\n")
                return 1
            options['workload'] = w

        elif o in ("-h", "--help"):
            usage()
            return 0
    session = sql_session.open_session()
    query = session.query(job_status)
    if options['workload'] is not None:
        query = query.filter(job_status.workload_date == options['workload'])
    if options['host'] is not None:
        query = query.filter(job_status.host_id == options['host'])
    query = query.filter(job_status.status.in_(options['status']))
    query = query.order_by(desc(job_status.workload_date))
    query = query.order_by(desc(job_status.time_status_set))
    prev_workload = None
    counter_status = [0, 0, 0, 0, 0]
    space = ' ' * (1 - colors.CHAR_COLOR_LEN)
    for j in query.all():
        # Retrieve the full path name
        try:
            p = path.id2path(session, j.job_id, j.workload_date)
        except NoResultFound:
            continue
        # Retrieve the job (to check if it's a job or a jobset)
        try:
            job = simple_queries_job.sql_get_job(session, j.job_id,
                                                 options['workload'])
        except NoResultFound:
            continue
        if options['only_dir'] and job.type != 0:
            continue
        counter_status[j.status] += 1
        # New workload
        if options['workload'] is None and j.workload_date != prev_workload:
            if prev_workload is not None:
                print
            print str(j.workload_date) + ":"
            prev_workload = j.workload_date
        if options['show_status']:
            status_color_start = status_utils.status2color(j.status)
            status_char = space + status_utils.status2char(j.status) + space
            status_color_end = colors.NORMAL
        else:
            status_color_start, status_char, status_color_end = '', '', ''
        if options['append_type'] and job.type == 0 and job.id != 1:
            type_char = '/'
        else:
            type_char = ''
        message = ''
        if options['print_message']:
            if j.status == status_utils.WAITING:
                reason = j.get_waiting_message()
                if reason:
                    message = " - %s" % reason
            elif j.error_msg:
                message = _(" - %s - try %d") % (j.error_msg, j.retry_num)
        if options['print_time']:
            date_time = time.strftime(options['time_format'], time.localtime(
                                      j.time_status_set)) + ' - '
        else:
            date_time = ''
        print "%s%s%s%s%s%s%s" % (date_time,
                                  status_color_start, status_char,
                                  status_color_end, p, type_char, message)
    sql_session.close_session(session)
    if options['print_report']:
        print
        if status_utils.WAITING in options['status']:
            print "%4d %s%s%s" % (
                counter_status[status_utils.WAITING],
                status_utils.status2color(status_utils.WAITING),
                status_utils.status2string(status_utils.WAITING),
                colors.NORMAL)
        if status_utils.RUNNING in options['status']:
            print "%4d %s%s%s" % (
                counter_status[status_utils.RUNNING],
                status_utils.status2color(status_utils.RUNNING),
                status_utils.status2string(status_utils.RUNNING),
                colors.NORMAL)
        if status_utils.COMPLETED in options['status']:
            print "%4d %s%s%s" % (
                counter_status[status_utils.COMPLETED],
                status_utils.status2color(status_utils.COMPLETED),
                status_utils.status2string(status_utils.COMPLETED),
                colors.NORMAL)
        if status_utils.FAILED in options['status']:
            print "%4d %s%s%s" % (
                counter_status[status_utils.FAILED],
                status_utils.status2color(status_utils.FAILED),
                status_utils.status2string(status_utils.FAILED),
                colors.NORMAL)
    return 0
