# Schedwi
# Copyright (C) 2011-2015 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to move or rename jobs and jobsets."""

import sys
import getopt

import path
import simple_queries_job
from tables.job_main import job_main
from tables.job_main_s import job_main_s
from help import print_trim


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: mv SOURCE DEST
   or: mv SOURCE... JOBSET
        Rename SOURCE to DEST, or move SOURCE(s) to JOBSET.

        Option:
          -h, --help  display this help.
    """))


def rename(sql_session, cwd, src_path, dst, workload):
    """Rename the given job/jobset.

    Arguments:
    sql_session -- SQLAlchemy session
    cwd -- current working jobset (a path.Path object)
    src_path -- the source job/jobset to copy
    dst -- the destination
    workload -- workload to use

    """
    (base, dest_path) = path.get_new_paths(sql_session, dst, cwd, workload)
    lp = len(dest_path)
    if lp == 0:
        sys.stderr.write(_("mv: no such destination jobset\n"))
        return 1
    if lp > 1:
        sys.stderr.write(_("mv: `%s/%s': no such job or jobset\n") %
                         (dest_path[0], base))
        return 1
    session = sql_session.open_session()

    dstjobset = simple_queries_job.sql_get_job(session, dest_path[0].id[-1],
                                               workload)
    try:
        src_job = simple_queries_job.sql_get_job(session, src_path.id[-1],
                                                 workload)
    except:
        sys.stderr.write(_("mv: `%s': no such job or jobset\n") % src_path)
        sql_session.cancel_session(session)
        return 1
    src_job.parent = dstjobset.id
    src_job.name = base.decode('utf-8')
    sql_session.close_session(session)
    return 0


def _one_move(session, source, dest_jobset, dest_as_path, workload=None):
    """Move one object into a jobset.

    @param session:
                an open SQLAlchemy session.
    @param source:
                the source object to move (a L{path.Path} object)
    @param dest_jobset:
                the destination jobset (a L{tables.job_main.job_main} or
                L{tables.job_main_s.job_main_s} object)
    @param dest_as_path:
                the destination jobset as a L{path.Path} object.
    @param workload:
                the workload to look for (YYYYMMDD)
    @raise sqlalchemy.orm.exc.NoResultFound:
                the source object does not exist
    @raise Exception:
                on error
    """
    # Check that the destination is a jobset
    if dest_jobset.type != 0:
        sys.stderr.write(_("mv: target `%s' is not a jobset\n") % dest_as_path)
        raise Exception("Target is not a jobset")

    # Get details on the source object
    try:
        job = simple_queries_job.sql_get_job(session, source.id[-1], workload)
    except:
        sys.stderr.write(_("mv: `%s': no such job or jobset\n") % source)
        raise

    # Check if the destination does not already exists
    if workload is None:
        query = session.query(job_main)
    else:
        query = session.query(job_main_s)
        query = query.filter(job_main_s.workload_date == workload)
    query = query.filter_by(parent=dest_jobset.id)
    try:
        query = query.filter_by(name=job.name).all()
    except:
        pass
    else:
        if len(query) != 0:
            sys.stderr.write(_("mv: `%s' already exists in `%s'\n")
                             % (job.name, dest_as_path))
            raise ("Destination already exists")

    # Check if the destination jobset is not a child of the source
    try:
        dest_as_path.id.index(job.id)
    except:
        job.parent = dest_jobset.id
    else:
        sys.stderr.write(
            _("mv: cannot move `%s' to a sub-jobset of itself\n") % source)
        raise("Try to move in a sub-jobset of itself")


def move(sql_session, paths, workload):
    """Move jobs/jobsets in a new jobset.

    Arguments:
    sql_session -- SQLAlchemy session
    paths -- list of jobs/jobsets to move (the last item is the destination)
    workload -- workload to use

    """
    dest = paths.pop()
    session = sql_session.open_session()
    try:
        dest_jobset = simple_queries_job.sql_get_job(session, dest.id[-1],
                                                     workload)
    except:
        sys.stderr.write(_("mv: no such destination job or jobset\n"))
        sql_session.cancel_session(session)
        return 1

    for p in paths:
        try:
            _one_move(session, p, dest_jobset, dest, workload)
        except:
            sql_session.cancel_session(session)
            return 1
    sql_session.close_session(session)
    return 0


def mv(sql_session, current_cwd, arguments, workload=None):
    """Rename a job or jobset.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working jobset (a path.Path object)
    arguments -- list of arguments given to the mv command (list
                 of jobs/jobsets)
    workload -- workload to use

    """
    try:
        optlist, args = getopt.getopt(arguments, "h", ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write(_("mv: ") + str(err) + "\n")
        return 1
    for o, a in optlist:
        if o in ("-h", "--help"):
            usage()
            return 0
    paths = list()
    num_args = len(args)
    dest_rename = None
    if args:
        for i in range(0, num_args):
            p = path.get_paths(sql_session, args[i], current_cwd,
                               workload=workload)
            # An argument is an unknow path (if it's not the last one)
            if not p:
                if i + 1 != num_args:
                    sys.stderr.write(_("mv: `%s': no such job or jobset\n") %
                                     args[i].decode('utf-8'))
                    return 1
                else:
                    dest_rename = args[i]
            paths.extend(p)
    else:
        sys.stderr.write(_("mv: missing job or jobset operand\n"))
        return 1
    if not paths:
        sys.stderr.write(_("mv: `%s': no such job or jobset\n")
                         % args[0].decode('utf-8'))
        return 1
    lp = len(paths)
    # Rename
    if lp == 1:
        if num_args < 2:
            sys.stderr.write(
                _("mv: not enough operands or unknown job/jobset\n"))
            return 1
        return rename(sql_session, current_cwd, paths[0], args[1], workload)
    # Move
    if dest_rename is not None:
        sys.stderr.write(_("mv: target `%s' does not exist\n") %
                         dest_rename.decode('utf-8'))
        return 1
    return move(sql_session, paths, workload)
