# Schedwi
# Copyright (C) 2011-2015 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to create new jobs."""

import sys
import getopt
import time

from sqlalchemy.orm.exc import NoResultFound

import path
import set_options
import set_options_w
from status_utils import WAITING
from tables.job_main import job_main
from tables.job_main_s import (job_main_s, get_next_id)
from tables.job_status import (job_status, TIME)


def usage(in_workload=False):
    """Print a usage message on STDOUT."""
    if in_workload is False:
        print _("""Usage: mk [OPTION]... JOB...
Create new jobs.

Options:
  --enable=True|False     enable (True) or disable (False) the job.
  --cluster=NAME          cluster on which the job/jobset must be running.
  --host=HOSTNAME[:PORT]  host on which the job must be running.  If not set,
                          same as the parent jobset.
  --calendar=CAL_NAME     calendar name. If not set, same as the parent jobset.
  --start=TIME            start time of the job (HH:MM).
  --command=COMMAND       command to run (if set to `-', no command will be
                          started and the job will complete immediately)
                          If not set, use the one defined by the parent jobset.
  --parameter=PARAMETER   a parameter to pass to the command.  This option can
                          be specified several times to define more than one
                          parameter.  If not set, use the parameters defined
                          by the parent jobset.
  --noparameter           no parameter at all for the command (the ones from
                          the parent jobset are ignored).
  --return=RETURN_CODE    a return code from the command equal or below this
                          value means success.  If not set, use the one set
                          by the parent jobset.
  --user=USERNAME         username under which the command must be run.
                          If not set, same as the parent jobset.
  --stdout=FILE           command output file (stored on the Schedwi agent).
                          If not set, same as the parent jobset.
  --stderr=FILE           command error file (stored on the Schedwi agent).
                          If not set, same as the parent jobset.
  --detach=True|False     start the job in detach mode (True) or not (False).
                          In detach mode, Schedwi will not wait for the
                          completion of the job but immediately set the status
                          of the job on whether the job has successfully
                          started or not. The detach mode can be useful to
                          start daemons, services or long running processes.
                          If not set, same as the parent jobset.
  --manual=True|False     ask confirmation (True) to operators to start the
                          job or not (False).  Default to False.
  --manual-cmd=COMMAND    for manual jobs, command/script to run to alert
                          operators that a job is waiting for a start
                          confirmation. The following environment variables
                          are set and can be used in the command/script:
                          SCHEDWI_TEMPLATE (the path to a text file that
                          contains the message to send to operators),
                          SCHEDWI_JOBPATH, SCHEDWI_JOBID, SCHEDWI_START_TIME,
                          SCHEDWI_START_TIME_EPOCH, SCHEDWI_TIME_LIMIT,
                          SCHEDWI_TIME_LIMIT_EPOCH, SCHEDWI_URN and
                          SCHEDWI_PASSWORD.
                          If not set, same as the parent jobset.
  --cgroup=NAME           Linux Control Group name to set for the job.  This
                          name is relative to the controllers mount point.
                          For instance, for the /cgroup/cpu/schedwi/test mount
                          point, NAME must be `schedwi/test'.
                          If not set, same as the parent jobset.
  --duration=MINUTES      maximum number of minutes for the job to
                          complete once running.  If not set, same as the
                          parent jobset.
  --retries=NUM           number of retries when the job fails.
                          If not set, same as the parent jobset.
  --retries-int=MINUTES   number of minutes between retries.
                          If not set, same as the parent jobset.
  --limit=MINUTES         number of minutes for the job to be started past its
                          start time (number of minutes or HH:MM)
                          If not set, same as the parent jobset.
  --user-env=True|False   load (True) or not (False) the user account
                          environment (~/.profile or ~/.bashrc for instance) on
                          the remote host before running the job command.
                          If not set, same as the parent jobset.
  --addenv=ENV_NAME       add an environment group to the job.  This
                          option can be specified several times.
  --enableenv             enable the use of environment groups (default)
  --disableenv            do not use any environment group for this job.
  --addfile=HOSTNAME[:PORT],True|False,FILE     add a file to check before
                          starting the job.  The value is composed of
                          three fields separated by a comma.  The first one is
                          the remote agent hostname on which to check the file.
                          The second specifies if the file must be present
                          (True) or not (False).  The last one is the file or
                          directory to check.
  --description=TEXT      a description.
  -h, --help              display this help.
        """)
    else:
        print _("""Usage: mk [OPTION]... JOB...
Create new jobs.

Options:
  --enable=True|False     enable (True) or disable (False) the job.
  --cluster=NAME          cluster on which the job/jobset must be running.
  --host=HOSTNAME[:PORT]  host on which the job must be running.  If not set,
                          same as the parent jobset.
  --start=TIME            start time of the job (HH:MM or `now').
  --command=COMMAND       command to run (if set to `-', no command will be
                          started and the job will complete immediately)
                          If not set, use the one defined by the parent jobset.
  --parameter=PARAMETER   a parameter to pass to the command.  This option can
                          be specified several times to define more than one
                          parameter.  If not set, use the parameters defined
                          by the parent jobset.
  --noparameter           no parameter at all for the command (the ones from
                          the parent jobset are ignored).
  --return=RETURN_CODE    a return code from the command equal or below this
                          value means success.  If not set, use the one set
                          by the parent jobset.
  --user=USERNAME         username under which the command must be run.
                          If not set, same as the parent jobset.
  --stdout=FILE           command output file (stored on the Schedwi agent).
                          If not set, same as the parent jobset.
  --stderr=FILE           command error file (stored on the Schedwi agent).
                          If not set, same as the parent jobset.
  --detach=True|False     start the job in detach mode (True) or not (False).
                          In detach mode, Schedwi will not wait for the
                          completion of the job but immediately set the status
                          of the job on whether the job has successfully
                          started or not. The detach mode can be useful to
                          start daemons, services or long running processes.
                          If not set, same as the parent jobset.
  --manual=True|False     ask confirmation (True) to operators to start the
                          job or not (False).  Default to False.
  --manual-cmd=COMMAND    for manual jobs, command/script to run to alert
                          operators that a job is waiting for a start
                          confirmation. The following environment variables
                          are set and can be used in the command/script:
                          SCHEDWI_TEMPLATE (the path to a text file that
                          contains the message to send to operators),
                          SCHEDWI_JOBPATH, SCHEDWI_JOBID, SCHEDWI_START_TIME,
                          SCHEDWI_START_TIME_EPOCH, SCHEDWI_TIME_LIMIT,
                          SCHEDWI_TIME_LIMIT_EPOCH, SCHEDWI_URN and
                          SCHEDWI_PASSWORD.
                          If not set, same as the parent jobset.
  --cgroup=NAME           Linux Control Group name to set for the job.  This
                          name is relative to the controllers mount point.
                          For instance, for the /cgroup/cpu/schedwi/test mount
                          point, NAME must be `schedwi/test'.
                          If not set, same as the parent jobset.
  --duration=MINUTES      maximum number of minutes for the job to
                          complete once running.  If not set, same as the
                          parent jobset.
  --retries=NUM           number of retries when the job fails.
                          If not set, same as the parent jobset.
  --retries-int=MINUTES   number of minutes between retries.
                          If not set, same as the parent jobset.
  --limit=MINUTES         number of minutes for the job to be started past its
                          start time (number of minutes or HH:MM)
                          If not set, same as the parent jobset.
  --user-env=True|False   load (True) or not (False) the user account
                          environment (~/.profile or ~/.bashrc for instance) on
                          the remote host before running the job command.
                          If not set, same as the parent jobset.
  --addenv=ENV_NAME       add an environment group to the job.  This
                          option can be specified several times.
  --enableenv             enable the use of environment groups (default)
  --disableenv            do not use any environment group for this job.
  --addfile=HOSTNAME[:PORT],True|False,FILE     add a file to check before
                          starting the job.  The value is composed of
                          three fields separated by a comma.  The first one is
                          the remote agent hostname on which to check the file.
                          The second specifies if the file must be present
                          (True) or not (False).  The last one is the file or
                          directory to check.
  --description=TEXT      a description.
  -h, --help              display this help.
        """)


def mk(sql_session, current_cwd, current_cwd_cal, arguments, workload=None):
    """Create new jobs.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working jobset (a path.Path object)
    arguments -- list of arguments given to the mk command (list
                 of jobs/jobsets)
    workload -- workload to use

    """
    # Parse the parameters
    try:
        if workload is None:
            optlist, args = set_options.parse_args(arguments)
        else:
            optlist, args = set_options_w.parse_args(arguments)
    except getopt.GetoptError, err:
        sys.stderr.write(_("mk: ") + str(err) + "\n")
        return 1
    for o, a in optlist:
        if o in ("-h", "--help"):
            usage(False if workload is None else True)
            return 0
    if not args:
        sys.stderr.write(_("mk: missing the job name to create\n"))
        return 1
    paths = list()
    for arg in args:
        basename, p = path.get_new_paths(sql_session, arg, current_cwd,
                                         workload)
        if not p:
            sys.stderr.write(_("mk: no such jobset in %s\n")
                             % arg.decode('utf-8'))
            return 1
        paths.append([basename, p])
    if not paths:
        sys.stderr.write(_("mk: no such jobset\n"))
        return 1
    session = sql_session.open_session()
    if workload is not None:
        new_id = get_next_id(session, workload)
    for basename, dirs in paths:
        for p in dirs:
            # Check that the job to create does not already exist
            try:
                full_path = path.Path(session, basename, p, workload=workload)
            except NoResultFound:
                pass
            else:
                sys.stderr.write(_("mk: %s already exists\n") % full_path)
                sql_session.cancel_session(session)
                return 1
            if workload is None:
                job = job_main(parent=p.id[-1],
                               name=basename,
                               type=1,
                               enabled=1,
                               has_env=1,
                               manual=0,
                               description='',
                               x=0, y=0,
                               cal_id=0,
                               start_time=-1)
                if set_options.set(session, current_cwd_cal,
                                   job, optlist) != 0:
                    sql_session.cancel_session(session)
                    return 1
            else:
                job = job_main_s(id=new_id,
                                 parent=p.id[-1],
                                 name=basename,
                                 type=1,
                                 enabled=0,
                                 has_env=1,
                                 manual=0,
                                 description='',
                                 x=0, y=0,
                                 cal_id=0,
                                 start_time=-1,
                                 workload_date=workload)
                new_id += 1
                job.job_status = job_status(status=WAITING,
                                            time_status_set=int(time.time()),
                                            retry_num=0,
                                            error_msg=_("Manually added"),
                                            duration=0,
                                            host_id=None,
                                            wait_reason=TIME,
                                            workload_date=workload)
                if set_options_w.set(session, job, optlist, workload) != 0:
                    sql_session.cancel_session(session)
                    return 1
                path_name = p.path
                print _("mk: %s created in disabled state: run \
`set --enable=True \"%s/%s\"' to enable") % (basename.decode('utf-8'),
                                             path_name if path_name != u'/'
                                             else '',
                                             basename.decode('utf-8'))
            session.add(job)
    sql_session.close_session(session)
    return 0
