# Schedwi
# Copyright (C) 2011-2015 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to print the parameters of jobs and jobsets."""

import sys
import getopt
import time

from sqlalchemy import desc

import path
import path_cal
import sql_hierarchy
import status_utils
import simple_queries_job
import environments_utils
from tables.job_main import job_main
from tables.job_command import job_command
from tables.job_arguments import job_arguments
from tables.job_loadenv import job_loadenv
from tables.job_detach import job_detach
from tables.job_manual_command import job_manual_command
from tables.job_file_out import job_file_out
from tables.job_file_err import job_file_err
from tables.job_control_group import job_control_group
from tables.job_max_duration import job_max_duration
from tables.job_retries import job_retries
from tables.job_retries_interval import job_retries_interval
from tables.job_start_limit import job_start_limit
from tables.job_success_return_code import job_success_return_code
from tables.job_username import job_username
from tables.job_host import job_host
from tables.hosts import hosts
from tables.job_cluster import job_cluster
from tables.clusters import clusters
from tables.constraint_file import constraint_file
from tables.job_environment import job_environment
from tables.link import link
from tables.job_main_s import job_main_s
from tables.job_host_s import job_host_s
from tables.job_cluster_s import job_cluster_s
from tables.job_command_s import job_command_s
from tables.job_arguments_s import job_arguments_s
from tables.job_success_return_code_s import job_success_return_code_s
from tables.job_username_s import job_username_s
from tables.job_file_out_s import job_file_out_s
from tables.job_file_err_s import job_file_err_s
from tables.job_control_group_s import job_control_group_s
from tables.job_max_duration_s import job_max_duration_s
from tables.job_retries_s import job_retries_s
from tables.job_retries_interval_s import job_retries_interval_s
from tables.job_start_limit_s import job_start_limit_s
from tables.job_loadenv_s import job_loadenv_s
from tables.job_detach_s import job_detach_s
from tables.job_manual_command_s import job_manual_command_s
from tables.job_environment_s import job_environment_s
from tables.constraint_file_s import constraint_file_s
from tables.link_s import link_s
from help import print_trim


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: cat [JOB|JOBSET]...
    Print job or jobset parameters.
    If JOB|JOBSET is not specified, print the parameters of the current jobset.
    In the output, if a parameter name is followed by the `*' character, it
    means that this parameter value is inherited from the parent jobset (see
    example below).

    Option:
      -h, --help  display this help.

    Example:
    cat /Prod/Report/Convert
                     Name   : Convert
                     Type   : Job
                  Enabled   : True
                   Manual   : False
                     Host   : flower.example.com (2006/SSL)
                 Calendar   : /Samples/Every Friday
               Start time   : 18h14
                  Command   : /usr/local/bin/convert.sh
               Parameters   : "/data/share" "pdf"
      Success return code * : 0
                 Username * : hive
              Output file * : /dev/null
               Error file * : /dev/null
                   Detach * : false
     Alert cmd for manual * : cat "$SCHEDWI_TEMPLATE" | mail -s "waiting" root@localhost
      Linux Control Group * : schedwi
             Max duration * : No limit
        Number of retries * : 0
         Interval retries * : 0h05
    Limit past start time * : 24h00
            Load user env * : False
    Use environment group   : True
        Environment group   : hive-env
        Environment group   : convert-env
          Constraint file   : /data/share/rpt.txt must exist on flower.example.com(2006)
          Constraint file   : /data/share/rpt.pdf must NOT exist on flower.example.com(2006)
                     Link   : /Prod/Report/Build report must be completed
              Description   : Convert the text report (rpt.txt) in PDF.

    In this example, the `*' character near the `Success return code' parameter
    tells that this parameter is not set in this job but is inherited from the
    parent jobset (`/Prod/Report' here).
    If the separator between the parameter name and its value is `=' rather
    than `:', it means that the parameter cannot be changed (for already
    planned jobs and jobsets).
    """))

_FIELD_WIDTH = 21


def _print_val(title, hist, fmt=None, val=None, writable=True):
    """Print a parameter value.

    Arguments:
    title -- parameter name
    hist -- whether the parameter is inherited or not
    fmt -- format string for the parameter value
    val -- value
    writable -- is the parameter can be changed.  If True, the separator
                between the name and the value is `:'.  Otherwise it's `='

    """
    sep = ':' if writable else '='
    if fmt is not None:
        if val is not None:
            print "%s %c %s" % (title.rjust(_FIELD_WIDTH),
                                ' ' if hist else '*', sep), fmt % val
        else:
            print "%s %c %s" % (title.rjust(_FIELD_WIDTH),
                                ' ' if hist else '*', sep), fmt
    else:
        print "%s %c %s" % (title.rjust(_FIELD_WIDTH), ' ' if hist else '*',
                            sep)


def cat(sql_session, current_cwd, arguments, workload=None):
    """Print job/jobset parameters.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working jobset (a path.Path object)
    arguments -- list of arguments given to the cat command (list
                 of jobs/jobsets)
    workload -- workload to use

    """
    try:
        optlist, args = getopt.getopt(arguments, "h", ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write(_("cat: ") + str(err) + "\n")
        return 1
    for o, a in optlist:
        if o in ("-h", "--help"):
            usage()
            return 0
    paths = list()
    if args:
        for arg in args:
            paths.extend(path.get_paths(sql_session, arg, current_cwd,
                                        workload=workload))
    else:
        paths.append(current_cwd)
    if not paths:
        sys.stderr.write(_("cat: no such job or jobset\n"))
        return 1
    first = True
    error = False
    for p in paths:
        if first is False:
            sys.stdout.write("\n")
        else:
            first = False
        session = sql_session.open_session()
        try:
            job = simple_queries_job.sql_get_job(session, p.id[-1], workload)
        except:
            sys.stderr.write(_("cat: no such job or jobset\n"))
            error = True
            continue
        _print_val(_("Name"), True, "%s", job.name if job.id > 1 else '/',
                   True if workload is None else job_main_s.rw_name)
        if workload is not None:
            _print_val(_("Current status"), True, "%s",
                       status_utils.get_status_string(session, job.id,
                                                      workload))
            d = status_utils.get_status_duration(session, job.id, workload)
            _print_val(_("Duration"), True, "%d (%dh%02dm%02ds)",
                       (d, d / 3600, d / 60 % 60, d % 60), False)
        _print_val(_("Type"), True,
                   "%s", _("Jobset") if not job.type else _("Job"),
                   True if workload is None else job_main_s.rw_type)
        _print_val(_("Enabled"), True, "%s", bool(job.enabled),
                   True if workload is None else job_main_s.rw_enabled)
        _print_val(_("Manual"), True, "%s", bool(job.manual),
                   True if workload is None else job_main_s.rw_manual)

        in_object, obj = sql_hierarchy.get_cluster_or_host(session, p,
                                                           workload)
        if obj is not None:
            if isinstance(obj, job_cluster) or isinstance(obj, job_cluster_s):
                c = session.query(clusters).filter_by(id=obj.cluster_id).one()
                _print_val(_("Cluster"),
                           in_object,
                           "%s",
                           c.name,
                           True if workload is None
                           else job_main_s.rw_job_cluster)
            elif isinstance(obj, job_host) or isinstance(obj, job_host_s):
                host = session.query(hosts).filter_by(id=obj.host_id).one()
                _print_val(
                    _("Host"),
                    in_object,
                    "%s (SSL)" if host.sslenable else "%s",
                    str(host),
                    True if workload is None else job_main_s.rw_job_host)

        if job.cal_id == 0:
            for i in range(len(p.id) - 2, -1, -1):
                if workload is None:
                    query = session.query(job_main.cal_id)
                else:
                    query = session.query(job_main_s.cal_id)
                    query = query.filter(job_main_s.workload_date == workload)
                cal_id, = query.filter_by(id=p.id[i]).one()
                if cal_id != 0:
                    _print_val(
                        _("Calendar"),
                        False,
                        "%s",
                        path_cal.PathCal(sql_session, id=cal_id,
                                         workload=workload),
                        True if workload is None else job_main_s.rw_cal_id)
                    break
            else:
                _print_val(_("Calendar"), False, "%s", _("Everyday"),
                           True if workload is None else job_main_s.rw_cal_id)
        else:
            _print_val(_("Calendar"), True, "%s",
                       path_cal.PathCal(sql_session, id=job.cal_id,
                                        workload=workload),
                       True if workload is None else job_main_s.rw_cal_id)

        if job.start_time < 0:
            foo, j = sql_hierarchy.get_start_time_id(session, p.id[:-1],
                                                     workload)
            if j is not None:
                start_time = j.start_time
                if workload is None:
                    _print_val(_("Start time"), False, "%02dh%02d",
                               (start_time / 100, start_time % 100), True)
                else:
                    _print_val(_("Start time"), False, "%s",
                               time.strftime('%c', time.localtime(start_time)),
                               job_main_s.rw_start_time)
            else:
                _print_val(
                    _("Start time"), False, "%02dh%02d", (0, 0),
                    True if workload is None else job_main_s.rw_start_time)
        else:
            if workload is None:
                _print_val(_("Start time"), True, "%02dh%02d",
                           (job.start_time / 100, job.start_time % 100), True)
            else:
                _print_val(_("Start time"), True, "%s",
                           time.strftime('%c', time.localtime(job.start_time)),
                           job_main_s.rw_start_time)
        sql_session.close_session(session)

        in_object, obj = sql_hierarchy.get(
            sql_session,
            p,
            job_command if workload is None else job_command_s,
            workload)
        if obj is not None:
            _print_val(_("Command"), in_object, "%s", obj.command,
                       True if workload is None else job_main_s.rw_job_command)
        else:
            _print_val(_("Command"), True, "%s", _("None"),
                       True if workload is None else job_main_s.rw_job_command)

        in_object, lst = sql_hierarchy.get_list(
            sql_session,
            p,
            job_arguments if workload is None else job_arguments_s,
            job_arguments.position if workload is None
            else job_arguments_s.position,
            workload)

        if lst is not None:
            params = list()
            for param in lst:
                if param.position == -1:
                    _print_val(_("Parameters"), in_object, None, None,
                               True if workload is None
                               else job_main_s.rw_job_arguments)
                    break
                params.append('"' + param.argument + '"')
            else:
                _print_val(
                    _("Parameters"),
                    in_object,
                    "%s",
                    ' '.join(params),
                    True if workload is None else job_main_s.rw_job_arguments)

        in_object, obj = sql_hierarchy.get(
            sql_session,
            p,
            job_success_return_code if workload is None
            else job_success_return_code_s,
            workload)
        if obj is not None:
            _print_val(_("Success return code"), in_object, "%s",
                       obj.success_ret,
                       True if workload is None
                       else job_main_s.rw_job_success_return_code)

        in_object, obj = sql_hierarchy.get(
            sql_session,
            p,
            job_username if workload is None else job_username_s, workload)
        if obj is not None:
            _print_val(
                _("Username"),
                in_object,
                "%s",
                obj.username,
                True if workload is None else job_main_s.rw_job_username)

        in_object, obj = sql_hierarchy.get(
            sql_session,
            p,
            job_file_out if workload is None else job_file_out_s,
            workload)
        if obj is not None:
            _print_val(
                _("Output file"),
                in_object,
                "%s",
                obj.file_out,
                True if workload is None else job_main_s.rw_job_file_out)

        in_object, obj = sql_hierarchy.get(
            sql_session,
            p,
            job_file_err if workload is None else job_file_err_s,
            workload)
        if obj is not None:
            _print_val(
                _("Error file"),
                in_object,
                "%s",
                obj.file_err,
                True if workload is None else job_main_s.rw_job_file_err)

        in_object, obj = sql_hierarchy.get(
            sql_session,
            p,
            job_detach if workload is None else job_detach_s,
            workload)
        if obj is not None:
            _print_val(_("Detach"), in_object, "%s", bool(obj.detach),
                       True if workload is None else job_main_s.rw_job_detach)

        in_object, obj = sql_hierarchy.get(
            sql_session,
            p,
            job_manual_command if workload is None else job_manual_command_s,
            workload)
        if obj is not None:
            _print_val(
                _("Alert cmd for manual"),
                in_object,
                "%s",
                obj.manual_command,
                True if workload is None else job_main_s.rw_job_manual_command)
        else:
            _print_val(_("Alert cmd for manual"), True, "%s", _("None"),
                       True if workload is None
                       else job_main_s.rw_job_manual_command)

        in_object, obj = sql_hierarchy.get(
            sql_session,
            p,
            job_control_group if workload is None else job_control_group_s,
            workload)
        if obj is not None:
            _print_val(
                _("Linux Control Group"),
                in_object,
                "%s",
                obj.control_group,
                True if workload is None else job_main_s.rw_job_control_group)

        in_object, obj = sql_hierarchy.get(
            sql_session,
            p,
            job_max_duration if workload is None else job_max_duration_s,
            workload)
        if obj is not None:
            if obj.max_duration == 0:
                _print_val(
                    _("Max duration"),
                    in_object,
                    "%s",
                    _("No limit"),
                    True if workload is None
                    else job_main_s.rw_job_max_duration)
            else:
                _print_val(_("Max duration"), in_object, "%dh%02d",
                           (obj.max_duration / 60, obj.max_duration % 60),
                           True if workload is None
                           else job_main_s.rw_job_max_duration)

        in_object, obj = sql_hierarchy.get(
            sql_session,
            p,
            job_retries if workload is None else job_retries_s,
            workload)
        if obj is not None:
            _print_val(
                _("Number of retries"),
                in_object,
                "%s",
                obj.retries,
                True if workload is None else job_main_s.rw_job_retries)

        in_object, obj = sql_hierarchy.get(
            sql_session,
            p,
            job_retries_interval if workload is None
            else job_retries_interval_s,
            workload)
        if obj is not None:
            _print_val(_("Interval retries"), in_object, "%dh%02d",
                       (obj.retries_interval / 60, obj.retries_interval % 60),
                       True if workload is None
                       else job_main_s.rw_job_retries_interval)

        in_object, obj = sql_hierarchy.get(
            sql_session,
            p,
            job_start_limit if workload is None else job_start_limit_s,
            workload)
        if obj is not None:
            if obj.start_limit <= 0:
                _print_val(_("Limit past start time"), in_object, "%s",
                           _("No limit"),
                           True if workload is None
                           else job_main_s.rw_job_start_limit)
            else:
                _print_val(_("Limit past start time"), in_object, "%dh%02d",
                           (obj.start_limit / 60, obj.start_limit % 60),
                           True if workload is None
                           else job_main_s.rw_job_start_limit)

        in_object, obj = sql_hierarchy.get(
            sql_session,
            p,
            job_loadenv if workload is None else job_loadenv_s,
            workload)
        if obj is not None:
            _print_val(_("Load user env"), in_object, "%s", bool(obj.loadenv),
                       True if workload is None else job_main_s.rw_job_loadenv)

        _print_val(_("Use environment group"), True, "%s", bool(job.has_env),
                   True if workload is None else job_main_s.rw_has_env)

        session = sql_session.open_session()

        env_list = list()
        for i in range(len(p.id) - 2, -1, -1):
            j = simple_queries_job.sql_get_job(session, p.id[i], workload)
            if not j.has_env:
                break
            if workload is None:
                query = session.query(job_environment)
                query = query.filter_by(job_id=p.id[i])
                query = query.order_by(desc(job_environment.position))
            else:
                query = session.query(job_environment_s)
                query = query.filter_by(job_id=p.id[i])
                query = query.filter(job_environment_s.workload_date ==
                                     workload)
                query = query.order_by(desc(job_environment_s.position))
            for je in query.all():
                env_list.insert(0, je.env_id)
        for env_id in env_list:
            e = environments_utils.get_env(session, env_id, workload)
            _print_val(_("Inherited env group"), True, "%s", e.name)

        if workload is None:
            query = session.query(job_environment).filter_by(job_id=p.id[-1])
            query = query.order_by(job_environment.position)
        else:
            query = session.query(job_environment_s).filter_by(job_id=p.id[-1])
            query = query.filter(job_environment_s.workload_date == workload)
            query = query.order_by(job_environment_s.position)
        for je in query.all():
            e = environments_utils.get_env(session, je.env_id, workload)
            _print_val(_("Environment group"), True, "%s", e.name)

        if workload is None:
            query = session.query(constraint_file, hosts)
            query = query.filter_by(job_id=p.id[-1])
            query = query.filter(constraint_file.host_id == hosts.id)
            query = query.order_by(desc(constraint_file.exist))
        else:
            query = session.query(constraint_file_s, hosts)
            query = query.filter(constraint_file_s.workload_date == workload)
            query = query.filter_by(job_id=p.id[-1])
            query = query.filter(constraint_file_s.host_id == hosts.id)
            query = query.order_by(desc(constraint_file_s.exist))

        for f, h in query.all():
            _print_val(
                _("Constraint file"),
                True,
                _("%s must %sexist on %s"),
                (f.filename, '' if f.exist else _('NOT '), str(h)),
                True if workload is None else job_main_s.rw_constraint_file)

        if workload is None:
            query = session.query(link)
        else:
            query = session.query(link_s)
            query = query.filter(link_s.workload_date == workload)
        for lnk in query.filter_by(job_id_source=p.id[-1]).all():
            destination = path.id2path(sql_session, lnk.job_id_destination,
                                       workload)
            _print_val(_("Link"), True, _("%s must be %s"),
                       (destination,
                        status_utils.status2string(lnk.required_status)),
                       True if workload is None else job_main_s.rw_links)

        sql_session.close_session(session)
        _print_val(_("Description"), True, "%s", job.description,
                   True if workload is None else job_main_s.rw_description)
    return 0 if not error else 1
