# Schedwi
# Copyright (C) 2011-2015 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to list hosts."""

import sys
import getopt

from tables.hosts import hosts
import host_utils
from help import print_trim


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: ls [OPTION]... [HOST]...
    List information about the HOSTs (all the hosts are listed by default).
    Sort entries alphabetically.  HOST must have the following format:
       HOSTNAME[:PORT]
    For IPv6, the address must be enclosed between square brackets (ie.
    [fe80::210:a2ff:fa1d:aabb:ccdd]:2006)

    Options:
      -l         use a long listing format according to the following schema:
                  - or s  whether the connection should use SSL for encryption
                  - or e  if any environment group is associated whith the host
                  PORT    TCP port name or number
                  NAME    host name
      -h, --help display this help

    Example:
    > ls -l
    -e 2006      apple.example.com
    s- 2006      cherry.example.com
    s- 2006      flower.example.com
    -- 2006      localhost
    se 2007      plum.example.com
    """))


def _print_host(host, long_listing):
    """Print a host name."""
    if not long_listing:
        print str(host)
        return
    print "%(ssl)s%(env)s %(port)-16s %(name)s" % \
          {"ssl": 's' if host.sslenable else '-',
           "env": 'e' if host.host_environment else '-',
           "port": host.portnum,
           "name": host.hostname}


def ls(sql_session, arguments):
    """List hosts.

    Arguments:
    sql_session -- SQLAlchemy session
    arguments -- list of arguments given to the ls command (list of hosts)

    """
    try:
        optlist, args = getopt.getopt(arguments, 'lh', ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write(_("ls: ") + str(err) + "\n")
        return 1
    long_listing = False
    for o, a in optlist:
        if o == "-l":
            long_listing = True
        elif o in ("-h", "--help"):
            usage()
            return 0
    session = sql_session.open_session()
    if args:
        for a in args:
            host_list = host_utils.name2host_list(session, a)
            if not host_list:
                sys.stderr.write(_("ls: `%s': no such host\n")
                                 % a.decode('utf-8'))
            else:
                for h in host_list:
                    _print_host(h, long_listing)
    else:
        for h in session.query(hosts).order_by(hosts.hostname):
            _print_host(h, long_listing)
    sql_session.close_session(session)
    return 0
