# Schedwi
# Copyright (C) 2011-2015 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to set environment group parameters."""

import getopt
import sys
import re

from tables.environment_var import environment_var
from tables.environment_var_s import environment_var_s


def usage_options(with_name=True):
    """Print the options usage on STDOUT.

    Argument:
    with_name -- wether the --name option should be listed or not

    """
    print _("\nOptions:")
    if with_name:
        print _("  --name=NAME             environment group name.")
    print _(
        """  --append=NAME=VALUE     add an environment variable at the end of
                          the list. This option can be specified more than one
                          time to add several environment variables in one go.
  --insert=NAME=VALUE     insert an environment variable at the position
                          specified by the --position parameter or in first
                          position if --position is not specified.  Cannot be
                          used if --delete is specified.
  --position=POS          when used with --insert specified the position to
                          which insert the environment variable.
  --delete=POS            delete the environment variable at the specified
                          position.  Cannot be used if --insert is specified.
  --description=TEXT      a description.
  -h, --help              display this help.
    """)


def usage():
    """Print a usage message on STDOUT."""
    print _("""Usage: set [OPTION]... ENV
Change environment group parameters.""")
    usage_options()


def parse_args(argv):
    """Parse the argument list.

    Argument:
    argv --  argument list to parse

    Exception:
    getopt.getopterror -- option error (unknow option or missing parameter)

    """
    optlist, args = getopt.getopt(argv, "n:d:a:i:p:h",
                                  [
                                      "name=",
                                      "delete=",
                                      "append=",
                                      "insert=",
                                      "position=",
                                      "description=",
                                      "help"
                                  ])
    return (optlist, args)


def set(session, env, opts, workload=None):
    """Change environment group parameters.

    Arguments:
    session -- SQLAlchemy session
    env -- environment group object to change
    opts -- option list (from getopt.getopt() - see parse_args())
    workload -- workload to use

    """
    position_insert = 1
    position_delete = None
    insert_var = None
    append_vars = list()
    max_pos = len(env.environment_var)
    for o, a in opts:
        if o in ("-n", "--name"):
            env.name = a.decode('utf-8')

        elif o == "--description":
            env.description = a.decode('utf-8')

        elif o in ("-h", "--help"):
            usage()

        elif o in ("-p", "--position"):
            try:
                position_insert = int(a)
            except ValueError:
                sys.stderr.write(_("Invalid value `%s' (--position)\n")
                                 % a.decode('utf-8'))
                return 1
            if position_insert <= 0 or position_insert > max_pos + 1:
                sys.stderr.write(_("Out of bound value `%s' (--position must \
be between 1 and %d)\n") % (a, max_pos + 1))
                return 1

        elif o in ("-d", "--delete"):
            if insert_var is not None:
                sys.stderr.write(_("Cannot use --delete with --insert\n"))
                return 1
            try:
                position_delete = int(a)
            except ValueError:
                sys.stderr.write(_("Invalid value `%s' (--delete)\n")
                                 % a.decode('utf-8'))
                return 1
            if position_delete <= 0 or position_delete > max_pos:
                sys.stderr.write(_("Out of bound value `%s' (--delete must \
be between 1 and %d)\n") % (a, max_pos))
                return 1

        elif o in ("-i", "--insert"):
            if position_delete is not None:
                sys.stderr.write(_("Cannot use --insert with --delete\n"))
                return 1
            if insert_var is not None:
                sys.stderr.write(_("Only one environment variable can be \
inserted in one go (--insert can only be specified once)\n"))
                return 1
            try:
                a.index('=')
            except ValueError:
                sys.stderr.write(_("Invalid environment variable `%s' \
(--insert must be NAME=VALUE)\n")
                                 % a.decode('utf-8'))
                return 1
            insert_var = a

        elif o in ("-a", "--append"):
            try:
                a.index('=')
            except ValueError:
                sys.stderr.write(_("Invalid environment variable `%s' \
(--append must be NAME=VALUE)\n") % a.decode('utf-8'))
                return 1
            append_vars.append(a)

    if position_delete is not None:
        env.environment_var.pop(position_delete - 1)
    if insert_var is not None:
        name, value = insert_var.split('=', 1)
        name = name.strip()
        if not name or not re.match('^[A-Za-z_][A-Za-z_0-9]*$', name):
            sys.stderr.write(_("Invalid environment variable name `%s' \
(cannot be empty and must contain only letters, numbers, and underscores, and \
begin with a letter or underscore)\n") % name)
            return 1
        new_list = list()
        is_added = False
        pos = 1
        for var in env.environment_var:
            if pos == position_insert:
                if workload is None:
                    new_list.append(environment_var(pos, name, value))
                else:
                    new_list.append(environment_var_s(pos, name, value,
                                                      workload))
                pos += 1
                is_added = True
            if workload is None:
                new_list.append(environment_var(pos, var.env_key,
                                                var.env_value))
            else:
                new_list.append(environment_var_s(pos, var.env_key,
                                                  var.env_value, workload))
            session.delete(var)
            pos += 1
        if not is_added:
            if workload is None:
                new_list.append(environment_var(pos, name, value))
            else:
                new_list.append(environment_var_s(pos, name, value, workload))
        env.environment_var = new_list
    if append_vars:
        if env.environment_var:
            pos = 1 + env.environment_var[-1].position
        else:
            pos = 1
            env.environment_var = list()
        for env_var in append_vars:
            name, value = env_var.split('=', 1)
            name = name.strip()
            if not name or not re.match('^[A-Za-z_][A-Za-z_0-9]*$', name):
                sys.stderr.write(_("Invalid environment variable name `%s' \
(cannot be empty and must contain only letters, numbers, and underscores, and \
begin with a letter or underscore)\n") % name)
                continue
            if workload is None:
                env.environment_var.append(environment_var(pos, name, value))
            else:
                env.environment_var.append(environment_var_s(pos, name, value,
                                                             workload))
            pos += 1
    return 0
