# Schedwi
# Copyright (C) 2011-2015 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to list jobs and jobsets that are using a calendar."""

import sys
import getopt

import simple_queries_cal
import path_cal
import path
from tables.job_main import job_main
from tables.job_main_s import job_main_s
from help import print_trim


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: whatuses CAL|DIR...
        List the jobs and jobsets that use CAL or DIR.  For DIR, recursively
        list the jobs and jobsets.

        Option:
          -h, --help  display this help.
    """))


def is_used(session, cal, workload=None):
    """Return whether a calendar is used by a job or jobset."""
    if workload is None:
        q = session.query(job_main)
    else:
        q = session.query(job_main_s)
        q = q.filter(job_main_s.workload_date == workload)
    if q.filter_by(cal_id=cal.id).count():
        return True
    return False

_FIELD_WIDTH = 10
_previous_name = None


def _print_val(foo, cal_name, title, val):
    """Print a parameter value.

    Arguments:
    cal_name -- calendar or directory name
    title -- parameter name
    val -- value

    """
    global _previous_name
    if cal_name != _previous_name:
        _previous_name = cal_name
        print "\n%s:" % cal_name
    print "%s :" % title.rjust(_FIELD_WIDTH), unicode(val).encode('utf-8')


def print_whatuses(session, cal, p, workload,
                   print_function=_print_val,
                   user_data=None):
    """Print a list of jobs/jobsets that are using the given calendar.

    Arguments:
    session -- SQLAlchemy session
    cal -- calendar object
    p -- calendar path
    only_if_used -- whether to print the calendar name if it is not in used
                    by any job or jobset.
    workload -- workload to use

    """
    cal_name = unicode(p).encode('utf-8')
    if workload is None:
        q = session.query(job_main).filter(job_main.cal_id == cal.id)
        q = q.order_by(job_main.parent)
    else:
        q = session.query(job_main_s).filter(job_main_s.cal_id == cal.id)
        q = q.filter(job_main_s.workload_date == workload)
        q = q.order_by(job_main_s.parent)
    for job in q:
        print_function(user_data,
                       cal_name,
                       _("Job") if job.type else _("Jobset"),
                       path.id2path(session, job.id, workload))


def print_whatuses_dir(session, cal, p, workload,
                       print_function=_print_val,
                       user_data=None):
    """Recursively print jobs and jobsets that use a calendar directory.

    Arguments:
    session -- SQLAlchemy session
    cal -- calendar directory object
    p -- calendar directory path
    workload -- workload to use

    """
    for c in simple_queries_cal.sql_get_cal_children(session, cal.id, False,
                                                     workload, False):
        if c.entry_type == 0:
            print_whatuses(session,
                           c,
                           path_cal.PathCal(session, id=c.id,
                                            workload=workload),
                           workload,
                           print_function,
                           user_data)
        else:
            print_whatuses_dir(session,
                               c,
                               path_cal.PathCal(session, id=c.id,
                                                workload=workload),
                               workload,
                               print_function,
                               user_data)


def whatuses(sql_session, current_cwd, arguments, workload=None):
    """Show what is referencing a calendar.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working directory (a path_cal.PathCal object)
    arguments -- list of arguments given to the whatuses command (list
                 of calendars)
    workload -- workload to use

    """
    global _previous_name
    _previous_name = None
    try:
        optlist, args = getopt.getopt(arguments, "h", ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write(_("whatuses: ") + str(err) + "\n")
        return 1
    for o, a in optlist:
        if o in ("-h", "--help"):
            usage()
            return 0
    paths = list()
    if args:
        for arg in args:
            p = path_cal.get_paths(sql_session, arg, current_cwd,
                                   workload=workload)
            if not p:
                sys.stderr.write(
                    _("whatuses: `%s': no such calendar or directory\n")
                    % arg.decode('utf-8'))
            else:
                paths.extend(p)
    else:
        sys.stderr.write(_("whatuses: missing operand\n"))
        return 1
    if not paths:
        return 1
    session = sql_session.open_session()
    error = False
    for p in paths:
        try:
            cal = simple_queries_cal.sql_get_cal(session, p.id[-1], workload)
        except:
            sys.stderr.write(_("whatuses: no such calendar or directory\n"))
            error = True
            continue
        if cal.entry_type == 0:
            print_whatuses(session, cal, p, workload)
        else:
            print_whatuses_dir(session, cal, p, workload)
    sql_session.close_session(session)
    return 0 if not error else 1
