/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* signal_utils.c -- Several signal functions */

#include <schedwi.h>
#if HAVE_SIGNAL_H
#include <signal.h>
#endif
#include <signal_utils.h>


/* Specifies if signals are masked or not */
static char signal_flag = 0;
/* Previous signal mask */
static sigset_t mask_out;


/*
 * Mask all signals (except SIGKILL, SIGSTOP, SIGCONT and SIGALRM)
 *
 * Return:
 *   0 --> OK
 *  -1 --> Error (errno is set)
 */
int
signal_mask_all ()
{
	sigset_t mask;

	/* If the signals are not already masked... */
	if (signal_flag == 0) {
		/*
		 * Block all signals except SIGKILL, SIGSTOP, SIGCONT and
		 * SIGALRM
		 */
		if(	   sigfillset (&mask) != 0
			|| sigdelset (&mask, SIGKILL) != 0
			|| sigdelset (&mask, SIGSTOP) != 0
			|| sigdelset (&mask, SIGCONT) != 0
			|| sigdelset (&mask, SIGALRM) != 0
			|| sigprocmask (SIG_SETMASK, &mask, &mask_out) != 0 )
		{
			return -1;
		}
		signal_flag = 1;
	}
	return 0 ;
}


/*
 * Unmask signals previously masked by signal_mask_all()
 *
 * Return:
 *   0 --> OK
 *  -1 --> Error (errno is set)
 */
int
signal_unmask ()
{
	/* If  signals are blocked... */
	if (signal_flag == 1) {
		/* Restore the old mask */
		if (sigprocmask (SIG_SETMASK, &mask_out, 0) != 0) {
			return -1;
		}
		signal_flag=0;
	}
	return 0;
}


/*
 * Unmask all signals
 *
 * Return:
 *   0 --> OK
 *  -1 --> Error (errno is set)
 */
int
signal_unmask_all ()
{
	sigset_t mask;

	if(	   sigemptyset (&mask) != 0
		|| sigprocmask (SIG_SETMASK, &mask, 0) != 0)
	{
		return -1;
	}
	signal_flag=0;
	return 0;
}

/*------------------------======= End Of File =======------------------------*/
