/* Schedwi
   Copyright (C) 2007-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* schedwiclnt.c -- Schedwi agent */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif

#if HAVE_SIGNAL_H
#include <signal.h>
#endif

#if HAVE_GETOPT_H
#include <getopt.h>
#endif

#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <utils.h>
#include <conf.h>
#include <conf_clnt.h>
#include <signal_utils.h>
#include <lwc_log.h>
#include <child_mgnt.h>
#include <send_process.h>
#include <lib_functions.h>
#include <net_utils_init.h>
#include <net_utils_glob.h>
#include <net_utils_ssl.h>
#include <net_parse.h>
#include <net_module.h>
#include <net_module_clnt.h>
#include <xmem.h>
#include <cert_client.h>

#if _GNU_SOURCE && !HAVE_CLEARENV
extern char **environ;
#endif

static const char *configuration_file = NULL;
static char **save_argv, **save_env;
#if HAVE_CHDIR && HAVE_GETCWD
#define MAX_SIZE_CWD 500
static char save_cwd[MAX_SIZE_CWD];
#endif


/*
 * Print a help message to stderr
 */
void
help (const char * prog_name)
{
	int i;
	char * msg[] = {
N_("The Schedwi agent."),
"",
#if HAVE_GETOPT_LONG
N_("  -c, --config=FILE  use the configuration file FILE rather than"),
N_("                     the default one"),
N_("  -t, --test         Run syntax tests for the configuration file. The"),
N_("                     program immediately exits after these syntax"),
N_("                     parsing tests with either a return code of 0 (syntax"),
N_("                     OK) or return code not equal to 0 (syntax error)"),
N_("  -f, --foreground   keep the program in foreground"),
N_("  -h, --help         display this help and exit"),
N_("  -V, --version      print version information and exit"),
#else
N_("  -c FILE            use the configuration file FILE rather than"),
N_("                     the default one"),
N_("  -t                 Run syntax tests for the configuration file. The"),
N_("                     program immediately exits after these syntax"),
N_("                     parsing tests with either a return code of 0 (syntax"),
N_("                     OK) or return code not equal to 0 (syntax error)"),
N_("  -f                 keep the program in foreground"),
N_("  -h                 display this help and exit"),
N_("  -V                 print version information and exit"),
#endif
"",
"~",
N_("Exit status is 1 on error (0 means no error)."),
NULL
	};

#if HAVE_ASSERT_H
	assert (prog_name != NULL && configuration_file != NULL);
#endif

	fprintf (stderr, _("Usage: %s [OPTION]...\n"), prog_name);
						
	for (i = 0; msg[i] != NULL; i++) {
		if (msg[i][0] == '~') {
			fputs (	_("The default configuration file is "),
				stderr);
			fputs (configuration_file, stderr);
		}
		else {
			fputs ((msg[i][0] == '\0')? msg[i]: _(msg[i]), stderr);
		}
		fputc ('\n', stderr);
	}

	fputc ('\n', stderr);
	fputs (_("Report bugs to "), stderr);
	fputs (PACKAGE_BUGREPORT, stderr);
	fputc ('\n', stderr);
}


/*
 * Signal handler for SIGHUP
 */
static void
sighup_action (int sig)
{
	/*
	 * Before anything else be sure that the configuration file
	 * can be read
	 */
#if HAVE_ACCESS && defined R_OK
	if (access (configuration_file, R_OK) != 0) {
		lwc_writeLog (	LOG_ERR, _("%s: %s"),
				configuration_file, strerror (errno));
		return;
	}
#endif

	send_process_stop ();
	lwc_closeLog ();
	close_pid_file ();
	conf_destroy_clnt ();
	net_free ();
	sanitise (-1);
#if HAVE_CHDIR && HAVE_GETCWD
	if (save_cwd[0] != '\0') {
		chdir (save_cwd);
	}
#endif
	execve (save_argv[0], save_argv, save_env);
	perror ("execve");
	exit (EXIT_FAILURE);
}


/*
 * Signal handler for SIGTERM
 */
static void
sigterm_action (int sig)
{
	int i;


	lwc_writeLog (LOG_INFO, _("schedwiclnt exiting"));
	send_process_stop ();
	lwc_closeLog ();
	close_pid_file ();
	conf_destroy_clnt ();
	net_free ();
	for (i = 0; save_env[i] != NULL; i++) {
		free (save_env[i]);
	}
	free (save_env);
	free (save_argv);
	exit (EXIT_SUCCESS);
}


/*
 * Signal handler for SIGCHILD
 */
static void
sigchild_action (int sig)
{
#if HAVE_PID_T
	pid_t child;
#else
	int child;
#endif

	signal_mask_all ();
	while ((child = waitpid (-1, NULL, WNOHANG)) > 0) {
		delete_child (child, NULL);
	}
	send_process_wakeup ();
	signal_unmask ();
}


/*
 * Main function
 *
 * The exit code is 1 in case of error or 0 if killed (SIGTERM)
 */
int
main (int argc, char **argv)
{
	net_id sock, s;
	char *buff;
	size_t buff_len;
	int ret, i;
	struct sigaction sa;
	const char *log_file, *date_format, *registrar_port, *pid_file;
	const char *iface, *agent_port, *address_family, *server, *allow_from;
	long int facility;
	const char *client_crt, *client_key, *server_crt;
	char quick_random, test, enable_ssl, foreground;
	const char *prog_name;
#if HAVE_GETOPT_LONG
	int option_index;
	struct option long_options[] =
	{
		{"help",       0, 0, 'h'},
		{"version",    0, 0, 'V'},
		{"test",       0, 0, 't'},
		{"foreground", 0, 0, 'f'},
		{"config",     1, 0, 'c'},
		{0, 0, 0, 0}
	};
#endif


#if HAVE_SETLOCALE
	setlocale (LC_ALL, "");
#endif
#if HAVE_BINDTEXTDOMAIN
	bindtextdomain (PACKAGE, LOCALEDIR);
#endif
#if HAVE_TEXTDOMAIN
	textdomain (PACKAGE);
#endif

	/* Set default values for options */
	test = foreground = 0;
	configuration_file = SCHEDWI_DEFAULT_CONFFILE_CLNT;

	prog_name = base_name (argv[0]);

	/* Parse options */
	while (1) {
#if HAVE_GETOPT_LONG
		option_index = 0;
		ret = getopt_long (argc, argv, "hVtfc:",
					long_options, &option_index);
#else
		ret = getopt (argc, argv, "hVtfc:");
#endif

		if (ret == -1) {
			break;
		}

		switch (ret) {
			case 'h':
				help (prog_name);
				return EXIT_SUCCESS;
			case 'V':
				version (prog_name);
				return EXIT_SUCCESS;
			case 't':
				test = 1;
				break;
			case 'f':
				foreground = 1;
				break;
			case 'c':
				configuration_file = optarg;
				break;
			default:
				help (prog_name);
				return EXIT_FAILURE;
		}
	}

	if (test == 1) {
		printf (_("Configuration file: %s\n"), configuration_file);
	}

	/*
	 * Read the configuration file
	 */
	ret = conf_init_clnt (configuration_file);
	if (ret == -2) {
			perror (configuration_file);
	}
	if (ret != 0) {
		return EXIT_FAILURE;
	}


	/*
	 * Load the network modules
	 */
	module_load_clnt ();


	/*
	 * Initialize the network (before detaching the program from its
	 * terminal as a password may be asked for the SSL private key)
	 */
	ret = conf_get_param_string ("SSLCertificateFile", &client_crt);
	ret += conf_get_param_string ("SSLCertificateKeyFile", &client_key);
	ret += conf_get_param_string ("SSLCACertificateFile", &server_crt);
	ret += conf_get_param_bool ("SSLQuickRandom", &quick_random);
	ret += conf_get_param_bool ("SSLEngine", &enable_ssl);
	ret += conf_get_param_string ("SERVER", &server);
	ret += conf_get_param_string ("AGENT_PORT", &agent_port);
	ret += conf_get_param_string ("REGISTRAR_PORT", &registrar_port);
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif


	/* It was only a test of the configuration file */
	if (test == 1) {
		fputs (_("Fine."), stdout);
		fputc ('\n', stdout);
		return EXIT_SUCCESS;
	}


	/* Copy argv and environ to use them to exec the process on SIGHUP */
	save_argv = (char **) xmalloc (sizeof (char *) * (argc + 1));
	for (i = 0; i < argc; i++) {
		save_argv[i] = argv[i];
	}
	save_argv[i] = NULL;

	for (i = 0; environ[i] != NULL; i++);
	save_env = (char **) xmalloc (sizeof (char *) * (i + 1));
	for (i = 0; environ[i] != NULL; i++) {
		save_env[i] = (char *) xmalloc (strlen (environ[i]) + 1);
		strcpy (save_env[i], environ[i]);
	}
	save_env[i] = NULL;

#if HAVE_CHDIR && HAVE_GETCWD
	if (getcwd (save_cwd, MAX_SIZE_CWD) == NULL) {
		save_cwd[0] = '\0';
	}
#endif


	/*
	 * Go background
	 */
	if (foreground == 0 && go_daemon () != 0) {
		perror ("fork");
		conf_destroy_clnt ();
		destroy_modules ();
		for (i = 0; save_env[i] != NULL; i++) {
			free (save_env[i]);
		}
		free (save_env);
		free (save_argv);
		return EXIT_FAILURE;
	}


	/*
	 * For sysV daemons, open the log file and write the PID file.
	 * For systemd service (foreground = 1), systemd will already have done
	 * all that (log messages are sent to stderr)
	 */
	if (foreground == 0) {

		/*
		 * Open the log file
		 */
		log_file = date_format = NULL;
		facility = -1;
		ret = conf_get_param_string ("LOG_FILE", &log_file);
		ret += conf_get_param_syslog_facility ("SYSLOG", &facility);
		ret += conf_get_param_string ("DATE_FORMAT", &date_format);
#if HAVE_ASSERT_H
		assert (ret == 0);
#endif

		lwc_newLog (    PACKAGE_NAME,
				(facility != -1) ? 1 : 0,
				(facility != -1) ? (int)facility : 0,
				log_file,
				date_format);
		lwc_writeLog (LOG_INFO, _("Starting"));


		/*
		 * Create the PID file and check that an other process is not
		 * already running
		 */
		ret = conf_get_param_string ("PID_FILE", &pid_file);
#if HAVE_ASSERT_H
		assert (ret == 0);
#endif
		ret = write_pid_file (pid_file);
		switch (ret) {
			case -1:
				lwc_writeLog (	LOG_CRIT,
					_("Cannot create the PID file %s: %s"),
						pid_file,
						strerror (errno));
				break;
			
			case 1:
				lwc_writeLog (LOG_CRIT,
				_("An other daemon is already running"));
				break;
		}
		if (ret != 0) {
			lwc_closeLog ();
			conf_destroy_clnt ();
			destroy_modules ();
			for (i = 0; save_env[i] != NULL; i++) {
				free (save_env[i]);
			}
			free (save_env);
			free (save_argv);
			return EXIT_FAILURE;
		}
	}


#if HAVE_LIBGNUTLS
	/*
	 * Initialize the GnuTLS layer
	 */
	if (enable_ssl != 0 && net_init_gnutls (quick_random) != 0) {
		conf_destroy_clnt ();
		destroy_modules ();
		for (i = 0; save_env[i] != NULL; i++) {
			free (save_env[i]);
		}
		free (save_env);
		free (save_argv);
		return EXIT_FAILURE;
	}
#endif


	/*
	 * Register this agent with the server if not already done
	 */
	if (client_cert_check_and_create (	enable_ssl, server_crt,
						client_crt, client_key,
						quick_random,
						server, registrar_port,
						agent_port) != 0)
	{
		lwc_closeLog ();
		close_pid_file ();
		conf_destroy_clnt ();
		destroy_modules ();
		for (i = 0; save_env[i] != NULL; i++) {
			free (save_env[i]);
		}
		free (save_env);
		free (save_argv);
		return EXIT_FAILURE;
	}


	/*
	 * Initialize the network
	 */
	if (net_init (	client_crt, client_key, server_crt,
			quick_random, enable_ssl) != 0)
	{
		lwc_closeLog ();
		close_pid_file ();
		conf_destroy_clnt ();
		destroy_modules ();
		for (i = 0; save_env[i] != NULL; i++) {
			free (save_env[i]);
		}
		free (save_env);
		free (save_argv);
		return EXIT_FAILURE;
	}


	/*
	 * SSL setup
	 */
#if HAVE_LIBGNUTLS
	if (enable_ssl != 0 && quick_random == 0) {
		lwc_writeLog (LOG_INFO,
		_("Initializing the SSL layer (may take several minutes)..."));
	}
#endif
	if (net_init_server () != 0) {
		lwc_closeLog ();
		close_pid_file ();
		conf_destroy_clnt ();
		destroy_modules ();
		net_free ();
		for (i = 0; save_env[i] != NULL; i++) {
			free (save_env[i]);
		}
		free (save_env);
		free (save_argv);
		return EXIT_FAILURE;
	}
#if HAVE_LIBGNUTLS
	if (enable_ssl != 0 && quick_random == 0) {
		lwc_writeLog (LOG_INFO, _("Done"));
	}
#endif


	/*
	 * Signal handlers
	 */
	schedwi_memset (&sa, 0, sizeof (struct sigaction));
	sa.sa_handler = sigterm_action;
	sigemptyset (&(sa.sa_mask));
	sa.sa_flags = 0;
	if (sigaction (SIGTERM, &sa, NULL) != 0) {
		lwc_writeLog (
			LOG_CRIT,
			_("Cannot install signal handler for SIGTERM: %s"),
			strerror (errno));
		lwc_closeLog ();
		close_pid_file ();
		conf_destroy_clnt ();
		destroy_modules ();
		net_free ();
		for (i = 0; save_env[i] != NULL; i++) {
			free (save_env[i]);
		}
		free (save_env);
		free (save_argv);
		return EXIT_FAILURE;
	}

	schedwi_memset (&sa, 0, sizeof (struct sigaction));
	sa.sa_handler = sigchild_action;
	sigemptyset (&(sa.sa_mask));
	sa.sa_flags = 0;
	if (sigaction (SIGCHLD, &sa, NULL) != 0) {
		lwc_writeLog (
			LOG_CRIT,
			_("Cannot install signal handler for SIGCHLD: %s"),
			strerror (errno));
		lwc_closeLog ();
		close_pid_file ();
		conf_destroy_clnt ();
		destroy_modules ();
		net_free ();
		for (i = 0; save_env[i] != NULL; i++) {
			free (save_env[i]);
		}
		free (save_env);
		free (save_argv);
		return EXIT_FAILURE;
	}

	schedwi_memset (&sa, 0, sizeof (struct sigaction));
	sa.sa_handler = sighup_action;
	sigemptyset (&(sa.sa_mask));
	sa.sa_flags = 0;
	if (sigaction (SIGHUP, &sa, NULL) != 0) {
		lwc_writeLog (
			LOG_CRIT,
			_("Cannot install signal handler for SIGHUP: %s"),
			strerror (errno));
		lwc_closeLog ();
		close_pid_file ();
		conf_destroy_clnt ();
		destroy_modules ();
		net_free ();
		for (i = 0; save_env[i] != NULL; i++) {
			free (save_env[i]);
		}
		free (save_env);
		free (save_argv);
		return EXIT_FAILURE;
	}

	schedwi_memset (&sa, 0, sizeof (struct sigaction));
	sa.sa_handler = SIG_IGN;
	sigemptyset (&(sa.sa_mask));
	sa.sa_flags = 0;
	if (sigaction (SIGPIPE, &sa, NULL) != 0) {
		lwc_writeLog (
			LOG_CRIT,
			_("Cannot install signal handler for SIGPIPE: %s"),
			strerror (errno));
		lwc_closeLog ();
		close_pid_file ();
		conf_destroy_clnt ();
		destroy_modules ();
		net_free ();
		for (i = 0; save_env[i] != NULL; i++) {
			free (save_env[i]);
		}
		free (save_env);
		free (save_argv);
		return EXIT_FAILURE;
	}


	/*
	 * Start the process which is responsible to send the job
	 * results to the server
	 */
	if (send_process_start ((foreground == 0) ? 1 : 0) != 0) {
		lwc_closeLog ();
		close_pid_file ();
		conf_destroy_clnt ();
		destroy_modules ();
		net_free ();
		for (i = 0; save_env[i] != NULL; i++) {
			free (save_env[i]);
		}
		free (save_env);
		free (save_argv);
		return EXIT_FAILURE;
	}


	/*
	 * Network setup
	 */
	ret = conf_get_param_string ("IFACE_LISTEN", &iface);
	ret += conf_get_param_string ("ADDRESS_FAMILY", &address_family);
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif

	if (net_server (&sock, agent_port, iface, address_family) != 0) {
		send_process_stop ();
		lwc_closeLog ();
		close_pid_file ();
		conf_destroy_clnt ();
		destroy_modules ();
		net_free ();
		for (i = 0; save_env[i] != NULL; i++) {
			free (save_env[i]);
		}
		free (save_env);
		free (save_argv);
		return EXIT_FAILURE;
	}

	lwc_writeLog (LOG_INFO, _("Ready"));

	/*
	 * Main loop
	 */
	buff = NULL;
	buff_len = 0;
	while (1) {
		ret = conf_get_param_string ("SERVER", &server);
		ret += conf_get_param_string ("ALLOW_FROM", &allow_from);
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif
		if (allow_from[0] == '\0') {
			allow_from = server;
		}

		/* Wait for connections */
		ret = net_accept (&sock, &s, server, allow_from);
		if (ret < 0) {
			send_process_stop ();
			lwc_closeLog ();
			close_pid_file ();
			net_close (&sock);
			conf_destroy_clnt ();
			destroy_modules ();
			if (buff != NULL) {
				free (buff);
			}
			for (i = 0; save_env[i] != NULL; i++) {
				free (save_env[i]);
			}
			free (save_env);
			free (save_argv);
			return EXIT_FAILURE;
		}

		/* Parse the request */
		if (ret == 0) {
			signal_mask_all ();
			net_parse (&s, &buff, &buff_len);
			net_close (&s);
			signal_unmask ();
		}
	}
	return EXIT_SUCCESS;
}

/*------------------------======= End Of File =======------------------------*/
