/* Schedwi
   Copyright (C) 2011-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* reg_getmycrt_clnt.c -- Get the certificate from the server */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif


#include <lwc_log.h>
#include <net_utils_sock.h>
#include <parse_json_result.h>
#include <memsend.h>
#include <reg_getmycrt_clnt.h>

/*
 * Get the certificate from the server (schedwireg)
 *
 * Return:
 *   0 --> No error.  If `out_crt' is not NULL, it is set with the encoded
 *         certificate (see string2PEM() in cert_utils.c).  It must be freed
 *         by the caller.
 *  -1 --> Error (a message has been logged)
 *   1 --> Error on the server side.  In that case, and if not NULL, `out_crt'
 *         contains the error message from the server (`out_crt' may be NULL
 *         or empty if the server didn't send any message). It must be freed by
 *         the caller.  Also, `reason' is set if not NULL.  It contains the
 *         error code which indentify the error.
 */
int
getmycrt (	int sock, const char *local_port, char **out_crt,
		error_reason_t *reason)
{
	char *p;
	char *buff;
	size_t len;
	ssize_t nb_read;
	int ret;
	memsend_t *membuff;

#if HAVE_ASSERT_H
	assert (local_port != NULL);
#endif

	/* Ask for the certificate */
	p = port2number (local_port);
	if (p == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

	membuff = memsend_new ();
	memsend_append (membuff, "[ \"getmycrt\", { \"port\" : ");
	memsend_append (membuff, p);
	free (p);
	memsend_append (membuff, " } ]");
	net_write_sock (sock, memsend_getstring (membuff),
			memsend_getlength (membuff));
	memsend_destroy (membuff);

	/* Read the reply */
	buff = NULL;
	len = 0;
	nb_read = net_read_sock (sock, &buff, &len);
	if (nb_read < 0) {
		if (buff != NULL) {
			free (buff);
		}
		return -1;
	}

	ret = parse_json_result (buff, nb_read, out_crt, reason);
	if (buff != NULL) {
		free (buff);
	}
	if (ret == 0) {
		return 0;
	}
	else if (ret < 0) {
		return -1;
	}
	return 1;
}

/*-----------------============== End Of File ==============-----------------*/
