/* Schedwi
   Copyright (C) 2007-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* net_module_killjob.c -- Module to stop a job */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_CTYPE_H
#include <ctype.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <child_mgnt.h>
#include <lib_functions.h>
#include <net_module_killjob.h>
#include <memsend.h>
#include <JSON_parser.h>

enum steps {
	S_START,
	S_OUTER_ARRAY,
	S_OUTER_ARRAY_MODULE,
	S_DATA,
	S_KEY,
	S_JOB_LIST,
	S_END
};

struct json_cb {
	enum steps step;
	int nb_errors;
};


/*
 * JSON callback function
 */
static int
json_char_callback (void* ctx, int type, const JSON_value* value)
{
	struct json_cb *s = (struct json_cb *)ctx;


	switch (type) {
		case JSON_T_ARRAY_BEGIN:
			if (s->step == S_START) {
				s->step = S_OUTER_ARRAY;
			}
			else if (s->step == S_KEY) {
				s->step = S_JOB_LIST;
			}
			break;
		case JSON_T_STRING:
			if (s->step == S_OUTER_ARRAY) {
				s->step = S_OUTER_ARRAY_MODULE;
			}
			else if (   s->step == S_JOB_LIST
				 && (value->vu.str.value)[0] != '\0')
			{
				if (terminate_job (value->vu.str.value) != 0) {
					(s->nb_errors)++;
				}
			}
			break;
		case JSON_T_OBJECT_BEGIN:
			if (s->step == S_OUTER_ARRAY_MODULE) {
				s->step = S_DATA;
			}
			break;
		case JSON_T_KEY:
			if (s->step == S_DATA) {
				s->step = S_KEY;
			}
			break;
		case JSON_T_ARRAY_END:
			if (s->step == S_JOB_LIST) {
				s->step = S_END;
			}
			break;
	}

	return 1;
}



/*
 * Parse the server request (in buffer) and stop the specified jobs
 *
 * Return:
 *   0 --> No error
 *  -2 --> Error kill a job (lwc_writeLog() is used to display an error
 *         message)
 */
int
kill_jobs (net_id *sock, char *buffer, size_t size)
{
	char *s;
	int i;
        JSON_config config;
        struct JSON_parser_struct *jc;
	struct json_cb cb_data;
	memsend_t *membuff;


#if HAVE_ASSERT_H
	assert (buffer != NULL);
#endif

	/* JSON parser initialization */
	init_JSON_config (&config);

	cb_data.step                  = S_START;
	cb_data.nb_errors             = 0;
	config.depth                  = 20;
	config.callback               = &json_char_callback;
	config.allow_comments         = 0;
	config.handle_floats_manually = 0;
	config.callback_ctx           = &cb_data;

	jc = new_JSON_parser (&config);

	/* Parse the JSON string */
	for (i = 0; i < size && cb_data.step != S_END; i++) {
		/*
		 * No need to check the return code as the JSON string has
		 * already been parse once in net_parse.c
		 */
		JSON_parser_char (jc, (unsigned char)(buffer[i]));
	}
	delete_JSON_parser (jc);

	if (cb_data.nb_errors > 0) {
		membuff = memsend_new ();
		memsend_append (membuff,
				"{ \"success\" : false, \"data\" : \"");
		memsend_append (membuff, _("Failed to stop the job(s)"));
		memsend_append (membuff, "\" }");
		net_write (	sock, memsend_getstring (membuff),
				memsend_getlength (membuff));
		memsend_destroy (membuff);
		return -2;
	}
	else {
		s = "{ \"success\" : true }";
		net_write (sock, s, schedwi_strlen (s));
		return 0;
	}
}

/*-----------------============== End Of File ==============-----------------*/
