/* Schedwi
   Copyright (C) 2012-2014 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* @file memsend.c
 * Memory buffer to store data (append allowed)
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#include <xmem.h>
#include <memsend.h>


/*
 * Create a new memsend object
 *
 * Return:
 *   The new object (to free with memsend_destroy)
 */
memsend_t *
memsend_new ()
{
	memsend_t *r;


	r = (memsend_t *) xmalloc (sizeof (memsend_t));
	r->buff = (char *) xmalloc (1024);
	r->len = 1024;
	r->idx = 0;
	return r;
}


/*
 * Free the provided memsend object
 */
void
memsend_destroy (memsend_t *r)
{
	if (r != NULL) {
		if (r->buff != NULL) {
			free (r->buff);
		}
		free (r);
	}
}


/*
 * Get the null-terminated string containing the added data
 *
 * Return:
 *   The null-terminated string
 */
const char *
memsend_getstring (memsend_t *r)
{
	if (r == NULL || r->buff == NULL) {
		return NULL;
	}
	r->buff[r->idx] = '\0';
	return r->buff;
}


/*
 * Get the length of the string
 *
 * Return:
 *   The length of the string
 */
unsigned int
memsend_getlength (memsend_t *r)
{
	if (r == NULL || r->buff == NULL) {
		return 0;
	}
	return r->idx;
}


/*
 * Reset the memory of the object. Previously added data are lost
 */
void
memsend_clear (memsend_t *r)
{
	if (r != NULL) {
		r->idx = 0;
	}
}


/*
 * Append a string to the object
 */
void
memsend_append_len (memsend_t *r, const char *s, size_t len)
{
	if (r == NULL) {
		return;
	}
	if (r->idx + len >= r->len) {
		r->buff = (char *) xrealloc (r->buff, r->len + len + 1024);
		r->len += len + 1024;
	}
	strncpy (r->buff + r->idx, s, len);
	r->idx += len;
}


/*
 * Append a string to the object
 */
void
memsend_append (memsend_t *r, const char *s)
{
	if (s == NULL || s[0] == '\0') {
		return;
	}
	memsend_append_len(r, s, strlen(s));
}

/*------------------------======= End Of File =======------------------------*/
