/* Schedwi
   Copyright (C) 2007-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * job_parameters.c -- Execution parameters for a job
 *
 * The parameters (members of the job_parameters structure) are the
 * following:
 *
 *           path --> The job full name
 *        command --> The command to be run
 *       username --> User name to use to run the command
 *    file_stdout --> File name of the command stdout redirection
 *    file_stderr --> File name of the command stderr redirection
 *    cgroup_name --> Linux Control Group name (path from a controller mount
 *                    directory)
 *       load_env --> Do the user environement should be load (1) or
 *                    not (0).  If yes (1), the user shell is run with the
 *                    command as a parameter (ie. /bin/sh -c command)
 *         detach --> Whether the job should be start in detach mode or not.
 *                    In detach mode, Schedwi will not wait on the end of the
 *                    job.  This may be usefull to start a daemon, a service
 *                    or a long running process.  Only the success of the
 *                    start of the job is reported (basically if the provided
 *                    command can be exec)
 *      arguments --> Arguments list
 *        environ --> Environement to load (several other variables will be
 *                    added, like SHELL, PATH, LOGNAME, ...)
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <job_parameters.h>


/*
 * Initialize a job_parameters structure
 */
void
init_job_parameters (job_parameters_t *j)
{
	if (j != NULL) {
		j->path = NULL;
		j->command = NULL;
		j->username = NULL;
		j->file_stdout = NULL;
		j->file_stderr = NULL;
		j->cgroup_name = NULL;
		j->load_env = 0;
		j->detach = 0;
		init_argument (&(j->arguments));
		init_environment (&(j->environ));
	}
}


/*
 * Free the content of a job_parameters structure
 */
void
destroy_job_parameters (job_parameters_t *j)
{
	if (j != NULL) {
		if (j->path != NULL) {
			free (j->path);
		}
		if (j->command != NULL) {
			free (j->command);
		}
		if (j->username != NULL) {
			free (j->username);
		}
		if (j->file_stdout != NULL) {
			free (j->file_stdout);
		}
		if (j->file_stderr != NULL) {
			free (j->file_stderr);
		}
		if (j->cgroup_name != NULL) {
			free (j->cgroup_name);
		}
		destroy_argument (&(j->arguments));
		destroy_environment (&(j->environ));
		init_job_parameters (j);
	}
}


/*
 * Copy a string
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
static int
add_attr_job_parameters (char **attr, const char *str, size_t len)
{
	char *tmp;

#if HAVE_ASSERT_H
	assert (attr != NULL);
#endif

	if (str == NULL) {
		if (*attr != NULL) {
			free (*attr);
			*attr = NULL;
		}
		return 0;
	}

	if (*attr == NULL || len >= schedwi_strlen (*attr)) {
		tmp = (char *)malloc (len + 1);
		if (tmp == NULL) {
			return -1;
		}
		if (*attr != NULL) {
			free (*attr);
		}
		*attr = tmp;
	}
	strncpy (*attr, str, len);
	(*attr)[len] = '\0';
	return 0;
}


/*
 * Add the path to the job_parameters structure
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
int
add_path_job_parameters (job_parameters_t *j, const char *path, size_t len)
{
	if (j == NULL) {
		return 0;
	}

	return add_attr_job_parameters (&(j->path), path, len);
}


/*
 * Add the command to the job_parameters structure
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
int
add_command_job_parameters (	job_parameters_t *j, const char *command,
				size_t len)
{
	if (j == NULL) {
		return 0;
	}

	return add_attr_job_parameters (&(j->command), command, len);
}


/*
 * Add the username to the job_parameters structure
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
int
add_username_job_parameters (	job_parameters_t *j, const char *username,
				size_t len)
{
	if (j == NULL) {
		return 0;
	}

	return add_attr_job_parameters (&(j->username), username, len);
}


/*
 * Add the output file to the job_parameters structure
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
int
add_stdout_job_parameters (job_parameters_t *j, const char *file, size_t len)
{
	if (j == NULL) {
		return 0;
	}

	return add_attr_job_parameters (&(j->file_stdout), file, len);
}


/*
 * Add the output error file to the job_parameters structure
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
int
add_stderr_job_parameters (job_parameters_t *j, const char *file, size_t len)
{
	if (j == NULL) {
		return 0;
	}

	return add_attr_job_parameters (&(j->file_stderr), file, len);
}


/*
 * Add the Linux Control Group name to the job_parameters structure
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
int
add_cgroup_job_parameters (job_parameters_t *j, const char *cgroup_name,
			   size_t len)
{
	if (j == NULL) {
		return 0;
	}

	return add_attr_job_parameters (&(j->cgroup_name), cgroup_name, len);
}


/*
 * Set the `load user environement' flag
 */
void
add_loadenv_job_parameters (job_parameters_t *j, int flag)
{
	if (j != NULL) {
		j->load_env = flag;
	}
}


/*
 * Set the `detach mode' flag
 */
void
add_detach_job_parameters (job_parameters_t *j, int flag)
{
	if (j != NULL) {
		j->detach = flag;
	}
}


/*
 * Add an argument
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
int
add_arg_job_parameters (job_parameters_t *j, const char *arg, size_t len)
{
	if (j == NULL) {
		return 0;
	}
	return add_arg_s (&(j->arguments), arg, len);
}


/*
 * Add an environment variable
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
int
add_env_job_parameters (job_parameters_t *j, const char *key, const char *val)
{
	if (j == NULL) {
		return 0;
	}
	return add_env (&(j->environ), key, val);
}


/*
 * Concat the provided environment_t object at the end of the job environment.
 * Warning: the provided environment_t object will be emptied by this function.
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
int
concat_env_job_parameters (job_parameters_t *j, environment_t *to_concat)
{
	if (j == NULL) {
		return 0;
	}
	return concat_env (&(j->environ), to_concat);
}

/*-----------------============== End Of File ==============-----------------*/
