/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * env_utils.h -- Environment array management functions
 *
 * The environment array is an array of strings.  By convention these strings
 * have the form `name=value'.  This array is embedded in the environment_t
 * structure but can be used directly by system functions like exec.
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#endif

#include <lib_functions.h>
#include <env_utils.h>


/*
 * Init the environment structure
 */
void
init_environment (environment_t *e)
{
	if (e != NULL) {
		e->env = NULL;
		e->length = 0;
		e->idx = 0;
	}
}


/*
 * Free the environment structure
 */
void
destroy_environment (environment_t *e)
{
	int i;

	if (e != NULL) {
		if (e->env != NULL && e->length > 0) {
			for (i = 0; i < e->idx; i++) {
				if ((e->env)[i] != NULL) {
					free ((e->env)[i]);
				}
			}
			free (e->env);
		}
		e->env = NULL;
		e->length = 0;
		e->idx = 0;
	}
}


/*
 * Add a pair key-value at the end of the environment array.  The key and
 * value parameters are provided with their length.
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
int
add_env_s (	environment_t *e,
		const char *key, size_t key_len,
		const char *val, size_t val_len)
{
	char **tmp;

	if (e == NULL || key == NULL || key_len == 0) {
		return 0;
	}

	if (e->idx + 1 >= e->length) {
		tmp = (char **) realloc (e->env,
					(e->length + 10) * sizeof (char *));
		if (tmp == NULL) {
			return -1;
		}
		e->env = tmp;
		e->length += 10;
	}

	(e->env)[e->idx] = (char *) malloc (key_len + val_len + 2);
	if ((e->env)[e->idx] == NULL) {
		return -1;
	}
	strncpy ((e->env)[e->idx], key, key_len);
	(e->env)[e->idx][key_len] = '=';
	if (val == NULL || val_len == 0) {
		(e->env)[e->idx][key_len + 1] = '\0';
	}
	else {
		strncpy ((e->env)[e->idx] + key_len + 1, val, val_len);
		(e->env)[e->idx][key_len + val_len + 1] = '\0';
	}
	(e->env)[++e->idx] = NULL;
	return 0;
}


/*
 * Add a pair key-value at the end of the environment array.
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
int
add_env (environment_t *e, const char *key, const char *val)
{
	return add_env_s (e, key, schedwi_strlen (key),
				val, schedwi_strlen (val));
}


/*
 * Search the provided string (var) in the env array (of size var_len)
 *
 * Return:
 *    The index of the var in the env array or
 *    -1 if not found
 */
int
search_environment (environment_t *e, const char *var, size_t var_len)
{
	int i;

	if (e == NULL || var == NULL || var_len == 0) {
		return -1;
	}

	for (i = 0; i < e->idx; i++) {
		if (	   schedwi_strncmp (var, (e->env)[i], var_len) == 0
			&& (e->env)[i][var_len] == '=')
		{
			return i;
		}
	}
	return -1;
}


/*
 * Return the variable at the idx position in the array structure (or NULL
 * if idx is out of range or e is NULL)
 */
const char *
getvar_environment (environment_t *e, int idx)
{
	if (e == NULL || idx < 0 || idx >= e->idx) {
		return NULL;
	}
	return (e->env)[idx];
}


/*
 * Set the dst environment with the src environment (which is empty afterward)
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
int
concat_env (environment_t *dst, environment_t *src)
{
	int i;
	char **tmp;

	if (dst == NULL || src == NULL) {
		return 0;
	}

	if (dst->idx + src->idx >= dst->length) {
		tmp = (char **) realloc (dst->env,
				(dst->idx + src->idx + 10) * sizeof (char *));
		if (tmp == NULL) {
			return -1;
		}
		dst->env = tmp;
		dst->length = dst->idx + src->idx + 10;
	}
	for (i = 0; i < src->idx; i++) {
		(dst->env)[(dst->idx)++] = (src->env)[i];
	}
	(dst->env)[dst->idx] = NULL;
	free (src->env);
	init_environment (src);

	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
