/* Schedwi
   Copyright (C) 2007-2015 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* child_mgnt.c -- Manage running child processes in a linked list */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_SIGNAL_H
#include <signal.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lwc_log.h>
#include <lib_functions.h>
#include <lwc_linkedlist.h>
#include <memsend.h>
#include <utils.h>
#include <child_mgnt.h>


/* List of the running jobs */
static lwc_LL *list_jobs = NULL;


/* Internal structure that define a running job */
struct child_str {
#if HAVE_PID_T
	pid_t child;
#else
	int child;
#endif
	char *job_id;
	char *path;
};
typedef struct child_str child_t;


/*
 * Comparison function between a process id and a child_t structure
 */
static int
compar_pid (const void *a, const void *b)
{
#if HAVE_PID_T
	const pid_t *p = a;
#else
	const int *p = a;
#endif
	const child_t *c = b;

#if HAVE_ASSERT_H
	assert (p != NULL && c != NULL);
#endif

	return *p - c->child;
}

/*
 * Comparison function between a job id and a child_t structure
 */
static int
compar_job_id (const void *a, const void *b)
{
	const char *id = a;
	const child_t *c = b;

#if HAVE_ASSERT_H
	assert (c != NULL);
#endif

	if (id == NULL) {
		if (c->job_id == NULL) {
			return 0;
		}
		else {
			return -1;
		}
	}
	if (c->job_id == NULL) {
		return 1;
	}
	return strcmp (id, c->job_id);
}

/*
 * Comparison function between two child_t structures
 */
static int
compar_child (const void *a, const void *b)
{
	const child_t *k = a;
	const child_t *c = b;

#if HAVE_ASSERT_H
	assert (k != NULL && c != NULL);
#endif

	return k->child - c->child;
}


/*
 * Add a new child to the list of the running children
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
#if HAVE_PID_T
int
add_child (pid_t child, const char *job_id, const char *path)
#else
int
add_child (int child, const char *job_id, const char *path)
#endif
{
	child_t *new_child;
	void *old;


#if HAVE_ASSERT_H
	assert (job_id != NULL);
#endif

	if (list_jobs == NULL) {
		list_jobs = lwc_newLL ();
	}

	new_child = (child_t *) malloc (sizeof (child_t));
	if (new_child == NULL) {
		return -1;
	}

	new_child->job_id = (char *) malloc (schedwi_strlen (job_id) + 1);
	if (new_child->job_id == NULL) {
		free (new_child);
		return -1;
	}
	strcpy (new_child->job_id, job_id);

	if (path == NULL) {
		new_child->path = NULL;
	}
	else {
		new_child->path = (char *) malloc (schedwi_strlen (path) + 1);
		if (new_child->path == NULL) {
			free (new_child->job_id);
			free (new_child);
			return -1;
		}
		strcpy (new_child->path, path);
	}

	new_child->child = child;
	lwc_replaceLL (list_jobs, new_child, compar_child, &old);
	if (old != NULL) {
		if (((child_t *)old)->path != NULL) {
			free (((child_t *)old)->path);
		}
		if (((child_t *)old)->job_id != NULL) {
			free (((child_t *)old)->job_id);
		}
		free (old);
	}

	return 0;
}


/*
 * Remove a child from the list of the running children.  Return in
 * *job_id the job_id of the removed child (must be freed my the caller using
 * free()).  If job_id is NULL, no value is returned in this variable.
 *
 * Return
 *    0 --> No error (*job_id is set if not NULL and must be freed by free())
 *   -1 --> Child not found in the list of children
 */
#if HAVE_PID_T
int
delete_child (pid_t child, char **job_id)
#else
int
delete_child (int child, char **job_id)
#endif
{
	child_t *ret_child;


	ret_child = lwc_delNodeLL (list_jobs, &child, compar_pid);
	if (ret_child == NULL) {
		return -1;
	}

	if (job_id != NULL) {
		*job_id = ret_child->job_id;
	}
	else {
		if (ret_child->job_id != NULL) {
			free (ret_child->job_id);
		}
	}
	if (ret_child->path != NULL) {
		free (ret_child->path);
	}

	free (ret_child);
	return 0;
}


/*
 * Terminate a job
 *
 * Return:
 *    0 --> No error
 *   -1 --> kill error
 *    1 --> Unknown job
 */
int
terminate_job (const char *job_id)
{
	child_t *ret_child;
	int ret, workload;
	char *id;


	ret_child = lwc_searchLL (list_jobs, job_id, compar_job_id);
	if (ret_child == NULL) {
		lwc_writeLog (	LOG_WARNING,
				_("Job %s: Cannot stop (not found)"),
				job_id);
		return 1;
	}
	id = split_jobid (job_id, &workload);
	lwc_writeLog (	LOG_INFO,
			_("Workload %d: %s (id %s): Stopping (PID %d)"),
			workload,
			(ret_child->path == NULL) ? "": ret_child->path,
			id,
			ret_child->child);
	ret = kill (ret_child->child, SIGTERM);
	if (ret != 0) {
		lwc_writeLog (LOG_WARNING,
			_("Workload %d: %s (id %s): Failed to stop: %s"),
			workload,
			(ret_child->path == NULL) ? "": ret_child->path,
			id,
			strerror (errno));
	}
	return ret;
}


/*
 * Write the list of all the running jobs to the provided socket
 */
void
print_jobs (net_id *fd)
{
	child_t *ret_child;
	char is_first = 1;
	memsend_t *membuff;


	membuff = memsend_new ();
	memsend_append (membuff, "{ \"success\" : true, \"data\" : [ ");
	lwc_rewindLL (list_jobs);
	while ((ret_child = lwc_nextLL (list_jobs)) != NULL) {
		if (ret_child->job_id != NULL) {
			if (is_first) {
				is_first = 0;
				memsend_append (membuff, "\"");
			}
			else {
				memsend_append (membuff, ", \"");
			}
			memsend_append (membuff, ret_child->job_id);
			memsend_append (membuff, "\"");
		}
	}

	memsend_append (membuff, " ] }");
	net_write (	fd, memsend_getstring (membuff),
			memsend_getlength (membuff));
	memsend_destroy (membuff);
}

/*-----------------============== End Of File ==============-----------------*/
