/* Schedwi
   Copyright (C) 2013 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* sql_cluster.c -- Get the host IDs for a job hierarchy */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <sql_common.h>
#include <sql_cluster.h>


#define SQL_GET_JOB_CLUSTER "SELECT host_clusters.host_id FROM host_clusters,job_cluster_s WHERE job_cluster_s.workload_date=%d AND job_cluster_s.job_id=%ld AND job_cluster_s.cluster_id = host_clusters.cluster_id"
#define SQL_GET_JOB_HOST "SELECT host_id FROM job_host_s WHERE workload_date=%d AND job_id=%ld"


/*
 * Retrieve the host IDs for a job hierarchy.  The provided hierarchy list
 * (in lst) must have been built by get_hierarchy_list() prior to calling
 * this function.
 *
 * Return:
 *     0 --> No error.  host_ids contains the host ID list and must be
 *           freed by the caller by
 *               lwc_delLL (host_ids, (void (*)(const void *))free);
 *           If no host ID  has not been found in the hierarchy, host_ids is
 *           set to NULL.
 *    -1 --> Memory allocation error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message).
 *    -2 --> Database error (if err_msg is not NULL it contains an error
 *           message which must be freed by the caller.  *err_msg may be NULL
 *           if there is not enough memory to store the error message).
 */
int
get_job_hosts (	int workload_date, lwc_LL *lst, lwc_LL **host_ids,
		char **err_msg)
{
	dbi_conn sql;
	int ret;
	lwc_LL *rows, *ids;
	row_item_t *row;
	char *buf, *s;
	unsigned int buf_len;
	unsigned long long int *id;

#if HAVE_ASSERT_H
	assert (lst != NULL && host_ids != NULL);
#endif

	/* Connect to the database */
	sql = begin_sql (err_msg);
	if (sql == NULL) {
		return -2;
	}

	buf = NULL;
	buf_len = 0;
	lwc_rewindLL (lst);
	while ((id = (unsigned long long int *)lwc_nextLL (lst)) != NULL) {
		/* Try to get the host IDs from a cluster */
		ret = sql_select (&buf, &buf_len, err_msg, NULL, &rows,
				SQL_GET_JOB_CLUSTER,
				SQL_INT, (long int)workload_date,
				SQL_INT, (long int)*id,
				SQL_END);
		if (ret != 0) {
			end_sql ();
			if (buf != NULL) {
				free (buf);
			}
			return ret;
		}

		/* No in a cluster */
		if (lwc_getNumNode (rows) <= 0) {
			/*
			 * Check if the job is directly associated with a host
			 */
			lwc_delLL (rows, (void (*)(const void *))sql_free_row);
			ret = sql_select (&buf, &buf_len, err_msg, NULL, &rows,
					SQL_GET_JOB_HOST,
					SQL_INT, (long int)workload_date,
					SQL_INT, (long int)*id,
					SQL_END);
			if (ret != 0) {
				end_sql ();
				if (buf != NULL) {
					free (buf);
				}
				return ret;
			}
			if (lwc_getNumNode (rows) <= 0) {
				/*
				 * Nothing found - The same request will then
				 * be issued for the parent
				 */
				lwc_delLL (rows,
					(void (*)(const void *))sql_free_row);
				continue;
			}
		}

		ids = lwc_newLL ();
		if (ids == NULL) {
			lwc_delLL (rows, (void (*)(const void *))sql_free_row);
			end_sql ();
			if (buf != NULL) {
				free (buf);
			}
			if (err_msg != NULL) {
				s = _("Memory allocation error");
				*err_msg = (char *) malloc (schedwi_strlen (s)
								+ 1);
				if (*err_msg != NULL) {
					strcpy (*err_msg, s);
				}
			}
			return -1;
		}

		/*
		 * Retrieve the host IDs and stored them in the list to
		 * be returned
		 */
		while ((row = (row_item_t *) lwc_delStartLL (rows)) != NULL) {
			id = (unsigned long long int *)
				malloc (sizeof (unsigned long long int));
			if (id == NULL || lwc_addEndLL (ids, id) != 0) {
				if (id != NULL) {
					free (id);
				}
				lwc_delLL (ids, (void (*)(const void *))free);
				sql_free_row (row);
				lwc_delLL (rows,
					(void (*)(const void *))sql_free_row);
				end_sql ();
				if (buf != NULL) {
					free (buf);
				}
				if (err_msg != NULL) {
					s = _("Memory allocation error");
					*err_msg = (char *) malloc (
						schedwi_strlen (s) + 1);
					if (*err_msg != NULL) {
						strcpy (*err_msg, s);
					}
				}
				return -1;
			}
			*id = (unsigned long long int) sql_row_item2ll (
								&(row[0]));
			sql_free_row (row);
		}
		lwc_delLL (rows, (void (*)(const void *)) sql_free_row);

		/* Copy the returned values */
		*host_ids = ids;

		/* Close the database and free the working buffer */
		end_sql ();
		if (buf != NULL) {
			free (buf);
		}
		return 0;
	}

	/*
	 * Host ID not found in the database
	 */

	/* Close the database and free the working buffer */
	end_sql ();
	if (buf != NULL) {
		free (buf);
	}
	*host_ids = NULL;

	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
