/* Schedwi
   Copyright (C) 2007-2010 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* sql_calendar.c -- calendars table management functions */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <sql_common.h>
#include <sql_calendar.h>

#define SQL_CAL_LIST "SELECT id,parent,name,entry_type FROM calendars ORDER BY parent,entry_type DESC,name"
#define SQL_CAL_DETAIL "SELECT id,parent,name,entry_type,description,formula FROM calendars WHERE id=%s"
#define SQL_CAL_PARENTS "SELECT id,parent,name,entry_type,description,formula FROM calendars WHERE parent=%s"
#define SQL_CAL_UPDATE "UPDATE calendars SET name=%s,description=%s,formula=%s WHERE id=%s"
#define SQL_CAL_DELETE "DELETE FROM calendars WHERE id IN (%s) OR parent IN (%s)"
#define SQL_CAL_REPARENT "UPDATE calendars SET parent=%s WHERE id=%s"
#define SQL_CAL_NEW "INSERT INTO calendars (name,parent,entry_type,description,formula) VALUES (%s,%s,%s,%s,%s)"
#define SQL_CAL_ONLY_LIST "SELECT calendars_s.id,calendars_s.name,calendars_s.formula FROM calendars_s,job_main_s WHERE calendars_s.workload_date=%d AND job_main_s.workload_date=%d AND calendars_s.entry_type='0' AND calendars_s.id=job_main_s.cal_id"


/*
 * Retrieve the calendar list from the database
 *
 * Return:
 *     0 --> No error.  rows contains the retrieved calendars and must be freed
 *           by the caller by
 *              lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_cal_list (	lwc_LL **rows,
		void (*error_func)(void *, const char *, int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;

#if HAVE_ASSERT_H
	assert (rows != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, rows,
				SQL_CAL_LIST, SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Retrieve the calendar list (and not the folders) from the database.  Only
 * used calendars (associated with at least a job or a jobsets) are retrieved.
 *
 * Return:
 *     0 --> No error.  rows contains the retrieved calendars and must be freed
 *           by the caller by
 *              lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_cal_only_list (int workload_date,
		lwc_LL **rows,
		void (*error_func)(void *, const char *, int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;

#if HAVE_ASSERT_H
	assert (rows != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, rows,
				SQL_CAL_ONLY_LIST,
				SQL_INT, (long int)workload_date,
				SQL_INT, (long int)workload_date,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Get the calendar details from the database
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_cal_get_details (	const char *id,
			void (*set_cal_id)(unsigned long long int, void *),
			void (*set_cal_parent)(unsigned long long int, void *),
			void (*set_cal_name)(const char *, void *),
			void (*set_cal_type)(char, void *),
			void (*set_cal_descr)(const char *, void *),
			void (*set_cal_formula)(const char *, void *),
			void *user_data_set_cal,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;
	lwc_LL *rows;
	row_item_t *row;

#if HAVE_ASSERT_H
	assert (id != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows,
				SQL_CAL_DETAIL, SQL_STRING, id, SQL_END);

	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	row = (row_item_t *) lwc_delStartLL (rows);
	if (row == NULL) {
		lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Unknown calendar in the database"),
					ret);
		}
		return -2;
	}
	lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
	if (set_cal_id != NULL) {
		set_cal_id (
			(unsigned long long int)sql_row_item2ll (&(row[0])),
			user_data_set_cal);
	}
	if (set_cal_parent != NULL) {
		set_cal_parent (
			(unsigned long long int)sql_row_item2ll (&(row[1])),
			user_data_set_cal);
	}
	if (set_cal_name != NULL) {
		set_cal_name (row[2].value_string, user_data_set_cal);
	}
	if (set_cal_type != NULL) {
		set_cal_type (
			(char)sql_row_item2ll (&(row[3])),
			user_data_set_cal);
	}
	if (set_cal_descr != NULL) {
		set_cal_descr (row[4].value_string, user_data_set_cal);
	}
	if (set_cal_formula != NULL) {
		set_cal_formula (row[5].value_string, user_data_set_cal);
	}

	sql_free_row (row);

	return 0;
}


/*
 * Get the calendar details from the database
 *
 * Return:
 *     0 --> No error.  rows contains the retrieved calendars and must be freed
 *           by the caller by
 *              lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_cal_get_main (	const char *id,
			lwc_LL **row,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;

#if HAVE_ASSERT_H
	assert (id != NULL && row != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, row,
				SQL_CAL_DETAIL, SQL_STRING, id, SQL_END);

	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Get the calendars where the parent is PARENT_ID
 *
 * Return:
 *     0 --> No error.  rows contains the retrieved calendars and must be freed
 *           by the caller by
 *              lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_cal_get_parent (	const char *parent_id,
			lwc_LL **rows,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;

#if HAVE_ASSERT_H
	assert (parent_id != NULL && rows != NULL);
#endif

	ret = sql_select (NULL, NULL, &err_msg, NULL, rows,
			SQL_CAL_PARENTS, SQL_STRING, parent_id, SQL_END);

	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Insert a new calendar in the database
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_cal_new (	const char *name,
		const char *parent_id,
		const char *type,
		const char *description,
		const char *formula,
		unsigned long long int *new_env_id,
		void (*error_func)(void *, const char *, int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;

#if HAVE_ASSERT_H
	assert (name != NULL && parent_id != NULL && type != NULL);
#endif

	ret = sql_non_select (	NULL, NULL, &err_msg, new_env_id, NULL,
				"calendars",
				SQL_CAL_NEW,
				SQL_STRING, name,
				SQL_STRING, parent_id,
				SQL_STRING, type,
				SQL_STRING, description,
				SQL_STRING, formula,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Update a calendar in the database
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_cal_update (	const char *id,
			const char *name,
			const char *description,
			const char *formula,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;

#if HAVE_ASSERT_H
	assert (id != NULL && name != NULL);
#endif

	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL, NULL,
				SQL_CAL_UPDATE,
				SQL_STRING, name,
				SQL_STRING, description,
				SQL_STRING, formula,
				SQL_STRING, id,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Change the parent ID if a calendar (or folder)
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_cal_reparent (	const char *parent_id,
			const char *id,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;

#if HAVE_ASSERT_H
	assert (parent_id != NULL && id != NULL);
#endif

	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL, NULL,
				SQL_CAL_REPARENT,
				SQL_STRING, parent_id,
				SQL_STRING, id,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Remove the folder/calendar from the database
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_cal_delete (	char **cal_id_array, unsigned int len_array,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	dbi_conn sql;
	char *err_msg = NULL;
	char *list_str, *value;
	unsigned int list_len, i, idx;
	int ret;


#if HAVE_ASSERT_H
	assert (cal_id_array != NULL);
#endif

	/* Connect to the database */
	sql = begin_sql (&err_msg);
	if (sql == NULL) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, 0);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return -2;
	}

	/* Build the SQL list */
	for (list_len = i = 0; i < len_array && cal_id_array[i] != NULL; i++) {
		list_len += schedwi_strlen (cal_id_array[i]) * 2 + 3;
	}

	list_str = (char *) malloc (list_len);
	if (list_str == NULL) {
		end_sql ();
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"),
					0);
		}
		return -1;
	}

	for (idx = i = 0; i < len_array && cal_id_array[i] != NULL; i++) {
		value = sql_escape (sql, cal_id_array[i]);
		if (value == NULL) {
			end_sql ();
			free (list_str);
			if (error_func != NULL) {
				error_func (	user_data_error_func,
						_("Memory allocation error"),
						-1);
			}
			return -1;
		}
		strcpy (list_str + idx, value);
		idx += schedwi_strlen (value);
		list_str[idx++] = ',';
		free (value);
	}
	list_str[idx - 1] = '\0';

	/* Delete the calendar */
	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL, NULL,
				SQL_CAL_DELETE,
				SQL_STRING_NON_ESCAPE, list_str,
				SQL_STRING_NON_ESCAPE, list_str,
				SQL_END);
	if (ret != 0) {
		free (list_str);
		end_sql ();
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	free (list_str);
	end_sql ();
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
