/* Schedwi
   Copyright (C) 2007-2013 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * net_utils_sock.c -- Useful network functions (basic socket connection
 * without SSL)
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif

#if HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif

#if HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif

#if HAVE_NETDB_H
#include <netdb.h>
#endif

#if HAVE_SYS_SELECT_H
#include <sys/select.h>
#else
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_TIME_H
#include <time.h>
#endif

#if HAVE_CTYPE_H
#include <ctype.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_TCPD_H
#include <tcpd.h>
#endif

#if HAVE_SYSLOG_H
#include <syslog.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#ifndef HAVE_RECV
#define recv(a,b,c,d) read(a,b,c)
#endif

#ifndef HAVE_SEND
#define send(a,b,c,d) write(a,b,c)
#endif

#ifndef PF_INET
#define PF_INET AF_INET
#endif

#ifndef HAVE_CLOSESOCKET
#define closesocket(x) close(x)
#endif

#include <lib_functions.h>
#include <lwc_log.h>
#include <net_utils_sock.h>
#include <JSON_parser.h>

#define BUFFER_LEN 1024


/*
 *  Check that the provided port number/name is valid
 *
 *  Return :
 *     0 on success OR
 *     getaddrinfo error code (invalid port). Use gai_strerror() to get a
 *                 description.
 */
int
check_port (const char *port)
{
	struct addrinfo hints;
	struct addrinfo *result;
	int ret;

#if HAVE_ASSERT_H
	assert (port != NULL);
#endif

	schedwi_memset (&hints, 0, sizeof (struct addrinfo));
	hints.ai_family   = AF_UNSPEC;     /* Allow IPv4 or IPv6 */
	hints.ai_socktype = SOCK_STREAM;   /* Stream socket */
	hints.ai_flags    = 0;
	hints.ai_protocol = 0;             /* Any protocol */

	ret = getaddrinfo (NULL, port, &hints, &result);
	if (ret != 0) {
		return ret;
	}
	freeaddrinfo (result);
	return 0;
}


/*
 * Create a socket (plus bind and listen)
 * domain, type, protocol_s : see socket(2)
 * addr, addrlen  : see bind(2)
 *
 * Return:
 *   The socket OR
 *   -1 in case of error
 */
static int
net_create_sock (int domain, int type, int protocol_s,
		const struct sockaddr *addr, socklen_t addrlen)
{
	int sock, opt;

	/* Create the socket */
	sock = socket (domain, type, protocol_s);
	if (sock < 0) {
		lwc_writeLog (LOG_ERR, _("socket: %s"), strerror (errno));
		return -1;
	}

	/* Set the REUSEADDR flag */
	opt = 1;
	if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof (opt)) != 0)
	{
		lwc_writeLog (LOG_ERR, _("setsockopt: %s"), strerror (errno));
	}

	/* Allow the IPv6 bind to the same port as in IPv4 */
	if (	   domain == AF_INET6
		&& setsockopt(sock, IPPROTO_IPV6, IPV6_V6ONLY,
				&opt, sizeof (opt)) != 0)
	{
		lwc_writeLog (LOG_ERR, _("setsockopt: %s"), strerror (errno));
	}

	/* Bind to the address */
	if (bind (sock, addr, addrlen) != 0) {
		lwc_writeLog (LOG_ERR, _("bind: %s"), strerror (errno));
		closesocket (sock);
		return -1;
	}

	/* Prepare to accept requests */
	if (listen (sock, SCHEDWI_LISTEN_BACKLOG) != 0) {
		lwc_writeLog (LOG_ERR, _("listen: %s"), strerror (errno));
		closesocket (sock);
		return -1;
	}

	return sock;
}


/*
 * Create a socket ready to accept requests
 *
 * port_number is the port number to listen to (a number or a service name
 *             in /etc/services)
 *       iface is the network interface to listen to (NULL or empty means all
 *             the interfaces)
 *
 * Return:
 *   0 --> No error (sock_set and sock_max are set)
 *  -1 --> Error (an error message has been logged by lwc_writeLog())
 */
int
net_server (	const char *port_number, const char *iface,
		const char *address_family,
		fd_set *sock_set, int *sock_max)
{
	int sock, try_no;
	fd_set fds;
	struct addrinfo hints;
	struct addrinfo *result, *rp;
	int ret;
	int max_sock;
#if HAVE_NANOSLEEP
	struct timespec req;

	req.tv_sec = SCHEDWI_HOSTLOOKUP_SLEEP;
	req.tv_nsec = 0;
#endif

#if HAVE_ASSERT_H
	assert (port_number != NULL);
#endif

	schedwi_memset (&hints, 0, sizeof (struct addrinfo));
	hints.ai_family = AF_UNSPEC;   /* Allow IPv4 and IPv6 */
	if (address_family != NULL) {
		if (schedwi_strcasecmp (address_family, "inet") == 0) {
			hints.ai_family = AF_INET; /* Only allow IPv4 */
		}
		else {
			if (schedwi_strcasecmp (address_family, "inet6") == 0) {
				hints.ai_family = AF_INET6; /* Only IPv6 */
			}
		}
	}
	hints.ai_socktype  = SOCK_STREAM; /* Stream socket */
	hints.ai_flags     = AI_PASSIVE;
	hints.ai_protocol  = 0;           /* Any protocol */
	hints.ai_canonname = NULL;
	hints.ai_addr      = NULL;
	hints.ai_next      = NULL;

	try_no = 0;
	do {
		ret = getaddrinfo (
			(iface == NULL || iface[0] == '\0') ? NULL : iface,
			port_number, &hints, &result);
		/*
		 * If the name server returned a temporary failure, try
		 * again.
		 */
		if (ret == EAI_AGAIN) {
			try_no++;
			if (try_no < SCHEDWI_HOSTLOOKUP_RETRIES) {
#if HAVE_NANOSLEEP
				nanosleep (&req, NULL);
#elif HAVE_SLEEP
				sleep (SCHEDWI_HOSTLOOKUP_SLEEP);
#elif HAVE_USLEEP
				usleep (SCHEDWI_HOSTLOOKUP_SLEEP * 1000000);
#endif
			}
		}
		else {
			if (ret != 0) {
				lwc_writeLog (LOG_ERR, _("getaddrinfo: %s"),
						gai_strerror (ret));
				return -1;
			}
		}
	} while (ret != 0 && try_no < SCHEDWI_HOSTLOOKUP_RETRIES);

	if (ret != 0) {
		lwc_writeLog (LOG_WARNING, _("Name server unavailable"));
		return -1;
	}

	/* Create the sockets for all the returned addresses */
	FD_ZERO (&fds);
	max_sock = -1;
	for (rp = result; rp != NULL; rp = rp->ai_next) {
		sock = net_create_sock (rp->ai_family,
					rp->ai_socktype,
					rp->ai_protocol,
					rp->ai_addr,
					rp->ai_addrlen);
		if (sock == -1) {
			/* Close all the previously opened sockets */
			while (max_sock >= 0) {
				if (FD_ISSET (max_sock, &fds) != 0) {
					closesocket (max_sock);
				}
				max_sock--;
			}
			freeaddrinfo (result);
			return -1;
		}
		FD_SET (sock, &fds);
		if (sock > max_sock) {
			max_sock = sock;
		}
	}

	freeaddrinfo (result);

	*sock_set = fds;
	*sock_max = max_sock + 1;
	return 0;
}


/*
 * From the provided sockaddr structure build a list of all the associated
 * names (dot notation and name from the DNS).
 *
 * Return:
 *   0 --> No error (or may be some DNS availability errors).  names is
 *         set and must be freed by the caller by the destroy_client_names()
 *         function.  names is a NULL terminated array.
 *  -1 --> Error.  An error message has been logged by lwc_writeLog()
 */
int
build_client_names_from_addr (	const struct sockaddr_storage *in,
				socklen_t inlen, char ***names)
{
	int try_no, ret;
	unsigned int i, j;
	char **lst;
#if HAVE_NANOSLEEP
	struct timespec req;

	req.tv_sec = SCHEDWI_HOSTLOOKUP_SLEEP;
	req.tv_nsec = 0;
#endif

#if HAVE_ASSERT_H
	assert (names != NULL);
#endif

	lst = (char **) malloc (3 * sizeof (char *));
	if (lst == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

	lst[0] = (char *) malloc (50);
	if (lst[0] == NULL) {
		free (lst);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

	/* First retrieve the host in dot notation */
	ret = getnameinfo (	(const struct sockaddr *)in, inlen,
				lst[0], 50, NULL, 0, NI_NUMERICHOST);
	if (ret != 0) {
		lwc_writeLog (LOG_WARNING, _("getnameinfo: %s"),
				gai_strerror (ret));
		free (lst[0]);
		i = 0;
	}
	else {
		lst[0][49] = '\0';
		i = 1;
	}

	/* Then, retrieve the associated host name */
	lst[i] = (char *)malloc (NI_MAXHOST);
	if (lst[i] == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		for (j = 0; j < i; j++) {
			free (lst[j]);
		}
		free (lst);
		return -1;
	}

	try_no = 0;
	do {
		ret = getnameinfo (	(const struct sockaddr *)in, inlen,
					lst[i], NI_MAXHOST, NULL, 0, 0);
		/*
		 * If the name server returned a temporary failure, try
		 * again.
		 */
		if (ret == EAI_AGAIN) {
			try_no++;
			if (try_no < SCHEDWI_HOSTLOOKUP_RETRIES) {
#if HAVE_NANOSLEEP
				nanosleep (&req, NULL);
#elif HAVE_SLEEP
				sleep (SCHEDWI_HOSTLOOKUP_SLEEP);
#elif HAVE_USLEEP
				usleep (SCHEDWI_HOSTLOOKUP_SLEEP * 1000000);
#endif
			}
		}
		else {
			if (ret != 0) {
				lwc_writeLog (LOG_ERR, _("getnameinfo: %s"),
						gai_strerror (ret));
				for (j = 0; j <= i; j++) {
					free (lst[j]);
				}
				free (lst);
				return -1;
			}
		}
	} while (ret != 0 && try_no < SCHEDWI_HOSTLOOKUP_RETRIES);

	if (ret != 0) {
		lwc_writeLog (LOG_WARNING, _("Name server unavailable"));
		for (j = 0; j <= i; j++) {
			free (lst[j]);
		}
		free (lst);
		return -1;
	}
	lst[i][NI_MAXHOST - 1] = '\0';
	lst[++i] = NULL;
	*names = lst;
	return 0;
}


/*
 * Parse an IPv6 address or name and return the hostname part in `name' and
 * the port part in `port'.
 *
 * Return:
 *   0 --> OK. `name' contains the host name part and `port' contains the port
 *         part.  Both must be freed by the caller with free()
 *  -1 --> Error. A message has been logged
 */
static int
parse_ipv6 (const char *host_name, char **name, char **port)
{
	int i;
	char *n, *s, *p;


#if HAVE_ASSERT_H
	assert (host_name != NULL && name != NULL && port != NULL);
#endif

	for (i = 0; host_name[i] != '\0' && isspace (host_name[i]) != 0; i++);
	if (host_name[i] == '\0') {
		lwc_writeLog (LOG_ERR, _("Host name is empty"));
		return -1;
	}

	/* Name part */
	if (host_name[i] == '[') {
		n = (char *) malloc (schedwi_strlen (host_name + i + 1) + 1);
		if (n == NULL) {
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
		strcpy (n, host_name + i + 1);
		s = strchr (n, ']');
		if (s == NULL) {
			free (n);
			lwc_writeLog (LOG_ERR,
			_("%s: wrong hostname syntax: missing closing `]'"),
				host_name);
			return -1;
		}
		*(s++) = '\0';
		p = strchr (s, ':');
		if (p != NULL) {
			for (p++; *p != '\0' && isspace (*p) != 0; p++);
			if (*p == '\0') {
				s = NULL;
			}
			else {
				s = p;
			}
		}
		else {
			s = NULL;
		}
	}
	else {
		n = (char *) malloc (schedwi_strlen (host_name + i) + 1);
		if (n == NULL) {
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
		strcpy (n, host_name + i);
		s = NULL;
	}

	/* Port part */
	if (s == NULL || *s == '\0') {
		/* Default port */
		s = SCHEDWI_DEFAULT_AGTPORT;
	}

	p = (char *) malloc (schedwi_strlen (s) + 1);
	if (p == NULL) {
		free (n);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}
	strcpy (p, s);
	*name = n;
	*port = p;
	return 0;
}


/*
 * Parse an IPv4 address or name and return the hostname part in `name' and
 * the port part in `port'.
 *
 * Return:
 *   0 --> OK. `name' contains the host name part and `port' contains the port
 *         part.  Both must be freed by the caller with free()
 *  -1 --> Error. A message has been logged
 */
static int
parse_ipv4 (const char *host_name, char **name, char **port)
{
	int i;
	char *n, *s, *p;


#if HAVE_ASSERT_H
	assert (host_name != NULL && name != NULL && port != NULL);
#endif

	for (i = 0; host_name[i] != '\0' && isspace (host_name[i]) != 0; i++);
	if (host_name[i] == '\0') {
		lwc_writeLog (LOG_ERR, _("Host name is empty"));
		return -1;
	}

	/* Name part */
	n = (char *) malloc (schedwi_strlen (host_name + i) + 1);
	if (n == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}
	strcpy (n, host_name + i);

	/* Port part */
	s = strchr (n, ':');
	if (s != NULL) {
		*(s++) = '\0';
		while (*s != '\0' && isspace (*s) != 0) {
			s++;
		}
	}
	if (s == NULL || *s == '\0') {
		/* Default port */
		s = SCHEDWI_DEFAULT_AGTPORT;
	}
	p = (char *) malloc (schedwi_strlen (s) + 1);
	if (p == NULL) {
		free (n);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}
	strcpy (p, s);
	*name = n;
	*port = p;
	return 0;
}


/*
 * Parse an IPv4/IPv6 address or name and return the hostname part in `name'
 * and the port part in `port'.
 *
 * Return:
 *   0 --> OK. `name' contains the host name part and `port' contains the port
 *         part.  Both must be freed by the caller with free()
 *  -1 --> Error. A message has been logged
 */
int
parse_hostname (const char *host_name, char **name, char **port)
{
	char *s;


#if HAVE_ASSERT_H
	assert (host_name != NULL && name != NULL && port != NULL);
#endif

	if (strchr (host_name, '[') != NULL) {
		return parse_ipv6 (host_name, name, port);
	}
	s = strchr (host_name, ':');
	if (s != NULL && strchr (s + 1, ':') != NULL) {
		return parse_ipv6 (host_name, name, port);
	}
	return parse_ipv4 (host_name, name, port);
}


/*
 * From the provided name build a list of all the associated
 * names (dot notation, name and aliases from the DNS). `host_name' can have
 * the following formats:
 *      host
 *      host:port
 *      [host]:port   (for IPv6)
 *
 * Return:
 *   0 --> No error (or may be some DNS availability errors).  names is
 *         set and must be freed by the caller by the destroy_client_names()
 *         function.  names is a NULL terminated array.  port is also set and
 *         must be freed by the caller with free()
 *  -1 --> Error.  An error message has been logged by lwc_writeLog()
 */
int
build_client_names_from_name (	const char *host_name, char ***names,
				char **port)
{
	int try_no, ret;
	unsigned int i, j, length;
	struct addrinfo hints;
	struct addrinfo *result, *rp;
	char **lst;
	char *n , *p;

#if HAVE_NANOSLEEP
	struct timespec req;

	req.tv_sec = SCHEDWI_HOSTLOOKUP_SLEEP;
	req.tv_nsec = 0;
#endif

#if HAVE_ASSERT_H
	assert (host_name != NULL && names != NULL && port != NULL);
#endif

	/* Split IP address and port */
	if (parse_hostname (host_name, &n, &p) != 0) {
		return -1;
	}

	schedwi_memset (&hints, 0, sizeof (struct addrinfo));
	hints.ai_family    = AF_UNSPEC;
	hints.ai_socktype  = SOCK_STREAM;
	hints.ai_flags     = AI_CANONNAME;
	hints.ai_protocol  = 0;
	hints.ai_canonname = NULL;
	hints.ai_addr      = NULL;
	hints.ai_next      = NULL;

	try_no = 0;
	do {
		ret = getaddrinfo (n, p, &hints, &result);
		/*
		 * If the name server returned a temporary failure, try
		 * again.
		 */
		if (ret == EAI_AGAIN) {
			try_no++;
			if (try_no < SCHEDWI_HOSTLOOKUP_RETRIES) {
#if HAVE_NANOSLEEP
				nanosleep (&req, NULL);
#elif HAVE_SLEEP
				sleep (SCHEDWI_HOSTLOOKUP_SLEEP);
#elif HAVE_USLEEP
				usleep (SCHEDWI_HOSTLOOKUP_SLEEP * 1000000);
#endif
			}
		}
		else {
			if (ret != 0) {
				free (n);
				free (p);
				lwc_writeLog (LOG_ERR, _("getaddrinfo: %s"),
						gai_strerror (ret));
				return -1;
			}
		}
	} while (ret != 0 && try_no < SCHEDWI_HOSTLOOKUP_RETRIES);

	if (ret != 0) {
		lwc_writeLog (LOG_WARNING, _("Name server unavailable"));
		result = NULL;
	}

	if (result == NULL) {
		/* Only add the name to the array */
		length = 2;
	}
	else {
		i = 0;
		for (rp = result; rp != NULL; rp = rp->ai_next) {
			i++;
		}
		/*
		 * Number of array elements:
		 *        number of numeric addresses
		 *      + number of host name
		 * 	+ host name from parameter + host name + NULL
		 */
		length = (2 * i) + 3;
	}
	lst = (char **) malloc (length * sizeof (char *));
	if (lst == NULL) {
		free (n);
		free (p);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		if (result != NULL) {
			freeaddrinfo (result);
		}
		return -1;
	}

	/* Add the canonical name in the array */
	i = 0;
	if (	   result != NULL
		&& result->ai_canonname != NULL
		&& (result->ai_canonname)[0] != '\0')
	{
		lst[i] = (char *)malloc (schedwi_strlen (result->ai_canonname)
						+ 1);
		if (lst[i] == NULL) {
			free (n);
			free (p);
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			for (j = 0; j < i; j++) {
				free (lst[j]);
			}
			free (lst);
			freeaddrinfo (result);
			return -1;
		}
		strcpy (lst[i++], result->ai_canonname);
	}

	/* Add the name provided as a parameter */
	lst[i++] = n;

	for (rp = result; rp != NULL; rp = rp->ai_next) {
		/* First add the numeric host value */
		lst[i] = (char *)malloc (50);
		if (lst[i] == NULL) {
			free (p);
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			for (j = 0; j < i; j++) {
				free (lst[j]);
			}
			free (lst);
			freeaddrinfo (result);
			return -1;
		}
		ret = getnameinfo (	rp->ai_addr, rp->ai_addrlen,
					lst[i], 50, NULL, 0, NI_NUMERICHOST);
		if (ret != 0) {
			lwc_writeLog (LOG_WARNING, _("getnameinfo: %s"),
					gai_strerror (ret));
			free (lst[i]);
		}
		else {
			lst[i++][49] = '\0';
		}

		/* Then, do a reverse lookup to add a host name */
		lst[i] = (char *)malloc (NI_MAXHOST);
		if (lst[i] == NULL) {
			free (p);
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			for (j = 0; j < i; j++) {
				free (lst[j]);
			}
			free (lst);
			freeaddrinfo (result);
			return -1;
		}

		try_no = 0;
		do {
			ret = getnameinfo (rp->ai_addr, rp->ai_addrlen,
					lst[i], NI_MAXHOST, NULL, 0, 0);
			/*
			 * If the name server returned a temporary failure, try
			 * again.
			 */
			if (ret == EAI_AGAIN) {
				try_no++;
				if (try_no < SCHEDWI_HOSTLOOKUP_RETRIES) {
#if HAVE_NANOSLEEP
					nanosleep (&req, NULL);
#elif HAVE_SLEEP
					sleep (SCHEDWI_HOSTLOOKUP_SLEEP);
#elif HAVE_USLEEP
					usleep (  SCHEDWI_HOSTLOOKUP_SLEEP
						* 1000000);
#endif
				}
			}
		} while (	   ret == EAI_AGAIN
				&& try_no < SCHEDWI_HOSTLOOKUP_RETRIES);

		if (ret != 0) {
			lwc_writeLog (LOG_WARNING, _("getnameinfo: %s"),
					gai_strerror (ret));
			free (lst[i]);
		}
		else {
			lst[i++][NI_MAXHOST - 1] = '\0';
		}
	}
	freeaddrinfo (result);
	lst[i] = NULL;
	*names = lst;
	*port = p;
	return 0;
}


/*
 * Free the provided array
 */
void
destroy_client_names (char **names)
{
	unsigned int i;

	if (names != NULL) {
		for (i = 0; names[i] != NULL; i++) {
			free (names[i]);
		}
		free (names);
	}
}


/*
 * Create a socket connected to a server
 * port_number is the port number to connect to (a number or a service name
 *             in /etc/services)
 *
 * Return:
 *   The socket or
 *   -1 in case of error (an error message is displayed using lwc_writeLog())
 */
int
net_client_sock (const char *port_number, const char *hostname)
{
	int sock, try_no;
	struct addrinfo hints;
	struct addrinfo *result, *rp;
	int ret;
#if HAVE_NANOSLEEP
	struct timespec req;

	req.tv_sec = SCHEDWI_HOSTLOOKUP_SLEEP;
	req.tv_nsec = 0;
#endif

#if HAVE_ASSERT_H
	assert (port_number != NULL && hostname != NULL);
#endif

	schedwi_memset (&hints, 0, sizeof (struct addrinfo));
	hints.ai_family   = AF_UNSPEC;     /* Allow IPv4 or IPv6 */
	hints.ai_socktype = SOCK_STREAM;   /* Stream socket */
	hints.ai_flags    = AI_ADDRCONFIG;
	hints.ai_protocol = 0;             /* Any protocol */

	try_no = 0;
	do {
		ret = getaddrinfo (hostname, port_number, &hints, &result);
		/*
		 * If the name server returned a temporary failure, try
		 * again.
		 */
		if (ret == EAI_AGAIN) {
			try_no++;
			if (try_no < SCHEDWI_HOSTLOOKUP_RETRIES) {
#if HAVE_NANOSLEEP
				nanosleep (&req, NULL);
#elif HAVE_SLEEP
				sleep (SCHEDWI_HOSTLOOKUP_SLEEP);
#elif HAVE_USLEEP
				usleep (SCHEDWI_HOSTLOOKUP_SLEEP * 1000000);
#endif
			}
		}
		else {
			if (ret != 0) {
				lwc_writeLog (LOG_ERR, _("getaddrinfo: %s"),
						gai_strerror (ret));
				return -1;
			}
		}
	} while (ret != 0 && try_no < SCHEDWI_HOSTLOOKUP_RETRIES);

	if (ret != 0) {
		lwc_writeLog (LOG_WARNING, _("Name server unavailable"));
		return -1;
	}

	/* Try the addresses one after the other until one is working */
	sock = -1;
	for (rp = result; rp != NULL; rp = rp->ai_next) {
		sock = socket(rp->ai_family, rp->ai_socktype, rp->ai_protocol);
		if (sock < 0) {
			continue;
		}

		if (connect (sock, rp->ai_addr, rp->ai_addrlen) == 0) {
			break;
		}
		closesocket (sock);
		sock = -1;
	}

	freeaddrinfo (result);
	if (sock < 0) {
		lwc_writeLog (LOG_ERR, _("Failed to connect to %s:%s"),
				hostname, port_number);
		return -1;
	}

	return sock;
}


/*
 * JSON callback function
 */
static int
json_char_callback (void* ctx, int type, const JSON_value* value)
{
	int *nb_open_brackets = (int *)ctx;

	switch (type) {
		case JSON_T_ARRAY_BEGIN:
		case JSON_T_OBJECT_BEGIN:
			if (*nb_open_brackets < 0) {
				*nb_open_brackets = 1;
			}
			else {
				(*nb_open_brackets)++;
			}
			break;
		case JSON_T_ARRAY_END:
		case JSON_T_OBJECT_END:
			(*nb_open_brackets)--;
			break;
	}
	return 1;
}


/*
 * Read data from the provided socket handler.
 * The data are store in *buff (of size *len). If buffer is NULL or not big
 * enough, it is reallocated to a bigger size. It must be freed by the caller
 *
 * Return:
 *    The total number of characters read and stored in *buff or
 *    -1 on error (a message is displayed using lwc_writeLog())
 *    -2 if too many data are received (a message is displayed using
 *       lwc_writeLog())
 */
ssize_t
net_read_sock (int sock, char **buff, size_t *len)
{
	ssize_t nb_read, total;
	size_t to_read;
	char *b, *tmp;
	fd_set rfds;
	struct timeval tv;
	int ret, nb_open_brackets, i;
	JSON_config config;
	struct JSON_parser_struct *jc;

#if HAVE_ASSERT_H
	assert (buff != NULL && len != NULL);
#endif

	if (*buff == NULL || *len == 0) {
		*buff = (char *) malloc (BUFFER_LEN);
		if (*buff == NULL) {
			*len = 0;
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
		*len = BUFFER_LEN;
	}

	/* JSON parser initialization */
	init_JSON_config (&config);

	nb_open_brackets              = -1;
	config.depth                  = 20;
	config.callback               = &json_char_callback;
	config.allow_comments         = 0;
	config.handle_floats_manually = 0;
	config.callback_ctx           = &nb_open_brackets;

	jc = new_JSON_parser (&config);

	total = 0;
	to_read = *len;
	b = *buff;
	do {
		do {
			FD_ZERO (&rfds);
			FD_SET (sock, &rfds);
			tv.tv_sec = 5;
			tv.tv_usec = 0;
			ret = select (sock + 1, &rfds, NULL, NULL, &tv);
		} while (ret < 0 && (errno == EINTR || errno == EAGAIN));

		if (ret < 0) {
			lwc_writeLog (LOG_ERR,
					_("Error while reading data: %s"),
					strerror (errno));
			delete_JSON_parser (jc);
			return -1;
		}

		if (ret == 0) {
			lwc_writeLog (LOG_NOTICE,
			_("Timeout while waiting to read network data"));
			delete_JSON_parser (jc);
			return -1;
		}

		do {
			nb_read = recv (sock, b, to_read, 0);
		} while (nb_read < 0 && (errno == EINTR || errno == EAGAIN));

		if (nb_read == 0) {
			delete_JSON_parser (jc);
			return total;
		}

		if (nb_read < 0) {
			lwc_writeLog (	LOG_ERR,
					_("Error while reading data: %s"),
					strerror (errno));
			delete_JSON_parser (jc);
			return -1;
		}

		total += nb_read;

		/*
		 * Count the numbers of open/close brackets.  If this number
		 * drops to zero it means that the full JSON request has
		 * been received.
		 */
		for (i = 0; i < nb_read; i++) {
			if (! JSON_parser_char (jc, b[i])) {
				lwc_writeLog (	LOG_ERR,
					_("Syntax error in received data"));
				delete_JSON_parser (jc);
				return -1;
			}
			if (nb_open_brackets == 0) {
				delete_JSON_parser (jc);
				return total;
			}
		}

		if ((size_t)nb_read < to_read) {
			to_read -= nb_read;
			b += nb_read;
		}
		else {
			if (*len + BUFFER_LEN > SCHEDWI_MAX_BUFFER_LEN) {
				/* Too many data */
				lwc_writeLog (LOG_NOTICE,
		_("Failed to read the network request: buffer overflow"));
				delete_JSON_parser (jc);
				return -2;
			}

			tmp = (char *) realloc (*buff, *len + BUFFER_LEN);
			if (tmp == NULL) {
				lwc_writeLog (	LOG_CRIT,
						_("Memory allocation error"));
				delete_JSON_parser (jc);
				return -1;
			}
			*buff = tmp;
			*len += BUFFER_LEN;
			b = *buff + total;
			to_read = BUFFER_LEN;
		}
	} while (1);

	delete_JSON_parser (jc);

	return total;
}


/*
 * Write data to the provided socket handler.
 *
 * Return:
 *   0 --> No error
 *  -1 --> Write error (a message is displayed using lwc_writeLog())
 */
int
net_write_sock (int sock, const char *buff, size_t len)
{
	ssize_t nb_write;

	if (buff == NULL || len == 0) {
		return 0;
	}

	do {
		nb_write = send (sock, buff, len, 0);
		if (nb_write == len) {
			return 0;
		}

		if (nb_write < 0) {
			if (errno != EINTR && errno != EAGAIN) {
				lwc_writeLog (LOG_ERR,
					_("Error while sending data: %s"),
					strerror (errno));
				return -1;
			}
		}
		else {
			buff += nb_write;
			len -= nb_write;
		}

	} while (1);

	return 0;
}


/*
 * Close the socket
 */
int
net_close_sock (int sock)
{
	if (sock >= 0) {
#if HAVE_SHUTDOWN
		shutdown (sock, SHUT_RDWR);
#endif
		return closesocket (sock);
	}
	return 0;
}

/*
 * Close a set of sockets
 */
void
net_close_sock_set (fd_set *sock_set, int *sock_max)
{
#if HAVE_ASSERT_H
	assert (sock_max != NULL);
#endif

	if (sock_set != NULL) {
		while (*sock_max > 0) {
			--(*sock_max);
			if (FD_ISSET (*sock_max, sock_set) != 0) {
				net_close_sock (*sock_max);
			}
		}
		FD_ZERO (sock_set);
		*sock_max = -1;
	}
}

/*------------------------======= End Of File =======------------------------*/
