/* Schedwi
   Copyright (C) 2012 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* memsend.c -- Memory buffer to store data (append allowed) */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#include <lib_functions.h>
#include <memsend.h>


/*
 * Create a new memsend object
 *
 * Return:
 *   The new object (to free with memsend_destroy) or
 *   NULL on memory allocation error
 */
memsend_t *
memsend_new ()
{
	memsend_t *r;

	r = (memsend_t *) malloc (sizeof (memsend_t));
	if (r == NULL) {
		return NULL;
	}
	r->buff = (char *) malloc (1024);
	if (r->buff == NULL) {
		free (r);
		return NULL;
	}
	r->len = 1024;
	r->idx = 0;
	return r;
}


/*
 * Free the provided memsend object
 */
void
memsend_destroy (memsend_t *r)
{
	if (r != NULL) {
		if (r->buff != NULL) {
			free (r->buff);
		}
		free (r);
	}
}


/*
 * Get the null-terminated string containing the added data
 *
 * Return:
 *   The null-terminated string
 */
const char *
memsend_getstring (memsend_t *r)
{
	if (r == NULL || r->buff == NULL) {
		return NULL;
	}
	r->buff[r->idx] = '\0';
	return r->buff;
}


/*
 * Get the length of the string
 *
 * Return:
 *   The length of the string
 */
unsigned int
memsend_getlength (memsend_t *r)
{
	if (r == NULL || r->buff == NULL) {
		return 0;
	}
	return r->idx;
}


/*
 * Reset the memory of the object. Previuosly added data are lost
 */
void
memsend_clear (memsend_t *r)
{
	if (r != NULL) {
		r->idx = 0;
	}
}


/*
 * Append a string to the object
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error
 */
int
memsend_append_len (memsend_t *r, const char *s, size_t len)
{
	char *tmp;

	if (r == NULL) {
		return 0;
	}
	if (r->idx + len >= r->len) {
		tmp = (char *) realloc (r->buff, r->len + len + 1024);
		if (tmp == NULL) {
			return -1;
		}
		r->buff = tmp;
		r->len += len + 1024;
	}
	strncpy (r->buff + r->idx, s, len);
	r->idx += len;
	return 0;
}


/*
 * Append a string to the object
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error
 */
int
memsend_append (memsend_t *r, const char *s)
{
	if (s == NULL || s[0] == '\0') {
		return 0;
	}
	return memsend_append_len(r, s, schedwi_strlen(s));
}

/*------------------------======= End Of File =======------------------------*/
