# Schedwi
# Copyright (C) 2012, 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Job/jobset selection widget."""

from muntjac.api import Panel, Tree
from muntjac.data.util.hierarchical_container import HierarchicalContainer
from muntjac.data.property import IValueChangeListener
from muntjac.ui.tree import (IExpandListener, ICollapseListener,
                             IItemStyleGenerator)
from muntjac.terminal.theme_resource import ThemeResource

from simple_queries_job import sql_get_children, sql_count_children
from path import id2path


# Memorize the last selected item to show the same containing jobset next time
_LAST_SELECTED = None


class SelectJob(Panel,
                IValueChangeListener, IExpandListener, ICollapseListener):

    """Job and jobset selection widget."""

    _ICON_JOB = ThemeResource('icons/job.png')
    _ICON_JOBSET = ThemeResource('icons/jobset.png')

    def __init__(self, jobsetids, sql_session, workload=None,
                 selected_job=None):
        """Build the job/jobset selection widget.

        @param jobsetids:
                    jobset to expand by default.  This a hierarchy list that
                    contains all the IDs from the root jobset (1).
        @type jobsetids: list
        @param sql_session:
                    SQLAlchemy session.
        @param workload:
                    workload to consider.
        @param selected_job:
                    Id of the job/jobset to select by default.
        """
        global _LAST_SELECTED
        super(SelectJob, self).__init__()
        self._sql_session = sql_session
        self._workload = workload
        self._selected = None
        self._jobsetids = jobsetids

        self.setCaption(_("Jobs/Jobsets:"))
        layout = self.getContent()
        layout.setMargin(False)
        layout.setSpacing(False)

        self._tree = Tree()
        self._tree.setImmediate(True)
        self._tree.addStyleName("selectjobtree")
        self._tree.setItemStyleGenerator(TreeStyleGenerator(self))

        jContainer = HierarchicalContainer()
        jContainer.addContainerProperty('name', str, None)
        jContainer.addContainerProperty('type', int, None)
        jContainer.addContainerProperty('icon', ThemeResource, None)
        # Root jobset
        item = jContainer.addItem(1)
        item.getItemProperty('name').setValue('/')
        item.getItemProperty('type').setValue(0)
        item.getItemProperty('icon').setValue(self._ICON_JOBSET)
        self._container = jContainer
        self._tree.setContainerDataSource(self._container)
        self._tree.setItemCaptionPropertyId('name')
        self._tree.setItemIconPropertyId('icon')
        self.addComponent(self._tree)

        # Callbacks
        self._tree.addListener(self, IValueChangeListener)
        self._tree.addListener(self, IExpandListener)
        self._tree.addListener(self, ICollapseListener)

        # Expand the tree
        if selected_job:
            p = id2path(sql_session, selected_job, workload)
            for i in p.id[:-1]:
                self._tree.expandItem(i)
            self._tree.select(selected_job)
        elif _LAST_SELECTED:
            try:
                p = id2path(sql_session, _LAST_SELECTED, workload)
            except:
                for i in jobsetids:
                    self._tree.expandItem(i)
            else:
                for i in p.id[:-1]:
                    self._tree.expandItem(i)
        else:
            for i in jobsetids:
                self._tree.expandItem(i)

    def getSelected(self):
        """Return the path of the selected job or jobset.

        @return:
                path (list of IDs) to the selected job or jobset or None if
                nothing is selected.
                [-1] is the ID of the selected job/jobset.
        """
        if not self._selected:
            return None
        i = self._selected
        ids = [i]
        while True:
            i = self._container.getParent(i)
            if i is not None:
                ids.insert(0, i)
            else:
                break
        return ids

    def valueChange(self, event):
        """Callback when a job or jobset has been selected or de-selected."""
        global _LAST_SELECTED
        self._selected = event.getProperty().getValue()
        if self._selected:
            _LAST_SELECTED = self._selected

    def nodeExpand(self, event):
        """Callback when a jobset has been expanded."""
        jobsetId = event.getItemId()
        session = self._sql_session.open_session()
        for child in sql_get_children(session, jobsetId,
                                      False, self._workload):
            child_id = child.id
            item = self._container.addItem(child_id)
            item.getItemProperty('name').setValue(child.name.encode('utf-8'))
            item.getItemProperty('type').setValue(child.type)
            if child.type == 0:
                item.getItemProperty('icon').setValue(self._ICON_JOBSET)
            else:
                item.getItemProperty('icon').setValue(self._ICON_JOB)
            self._container.setParent(child_id, jobsetId)
            # Only show the expand arrow if the jobset has children
            if child.type == 0 and sql_count_children(session, child_id,
                                                      False, self._workload):
                self._container.setChildrenAllowed(child_id, True)
            else:
                self._container.setChildrenAllowed(child_id, False)
        self._sql_session.close_session(session)

    def nodeCollapse(self, event):
        """Callback when a jobset has been collapsed."""
        jobsetId = event.getItemId()
        # Remove all the existing children first
        if self._container.hasChildren(jobsetId):
            for childId in self._container.getChildren(jobsetId):
                self._tree.collapseItem(childId)
                self._container.removeItemRecursively(childId)


class TreeStyleGenerator(IItemStyleGenerator):

    """IItemStyleGenerator object to style cells."""

    def __init__(self, c):
        self._c = c

    def getStyle(self, itemId):
        if itemId in self._c._jobsetids:
            return "notallowed"
