# Schedwi
# Copyright (C) 2012, 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Host selection widget."""

import sqlalchemy.orm.session

from muntjac.api import NativeSelect, ComboBox
from muntjac.data.property import IValueChangeListener
from muntjac.data.util.indexed_container import IndexedContainer
from muntjac.terminal.theme_resource import ThemeResource

import host_utils
import cluster_utils
from tables.hosts import hosts
from tables.job_cluster import job_cluster
from tables.job_cluster_s import job_cluster_s
from web.autorefresh import AutoRefresh

_HOST_TYPE, _CLUSTER_TYPE = 0, 1

# Memorize the last selected item to select the same host next time
_LAST_SELECTED_HOST = None
_LAST_SELECTED_HOST_CLUSTER = ()  # 0: type, 1: id


class SelectHost(NativeSelect, IValueChangeListener):

    """Host selection widget."""

    def __init__(self, sql_session, label=None, descr=None, host_id=None):
        """Build the host selection widget.

        @param sql_session:
                    SQLAlchemy session (it can be an opened session)
        @param label:
                    Selection box caption.
        @param descr:
                    Tooltip.
        @param host_id:
                    Database ID of the host to select by default.
        """
        global _LAST_SELECTED_HOST
        super(SelectHost, self).__init__()

        # Container
        c = IndexedContainer()
        c.addContainerProperty('host', str, None)
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            session = sql_session.open_session()
            session.expire_on_commit = False
        else:
            session = sql_session
        query = session.query(hosts)
        query = query.order_by(hosts.hostname)
        for h in query.all():
            item = c.addItem(h.id)
            s = '%s (SSL)' if h.sslenable else '%s'
            item.getItemProperty('host').setValue(s % str(h))

        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)

        self.setContainerDataSource(c)
        if label:
            self.setCaption(label)
        if descr:
            self.setDescription(descr)
        self.setNullSelectionAllowed(False)
        self.setItemCaptionPropertyId('host')
        if host_id is not None and c.containsId(host_id):
            self.setValue(host_id)
        elif (_LAST_SELECTED_HOST is not None and
              c.containsId(_LAST_SELECTED_HOST)):
            self.setValue(_LAST_SELECTED_HOST)
        else:
            self.setValue(c.firstItemId())
        self.addListener(self, IValueChangeListener)

    def get_selected_name(self):
        host_id = self.getValue()
        if not host_id:
            return ''
        else:
            c = self.getContainerDataSource()
            return c.getItem(host_id).getItemProperty('host').getValue()

    def valueChange(self, event):
        """Callback when host is selected from the list."""
        AutoRefresh.reset()
        global _LAST_SELECTED_HOST
        _LAST_SELECTED_HOST = self.getValue()


class SelectHostCluster(ComboBox, IValueChangeListener):

    """Host and cluster selection widget."""

    # No icon for hosts   _HOST_ICON = ThemeResource('icons/hosts.png')
    _CLUSTER_ICON = ThemeResource('icons/clusters.png')

    def __init__(self, sql_session, label=None, descr=None,
                 host_id=None, cluster_id=None):
        """Build the host/cluster selection widget.

        @param sql_session:
                    SQLAlchemy session (it can be an opened session)
        @param label:
                    Selection box caption.
        @param descr:
                    Tooltip.
        @param host_id:
                    Database ID of the host to select by default.
        @param cluster_id:
                    Database ID of the cluster to select by default.
        """
        super(SelectHostCluster, self).__init__()

        to_select = None
        last_selected = None
        # Container
        c = IndexedContainer()
        c.addContainerProperty('name', str, None)
        c.addContainerProperty('type', int, None)
        c.addContainerProperty('id', long, None)
        c.addContainerProperty('icon', ThemeResource, None)
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            session = sql_session.open_session()
            session.expire_on_commit = False
        else:
            session = sql_session
        # Clusters first...
        for cluster in cluster_utils.name2cluster_list(session, '*'):
            i = c.generateId()
            item = c.addItem(i)
            item.getItemProperty('name').setValue(cluster.name.encode('utf-8'))
            item.getItemProperty('type').setValue(_CLUSTER_TYPE)
            item.getItemProperty('id').setValue(long(cluster.id))
            item.getItemProperty('icon').setValue(self._CLUSTER_ICON)
            if cluster_id == cluster.id:
                to_select = i
            if _LAST_SELECTED_HOST_CLUSTER and \
               _LAST_SELECTED_HOST_CLUSTER[0] == _CLUSTER_TYPE and \
               _LAST_SELECTED_HOST_CLUSTER[1] == cluster.id:
                last_selected = i
        # ...and then hosts
        for h in host_utils.name2host_list(session, '*'):
            i = c.generateId()
            item = c.addItem(i)
            s = '%s (SSL)' if h.sslenable else '%s'
            item.getItemProperty('name').setValue(s % str(h))
            item.getItemProperty('type').setValue(_HOST_TYPE)
            item.getItemProperty('id').setValue(long(h.id))
            #No icon item.getItemProperty('icon').setValue(self._HOST_ICON)
            if host_id == h.id:
                to_select = i
            if _LAST_SELECTED_HOST_CLUSTER and \
               _LAST_SELECTED_HOST_CLUSTER[0] == _HOST_TYPE and \
               _LAST_SELECTED_HOST_CLUSTER[1] == h.id:
                last_selected = i
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)

        self.setContainerDataSource(c)
        if label:
            self.setCaption(label)
        if descr:
            self.setDescription(descr)
        self.setNullSelectionAllowed(False)
        self.setItemCaptionPropertyId('name')
        self.setItemIconPropertyId('icon')
        if to_select is not None:
            self.setValue(to_select)
        elif last_selected is not None:
            self.setValue(last_selected)
        else:
            self.setValue(c.firstItemId())
        self.addListener(self, IValueChangeListener)

    def get_selected_name(self):
        """Return the name of the selected item.

        @return:
                The name of the selected item (it can be a host or a cluster
                name)
        """

        item_id = self.getValue()
        if not item_id:
            return None
        else:
            c = self.getContainerDataSource()
            return c.getItem(item_id).getItemProperty('name').getValue()

    def get_selected_id(self):
        """Return the selected item.

        @return:
                The tuple (type,id) with type one of _HOST_TYPE or
                _CLUSTER_TYPE
        """
        item_id = self.getValue()
        if not item_id:
            return (None, None)
        else:
            c = self.getContainerDataSource()
            return (c.getItem(item_id).getItemProperty('type').getValue(),
                    c.getItem(item_id).getItemProperty('id').getValue())

    def get_internal_id(self, obj_to_search):
        """Return the container ID of the given host/cluster.

        @param obj_to_search:
                    the L{tables.job_cluster.job_cluster},
                    L{tables.job_host.job_host},
                    L{tables.job_cluster_s.job_cluster_s} or
                    L{tables.job_host_s.job_host_s} object to search for.
        @return:
                    the container ID of the object or None if not found.
        """
        if obj_to_search is None:
            return None
        if isinstance(obj_to_search, job_cluster) or \
           isinstance(obj_to_search, job_cluster_s):
            type_to_search = _CLUSTER_TYPE
            id_to_search = obj_to_search.cluster_id
        else:
            type_to_search = _HOST_TYPE
            id_to_search = obj_to_search.host_id
        c = self.getContainerDataSource()
        for item_id in c.getItemIds():
            item = c.getItem(item_id)
            if item.getItemProperty('type').getValue() == type_to_search and \
               item.getItemProperty('id').getValue() == id_to_search:
                return item_id
        return None

    def set_value(self, obj_to_select):
        """Select the given host/cluster in the list.

        @param obj_to_select:
                    the L{tables.job_cluster.job_cluster},
                    L{tables.job_host.job_host},
                    L{tables.job_cluster_s.job_cluster_s} or
                    L{tables.job_host_s.job_host_s} object to select.
        """
        item_id = self.get_internal_id(obj_to_select)
        if item_id is not None:
            self.setValue(item_id)

    def valueChange(self, event):
        """Callback when a host or cluster is selected from the list."""
        AutoRefresh.reset()
        global _LAST_SELECTED_HOST_CLUSTER
        item_id = event.getProperty().getValue()
        if item_id:
            c = self.getContainerDataSource()
            _LAST_SELECTED_HOST_CLUSTER = (
                c.getItem(item_id).getItemProperty('type').getValue(),
                c.getItem(item_id).getItemProperty('id').getValue())
