# Schedwi
# Copyright (C) 2012, 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Main"""

from muntjac.api import (Application, Label, Alignment, NativeButton, Window,
                         VerticalLayout, GridLayout)
from muntjac.terminal.theme_resource import ThemeResource
from muntjac.terminal.external_resource import ExternalResource
from muntjac.ui.button import IClickListener

from sql_session import SqlSession
import parse_config
import locale_utils
from web.mainwindow import MainWindow
from web.loginwindow import LoginWindow
from web.topbar import TopBar
from web.triggerwindow import TriggerManualJobWindow
from selectworkload import SelectWorkload


class Main(Application, SqlSession):

    """Main class.

    This class will also manage the three states of the application:
      - the login window
      - the activity window which is used to choose between the
        Monitor activity (to monitor the execution of workloads) and the
        Designer activity (to create/edit jobs, jobsets, hosts, environment
        groups, calendars, ...)
      - the main window (Designer or Monitor)
    """

    def __init__(self):
        """Initialize the object."""
        SqlSession.__init__(self,
                            drivername=parse_config.DBI_DRIVERNAME,
                            user=parse_config.DBI_USER,
                            password=parse_config.DBI_PASSWORD,
                            hostname=parse_config.DBI_DBHOSTNAME,
                            dbname=parse_config.DBI_DBNAME,
                            dbdir=parse_config.DBI_DBDIR)

        self._login_window = None
        self._select_activity_window = None
        self._main_window = None
        Application.__init__(self)

    def init(self):
        """Muntjac Application init."""
        self.state_login()

    def _destroy_windows(self):
        """Remove the activity and the main windows."""
        try:
            self.removeWindow(self._select_activity_window)
            self.removeWindow(self._main_window)
        except:
            pass
        self._select_activity_window = None
        self._main_window = None

    def state_login(self):
        """Transition to the login window."""
        self._destroy_windows()
        locale_utils.set_locale(self.getLocale())
        if self._login_window is None:
            self._login_window = LoginWindow(self)
            self._login_window.setName('login')
        self.setMainWindow(self._login_window)

    def state_select_activity(self):
        """Transition to the activity window."""
        self._destroy_windows()
        self._select_activity_window = SelectActivityWindow(self)
        self._select_activity_window.setName('activity')
        self.setMainWindow(self._select_activity_window)
        self._login_window.open(
            ExternalResource(self._select_activity_window.getURL()))

    def state_main(self, workload=None, workload_str=None):
        """Transition to the main window."""
        self._destroy_windows()
        self._main_window = SchedwiWindow(self, workload, workload_str)
        self._main_window.setName('main')
        self.setMainWindow(self._main_window)
        self._login_window.open(ExternalResource(self._main_window.getURL()))

    def logout(self):
        """Logout."""
        self.setUser('')
        self.state_login()

    def getWindow(self, name):
        w = super(Main, self).getWindow(name)
        if w is None and name.find('schedwi-') == 0:
            w = TriggerManualJobWindow(self, name)
            self.addWindow(w)
        return w

    def getURL(self):
        """Return the URL.
        If https is used, the Application getURL() method does not seem to
        return a proper URL (`http://' instead of `https://')
        This function overrides the parent's one.
        """
        url = super(Main, self).getURL()
        return url.replace('http://', 'https://', 1)


class SchedwiWindow(Window):

    """The main window."""

    _TITLE = 'Schedwi'

    def __init__(self, app, workload=None, workload_str=None):
        """Initialize the window.

        @param app:
                    the L{Main} object.
        @param workload:
                    the workload to condider.
        """
        super(SchedwiWindow, self).__init__()
        self._app = app
        self._workload = workload
        self._workload_str = workload_str

        # Main top/expanded-bottom layout
        mainExpand = VerticalLayout()
        mainExpand.setSizeFull()
        self.setContent(mainExpand)
        self.setSizeFull()
        self.setCaption(self._TITLE)
        self.setTheme('schedwi')

        nav = TopBar(self, app, app, workload, workload_str, True)
        mainExpand.addComponent(nav)

        # Main jobset tree view
        self._mainSplit = MainWindow(app, app, workload)
        mainExpand.addComponent(self._mainSplit)
        mainExpand.setExpandRatio(self._mainSplit, 1.0)

    def refresh(self):
        self._mainSplit.refresh()


class SelectActivityWindow(Window):

    """The Activity window."""

    _TITLE = 'Schedwi'

    def __init__(self, app):
        """Initialize the Activity window.

        @param app:
                    the L{Main} object.
        """
        super(SelectActivityWindow, self).__init__()
        self._app = app

        mainExpand = VerticalLayout()
        mainExpand.setSizeFull()
        self.setContent(mainExpand)
        self.setSizeFull()
        self.setCaption(self._TITLE)
        self.setTheme('schedwi')

        nav = TopBar(self, app, app, None, None, False)
        mainExpand.addComponent(nav)

        g = GridLayout(2, 4)
        g.setWidth('600pt')
        g.setSpacing(True)
        g.setMargin(True)
        mainExpand.addComponent(g)
        mainExpand.setComponentAlignment(g, Alignment.TOP_CENTER)

        b = NativeButton(_("Monitor"))
        b.setWidth('100%')
        b.setIcon(ThemeResource('icons/monitor.png'))
        b.setDescription(_("Monitor running workloads"))
        b.setStyleName('btbig')
        b.addListener(ButtonMonitor(self), IClickListener)
        g.addComponent(b, 0, 0, 0, 1)
        self._monitor_bt = b

        l = Label(_("Select a date:"))
        g.addComponent(l, 1, 0)
        g.setComponentAlignment(l, Alignment.BOTTOM_LEFT)
        self._date_select = l

        self._wl = SelectWorkload(app)
        g.addComponent(self._wl, 1, 1)
        g.setComponentAlignment(self._wl, Alignment.TOP_LEFT)
        g.addComponent(self._wl.get_refresher(), 1, 2)
        if self._wl.get_selected_workload() is None:
            self._monitor_bt.setEnabled(False)
            self._date_select.setEnabled(False)
            self._wl.setEnabled(False)

        b = NativeButton(_("Designer"))
        b.setWidth('100%')
        b.setIcon(ThemeResource('icons/designer.png'))
        b.setDescription(_("Create and edit jobs and jobsets, hosts, ..."))
        b.setStyleName('btbig')
        b.addListener(ButtonDesigner(self), IClickListener)
        g.addComponent(b, 0, 2, 0, 3)

    def get_workload(self):
        """Get the selected workload."""
        return self._wl.get_selected_workload()

    def get_workload_str(self):
        """Get the selected workload date as a textual date."""
        return self._wl.get_selected_workload_str()


class ButtonMonitor(IClickListener):

    """Callback class for the Monitor button."""

    def __init__(self, select_activity_window_obj):
        super(ButtonMonitor, self).__init__()
        self._c = select_activity_window_obj

    def buttonClick(self, event):
        wl = self._c.get_workload()
        wl_str = self._c.get_workload_str()
        self._c._app.state_main(wl, wl_str)


class ButtonDesigner(IClickListener):

    """Callback class for the Designer button."""

    def __init__(self, select_activity_window_obj):
        super(ButtonDesigner, self).__init__()
        self._c = select_activity_window_obj

    def buttonClick(self, event):
        self._c._app.state_main()
