# Schedwi
# Copyright (C) 2012, 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Hosts window."""


from muntjac.api import Window, Button, Alignment, HorizontalLayout, TabSheet
from muntjac.ui.button import IClickListener

from hoststab import HostsTab
from clusterstab import ClustersTab


class HostWindow(Window, IClickListener):

    """Host window."""

    def __init__(self, refresh_obj, main_window,
                 sql_session, workload=None,
                 host_id_to_select=None, cluster_id_to_select=None):
        """Build the Host window.

        @param refresh_obj:
                    the object to use to refresh the view when the host has
                    been changed.
        @param main_window:
                    the L{web.main.SchedwiWindow} object (the main window)
        @param sql_session:
                    SQLAlchemy session.
        @param workload:
                    workload to consider.
        @param host_id_to_select:
                    database ID of the host to select by default.
        @param cluster_id_to_select:
                    database ID of the cluster to select by default.
        """
        super(HostWindow, self).__init__()
        self._main_window = main_window

        # Layout
        self.setCaption(_('Hosts'))
        self.setHeight('494px')
        self.setWidth('800px')

        v = self.getContent()
        v.setSizeFull()
        v.setMargin(False)
        v.setSpacing(False)

        ts = TabSheet()
        ts.setSizeFull()
        v.addComponent(ts)
        v.setExpandRatio(ts, 1.0)

        h_bt = HorizontalLayout()
        h_bt.setMargin(True)
        h_bt.setSpacing(True)
        v.addComponent(h_bt)
        v.setComponentAlignment(h_bt, Alignment.BOTTOM_RIGHT)

        ok = Button(_('Close'))
        ok.addListener(self, IClickListener)
        h_bt.addComponent(ok)

        # Host tab
        t_hosts = HostsTab(self, refresh_obj, main_window,
                           sql_session, workload, host_id_to_select)

        # Cluster tab
        t_clusters = ClustersTab(self, refresh_obj, main_window,
                                 sql_session, workload, cluster_id_to_select)

        # Add the components to the tab sheet
        ts.addTab(t_hosts, _('Hosts'))
        ts.addTab(t_clusters, _('Clusters'))

        if host_id_to_select is not None:
            ts.setSelectedTab(t_hosts)
        elif cluster_id_to_select is not None:
            ts.setSelectedTab(t_clusters)

        # Show the window
        main_window.addWindow(self)
        self.center()

    def buttonClick(self, event):
        self._main_window.removeWindow(self)
