# Schedwi
# Copyright (C) 2012, 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Hosts window."""

import socket
import os
import subprocess

from muntjac.api import (Window, HorizontalSplitPanel, Button, Alignment,
                         VerticalLayout, HorizontalLayout, ListSelect,
                         TextField, TextArea, Table, Label, GridLayout,
                         CheckBox, TabSheet)
from muntjac.data.validators.abstract_validator import AbstractValidator
from muntjac.data.property import IValueChangeListener
from muntjac.ui.button import IClickListener
from muntjac.ui.window import Notification
from muntjac.ui.themes import BaseTheme
from muntjac.ui.abstract_text_field import TextChangeEventMode
from muntjac.event.field_events import ITextChangeListener
from muntjac.terminal.theme_resource import ThemeResource
from muntjac.terminal.sizeable import ISizeable
from muntjac.data.util.indexed_container import IndexedContainer

from web.boxes import DeleteBox, BoxCloseListener
from web.selectenvwidget import SelectEnvGroup
from web.register import get_pending, is_approval_possible, register
import environments_utils
import cmd_hosts.whatuses
import cmd_hosts.cert
import host_utils
import config
from tables.host_environment import host_environment
from tables.host_environment_s import host_environment_s
from tables.hosts import hosts


class HostsTab(HorizontalSplitPanel):

    """Host tab."""

    _schedwiping_prog = config.BINDIR + "/schedwiping"

    def __init__(self, host_window_obj, refresh_obj, main_window,
                 sql_session, workload=None, host_id_to_select=None):
        """Build the Host window.

        @param host_window_obj:
                    the parent L{web.hostwindow.HostWindow} window object.
        @param refresh_obj:
                    the object to use to refresh the view when the host has
                    been changed.
        @param main_window:
                    the L{web.main.SchedwiWindow} object (the main window)
        @param sql_session:
                    SQLAlchemy session.
        @param workload:
                    workload to consider.
        @param host_id_to_select:
                    database ID of the host to select by default.
        """
        super(HostsTab, self).__init__()
        self._host_window_obj = host_window_obj
        self._refresh_obj = refresh_obj
        self._main_window = main_window
        self._sql_session = sql_session
        self._workload = workload

        # Layout
        self.setSplitPosition(220, ISizeable.UNITS_PIXELS, False)
        self.setSizeFull()

        vert_left = VerticalLayout()
        vert_left.setSizeFull()
        vert_left.setMargin(True)
        self.addComponent(vert_left)

        host_list = ListSelect(_("Hosts:"))
        host_list.setSizeFull()
        host_list.setNullSelectionAllowed(False)
        host_list.addListener(HostSelected(self), IValueChangeListener)
        host_list.setImmediate(True)
        vert_left.addComponent(host_list)
        vert_left.setExpandRatio(host_list, 1.0)
        self._host_list = host_list

        if workload is None:
            bt_box_group = HorizontalLayout()
            bt_box_group.setWidth('100%')
            bt_box_group.setMargin(False)
            vert_left.addComponent(bt_box_group)

            edit_group = Button()
            edit_group.setIcon(ThemeResource('icons/edit14.png'))
            edit_group.setStyleName('small')
            edit_group.setDescription(_("Edit the selected host"))
            edit_group.setEnabled(False)
            edit_group.addListener(HostEdit(self), IClickListener)
            bt_box_group.addComponent(edit_group)
            self._edit_group = edit_group

            add_group = Button()
            add_group.setIcon(ThemeResource('icons/add14.png'))
            add_group.setStyleName('small')
            add_group.setDescription(_("Declare a host"))
            add_group.addListener(HostAdd(self), IClickListener)
            bt_box_group.addComponent(add_group)

            del_group = Button()
            del_group.setIcon(ThemeResource('icons/delete14.png'))
            del_group.setStyleName('small')
            del_group.setDescription(_("Delete the selected host"))
            del_group.setEnabled(False)
            del_group.addListener(HostDel(self), IClickListener)
            bt_box_group.addComponent(del_group)
            self._del_group = del_group

            where_group = Button()
            where_group.setIcon(ThemeResource('icons/search14.png'))
            where_group.setStyleName('small')
            where_group.setDescription(_("Host references"))
            where_group.setEnabled(False)
            where_group.addListener(HostWhere(self), IClickListener)
            bt_box_group.addComponent(where_group)
            self._where_group = where_group

            lst_pending_approvals = get_pending()
            if is_approval_possible():
                approve_group = Button(_("Approve"))
                approve_group.setWidth('100%')
                approve_group.setStyleName('small')
                approve_group.setDescription(
                    _("Approve pending host requests"))
                approve_group.addListener(HostApprove(self), IClickListener)
                vert_left.addComponent(approve_group)
                if not lst_pending_approvals:
                    approve_group.setEnabled(False)
                self._approve_group = approve_group

        vert_right = VerticalLayout()
        vert_right.setSizeFull()
        vert_right.setMargin(True)
        vert_right.setSpacing(True)
        self.addComponent(vert_right)

        name = Label()
        name.setStyleName('envname')
        name.setContentMode(Label.CONTENT_TEXT)
        vert_right.addComponent(name)
        self._name = name

        description = Label()
        description.setContentMode(Label.CONTENT_TEXT)
        vert_right.addComponent(description)
        self._descr = description

        g = GridLayout(2, 6)
        g.setSpacing(True)
        g.setStyleName('cert')
        self._grid = g

        l = Label(_("TCP Port:"))
        g.addComponent(l, 0, 0)
        g.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        l = Label()
        self._port = l
        g.addComponent(l, 1, 0)
        g.setComponentAlignment(l, Alignment.MIDDLE_LEFT)

        l = Label(_("SSL:"))
        g.addComponent(l, 0, 1)
        g.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        l = Label()
        self._ssl = l
        g.addComponent(l, 1, 1)
        g.setComponentAlignment(l, Alignment.MIDDLE_LEFT)

        l = Label()
        g.addComponent(l, 0, 2)
        g.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        self._cert_subject_label = l
        l = Label()
        self._cert_subject = l
        g.addComponent(l, 1, 2)
        g.setComponentAlignment(l, Alignment.MIDDLE_LEFT)

        l = Label()
        g.addComponent(l, 0, 3)
        g.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        self._cert_alt_names_label = l
        l = Label()
        self._cert_alt_names = l
        g.addComponent(l, 1, 3)
        g.setComponentAlignment(l, Alignment.MIDDLE_LEFT)

        l = Label()
        g.addComponent(l, 0, 4)
        g.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        self._cert_activation_label = l
        l = Label()
        self._cert_activation = l
        g.addComponent(l, 1, 4)
        g.setComponentAlignment(l, Alignment.MIDDLE_LEFT)

        l = Label()
        g.addComponent(l, 0, 5)
        g.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        self._cert_expiration_label = l
        l = Label()
        self._cert_expiration = l
        g.addComponent(l, 1, 5)
        g.setComponentAlignment(l, Alignment.MIDDLE_LEFT)

        vert_right.addComponent(g)

        variables = HorizontalLayout()
        variables.setSizeFull()
        vert_right.addComponent(variables)
        vert_right.setExpandRatio(variables, 1.0)

        env_list = ListSelect(_("Environment groups:"))
        env_list.setSizeFull()
        env_list.setNullSelectionAllowed(False)
        env_list.addListener(EnvSelected(self), IValueChangeListener)
        env_list.setImmediate(True)
        variables.addComponent(env_list)
        variables.setExpandRatio(env_list, 1.0)
        self._env_list = env_list

        if workload is None:
            env_bt = VerticalLayout()
            env_bt.setSpacing(True)
            env_bt.setMargin(False)
            env_bt.setWidth('20px')
            variables.addComponent(env_bt)
            variables.setComponentAlignment(env_bt, Alignment.MIDDLE_CENTER)

            b = Button()
            b.setStyleName(BaseTheme.BUTTON_LINK)
            b.setIcon(ThemeResource('icons/add14.png'))
            b.setDescription(_("Add an environment group"))
            b.addListener(EnvAdd(self), IClickListener)
            env_bt.addComponent(b)
            self._btadd = b

            b = Button()
            b.setStyleName(BaseTheme.BUTTON_LINK)
            b.setIcon(ThemeResource('icons/delete14.png'))
            b.setDescription(_("Remove the selected environment group"))
            b.addListener(EnvDel(self), IClickListener)
            env_bt.addComponent(b)
            self._btdel = b

            b = Button()
            b.setStyleName(BaseTheme.BUTTON_LINK)
            b.setIcon(ThemeResource('icons/go-top14.png'))
            b.setDescription(_("Move to the top"))
            b.addListener(EnvTop(self), IClickListener)
            env_bt.addComponent(b)
            self._bttop = b

            b = Button()
            b.setStyleName(BaseTheme.BUTTON_LINK)
            b.setIcon(ThemeResource('icons/go-up14.png'))
            b.setDescription(_("Raise the environment group"))
            b.addListener(EnvUp(self), IClickListener)
            env_bt.addComponent(b)
            self._btup = b

            b = Button()
            b.setStyleName(BaseTheme.BUTTON_LINK)
            b.setIcon(ThemeResource('icons/go-down14.png'))
            b.setDescription(_("Lower the environment group"))
            b.addListener(EnvDown(self), IClickListener)
            env_bt.addComponent(b)
            self._btdown = b

            b = Button()
            b.setStyleName(BaseTheme.BUTTON_LINK)
            b.setIcon(ThemeResource('icons/go-bottom14.png'))
            b.setDescription(_("Move to the bottom"))
            b.addListener(EnvBottom(self), IClickListener)
            env_bt.addComponent(b)
            self._btbottom = b

        b = Button(_('Check connection'))
        b.setStyleName('small')
        b.setDescription(_("Check if the agent can be accessed"))
        b.addListener(HostPing(self), IClickListener)
        self._btping = b
        if os.access(self._schedwiping_prog, os.X_OK):
            vert_right.addComponent(b)
            vert_right.setComponentAlignment(b, Alignment.MIDDLE_RIGHT)

        # Fill the environment group list
        self._set_host_container()

        # Initialize the variable table
        self._set_env_container()
        self.disable_button_env()

        if host_id_to_select is not None:
            self._host_list.setValue(host_id_to_select)

        if workload is None and lst_pending_approvals:
            n = Notification(_("Pending registration requests"),
                             _("<br/>Use the `Approve' button bellow."))
            n.setDelayMsec(10000)
            self._host_window_obj.showNotification(n)

    def refresh_main(self):
        """Refresh the parent view."""
        self._refresh_obj.refresh()

    def _set_env_container(self, selected_host_id=None):
        """Build and set the container for the environment group list.

        @param selected_host_id:
                        the database ID of the host for which the environment
                        group must be displayed in the list.
                        If None (default), an empty container is used.
        """
        c = IndexedContainer()
        c.addContainerProperty('name', str, None)
        if selected_host_id is not None:
            session = self._sql_session.open_session()
            if self._workload is None:
                query = session.query(host_environment)
                query = query.filter_by(host_id=selected_host_id)
                query = query.order_by(host_environment.position)
            else:
                query = session.query(host_environment_s)
                query = query.filter_by(host_id=selected_host_id)
                query = query.filter(host_environment_s.workload_date ==
                                     self._workload)
                query = query.order_by(host_environment_s.position)
            for he in query.all():
                e = environments_utils.get_env(session, he.env_id,
                                               self._workload)
                item = c.addItem(he.env_id)
                item.getItemProperty('name').setValue(e.name.encode('utf-8'))
            self._sql_session.close_session(session)
        self._env_list.setContainerDataSource(c)
        self._env_list.setItemCaptionPropertyId('name')

    def _set_host_container(self):
        """Build and set the container for the host list."""
        c = IndexedContainer()
        c.addContainerProperty('name', str, None)
        c.addContainerProperty('hostname', str, None)
        c.addContainerProperty('port', str, None)
        c.addContainerProperty('ssl', int, None)
        c.addContainerProperty('sslcert', str, None)
        c.addContainerProperty('description', str, None)
        for h in host_utils.name2host_list(self._sql_session, '*'):
            item = c.addItem(h.id)
            item.getItemProperty('hostname').setValue(
                h.hostname.encode('utf-8'))
            item.getItemProperty('description').setValue(
                h.description.encode('utf-8'))
            item.getItemProperty('port').setValue(h.portnum.encode('utf-8'))
            item.getItemProperty('ssl').setValue(h.sslenable)
            if h.sslcert:
                item.getItemProperty('sslcert').setValue(
                    h.sslcert.encode('ascii'))
            else:
                item.getItemProperty('sslcert').setValue('')
            item.getItemProperty('name').setValue(str(h))
        self._host_list.setContainerDataSource(c)
        self._host_list.setItemCaptionPropertyId('name')

    def get_selected_host(self):
        """Return the currently selected host ID.

        @return:    the database ID of the selected host.
        """
        return self._host_list.getValue()

    def refresh_host_list(self, host_id_to_select=None):
        """Rebuild the host list.

        @param host_id_to_select:
                        database ID of the host to select after
                        the refresh.  If None (default), the previously
                        selected host will be re-selected.
        """
        # Save the selection so we can restore it
        if host_id_to_select is None:
            host_id_to_select = self.get_selected_host()
        self._set_host_container()
        self._host_list.setValue(host_id_to_select)

    def set_state_button_host(self):
        """Enable or disable (greyed out) action buttons whether a
        host is selected or not.
        """
        if self._workload is not None:
            return
        selected_item_id = self.get_selected_host()
        if selected_item_id is None:
            self._edit_group.setEnabled(False)
            self._del_group.setEnabled(False)
            self._where_group.setEnabled(False)
            self.disable_button_env()
        else:
            self._del_group.setEnabled(True)
            self._edit_group.setEnabled(True)
            self._where_group.setEnabled(True)
            self.enable_button_env()

    def propagate_select_host(self):
        """Display the selected host details (name, description, port, ssl,
        ssl certificate details and environment groups) in the right pane of
        the window.
        """
        selected_item_id = self.get_selected_host()
        if selected_item_id is None:
            self._name.setValue('')
            self._descr.setValue('')
            self._descr.removeStyleName('envdescription')
            self._port.setValue('')
            self._ssl.setValue('')
            self._cert_subject_label.setValue('')
            self._cert_subject.setValue('')
            self._cert_alt_names_label.setValue('')
            self._cert_alt_names.setValue('')
            self._cert_activation_label.setValue('')
            self._cert_activation.setValue('')
            self._cert_expiration_label.setValue('')
            self._cert_expiration.setValue('')
        else:
            c = self._host_list.getContainerDataSource()
            item = c.getItem(selected_item_id)
            self._name.setValue(item.getItemProperty('hostname').getValue())
            d = item.getItemProperty('description').getValue().strip()
            self._descr.setValue(d)
            if d:
                self._descr.addStyleName('envdescription')
            else:
                self._descr.removeStyleName('envdescription')
            self._port.setValue(item.getItemProperty('port').getValue())
            self._ssl.setValue(_('Yes')
                               if item.getItemProperty('ssl').getValue()
                               else _('No'))
            crt = item.getItemProperty('sslcert').getValue()
            if not crt:
                self._cert_subject_label.setValue('')
                self._cert_subject.setValue('')
                self._cert_subject.removeStyleName('certerror')
                self._cert_alt_names_label.setValue('')
                self._cert_alt_names.setValue('')
                self._cert_activation_label.setValue('')
                self._cert_activation.setValue('')
                self._cert_activation.removeStyleName('certerror')
                self._cert_expiration_label.setValue('')
                self._cert_expiration.setValue('')
                self._cert_expiration.removeStyleName('certerror')
            else:
                try:
                    certX509 = cmd_hosts.cert.schedwiX509(crt)
                except:
                    self._cert_subject_label.setValue(_("Certificate:"))
                    self._cert_subject.setValue(_('Invalid certificate'))
                    self._cert_subject.addStyleName('certerror')
                    self._cert_alt_names_label.setValue('')
                    self._cert_alt_names.setValue('')
                    self._cert_activation_label.setValue('')
                    self._cert_activation.setValue('')
                    self._cert_activation.removeStyleName('certerror')
                    self._cert_expiration_label.setValue('')
                    self._cert_expiration.setValue('')
                    self._cert_expiration.removeStyleName('certerror')
                else:
                    self._cert_subject_label.setValue(_("Certificate:"))
                    self._cert_subject.setValue(certX509.get_subject())
                    self._cert_subject.removeStyleName('certerror')
                    dns = certX509.get_dns()
                    if dns:
                        self._cert_alt_names_label.setValue(
                            _("Cert alt names:"))
                        self._cert_alt_names.setValue(dns)
                    else:
                        self._cert_alt_names_label.setValue('')
                        self._cert_alt_names.setValue('')
                    self._cert_activation_label.setValue(_("Cert activation:"))
                    self._cert_activation.setValue(certX509.get_not_before())
                    if certX509.not_yet_active():
                        self._cert_activation.addStyleName('certerror')
                    else:
                        self._cert_activation.removeStyleName('certerror')
                    self._cert_expiration_label.setValue(_("Cert expiration:"))
                    self._cert_expiration.setValue(certX509.get_not_after())
                    if certX509.has_expired():
                        self._cert_expiration.addStyleName('certerror')
                    else:
                        self._cert_expiration.removeStyleName('certerror')
        self._set_env_container(selected_item_id)

    def get_item_details_host(self, host_id=None):
        """Get the specified environment group details.

        @param host_id:
                    the database ID of the host.
                    If None (default), the details of the selected host
                    are retrieved.
        @return:    a dictionnary containing the host details or None if
                    host_id is None and and no host is selected.
        """

        if host_id is None:
            host_id = self.get_selected_host()
            if host_id is None:
                return None
        c = self._host_list.getContainerDataSource()
        item = c.getItem(host_id)
        return {
            'name':   item.getItemProperty('name').getValue(),
            'hostname': item.getItemProperty('hostname').getValue(),
            'port': item.getItemProperty('port').getValue(),
            'ssl': item.getItemProperty('ssl').getValue(),
            'sslcert': item.getItemProperty('sslcert').getValue(),
            'description': item.getItemProperty('description').getValue(),
            'id':    host_id}

    def get_previous_host_id(self, host_id):
        """Return the database ID of the previous host in the list.

        @param host_id:
                    the database ID of the host.
        @return:    the ID of the previous host in the list.
        """
        c = self._host_list.getContainerDataSource()
        i = c.prevItemId(host_id)
        return i if i is not None else c.nextItemId(host_id)

    def get_selected_env(self):
        """Return the currently selected environment group.

        @return:    the database ID of the selected environment group.
        """
        return self._env_list.getValue()

    def get_datasource_env(self):
        """Return the container data source of the environment group list.

        @return:        the container data source associated with the
                        environment group list.
        """
        return self._env_list.getContainerDataSource()

    def select_env_group(self, env_id):
        """Select the specified environment group ID in the list.

        @param env_id:  database ID of the environment group to select.
        """
        self._env_list.select(env_id)

    def disable_button_env(self):
        """Disable (greyed out) the environment list action buttons."""
        if self._workload is not None:
            return
        self._btadd.setEnabled(False)
        self._btdel.setEnabled(False)
        self._bttop.setEnabled(False)
        self._btup.setEnabled(False)
        self._btdown.setEnabled(False)
        self._btbottom.setEnabled(False)
        self._btping.setEnabled(False)

    def enable_button_env(self):
        """Enable the environment list action buttons."""
        self._btadd.setEnabled(True)
        self._btping.setEnabled(True)
        self.set_state_button_env()

    def set_state_button_env(self):
        """Enable or disable (greyed out) action buttons whether an
        environment group is selected or not.
        """
        if self._workload is not None:
            return
        selected_item_id = self.get_selected_env()
        if selected_item_id is None:
            self._btdel.setEnabled(False)
            self._bttop.setEnabled(False)
            self._btup.setEnabled(False)
            self._btdown.setEnabled(False)
            self._btbottom.setEnabled(False)
        else:
            self._btdel.setEnabled(True)
            c = self.get_datasource_env()
            if c.isFirstId(selected_item_id):
                self._bttop.setEnabled(False)
                self._btup.setEnabled(False)
            else:
                self._bttop.setEnabled(True)
                self._btup.setEnabled(True)
            if c.isLastId(selected_item_id):
                self._btdown.setEnabled(False)
                self._btbottom.setEnabled(False)
            else:
                self._btdown.setEnabled(True)
                self._btbottom.setEnabled(True)

    def write_database_env(self):
        """Write in the database the environment groups associated with the
        host.
        """
        host_id = self.get_selected_host()
        if host_id is not None:
            # Build the list of environment groups from the list
            new_list = list()
            c = self.get_datasource_env()
            for idx in range(c.size()):
                if self._workload is None:
                    e = host_environment(c.getIdByIndex(idx), idx + 1)
                else:
                    e = host_environment_s(c.getIdByIndex(idx), idx + 1,
                                           self._workload)
                new_list.append(e)
            # Retrieve the host from the database and update the
            # environment group list
            session = self._sql_session.open_session()
            try:
                h = host_utils.get_host_by_id(session, host_id)
            except:
                self._sql_session.close_session(session)
                self._main_window.showNotification(
                    _("Cannot get the selected host details"),
                    _("<br/>Maybe someone else just removed it."),
                    Notification.TYPE_ERROR_MESSAGE)
                self.refresh_host_list()
                return
            h.host_environment = new_list
            self._sql_session.close_session(session)
            self.set_state_button_env()


class HostSelected(IValueChangeListener):

    """Callback for when a host is selected in the list."""

    def __init__(self, hosts_tab_obj):
        """Initialize the callback.

        @param hosts_tab_obj:
                    the associated L{HostsTab} object.
        """
        super(HostSelected, self).__init__()
        self._c = hosts_tab_obj

    def valueChange(self, event):
        # Enable/disable action buttons
        self._c.set_state_button_host()
        # Display the selected host details
        self._c.propagate_select_host()


class HostEdit(IClickListener):

    """Callback for the Edit host button."""

    def __init__(self, hosts_tab_obj):
        """Initialize the callback.

        @param hosts_tab_obj:
                    the associated L{HostsTab} object.
        """
        super(HostEdit, self).__init__()
        self._c = hosts_tab_obj

    def buttonClick(self, event):
        host_id = self._c.get_selected_host()
        if host_id is not None:
            NewHost(self._c, host_id)


class HostAdd(IClickListener):

    """Callback for the Define new host button."""

    def __init__(self, hosts_tab_obj):
        """Initialize the callback.

        @param hosts_tab_obj:
                    the associated L{HostsTab} object.
        """
        super(HostAdd, self).__init__()
        self._c = hosts_tab_obj

    def buttonClick(self, event):
        NewHost(self._c)


class HostApprove(IClickListener):

    """Callback for the Approve Pending Host Requests button."""

    def __init__(self, hosts_tab_obj):
        """Initialize the callback.

        @param hosts_tab_obj:
                    the associated L{HostsTab} object.
        """
        super(HostApprove, self).__init__()
        self._c = hosts_tab_obj

    def buttonClick(self, event):
        ApproveHost(self._c)


class DeleteConfirmed(BoxCloseListener):

    """Delete confirmation callback."""

    def __init__(self, hosts_tab_obj, host_id):
        """Initialize the object.

        @param hosts_tab_obj
                    the associated L{HostsTab} object.
        @param host_id:
                    database ID of the host to remove.
        """
        super(DeleteConfirmed, self).__init__()
        self._c = hosts_tab_obj
        self._host_id = host_id

    def boxClose(self, event):
        """Called when the user confirmes the deletion."""
        session = self._c._sql_session.open_session()
        try:
            h = host_utils.get_host_by_id(session, self._host_id)
        except:
            pass
        else:
            session.delete(h)
        self._c._sql_session.close_session(session)
        self._c.refresh_host_list(self._c.get_previous_host_id(self._host_id))


class HostDel(IClickListener):

    """Callback for the Remove host button."""

    def __init__(self, hosts_tab_obj):
        """Initialize the callback.

        @param hosts_tab_obj:
                    the associated L{HostsTab} object.
        """
        super(HostDel, self).__init__()
        self._c = hosts_tab_obj

    def buttonClick(self, event):
        host_id = self._c.get_selected_host()
        if host_id is not None:
            session = self._c._sql_session.open_session()
            try:
                h = host_utils.get_host_by_id(session, host_id)
            except:
                self._c._sql_session.close_session(session)
                self._c.refresh_host_list()
                return
            if cmd_hosts.whatuses.is_used(session, h):
                name = h.hostname.encode('utf-8')
                self._c._sql_session.close_session(session)
                self._c._main_window.showNotification(
                    _("This host is still in used"),
                    _("<br/>%s is in used somewhere else and cannot be \
                       deleted from the database.") % str(name),
                    Notification.TYPE_ERROR_MESSAGE)
            else:
                self._c._sql_session.close_session(session)
                d = DeleteBox(self._c._main_window,
                              _("Delete"),
                              _('Are you sure you want to delete this host?'))
                d.addListener(DeleteConfirmed(self._c, host_id))


class HostWhere(IClickListener):

    """Callback for the Where used button."""

    def __init__(self, hosts_tab_obj):
        """Initialize the callback.

        @param hosts_tab_obj:
                    the associated L{HostsTab} object.
        """
        super(HostWhere, self).__init__()
        self._c = hosts_tab_obj

    def buttonClick(self, event):
        host_id = self._c.get_selected_host()
        if host_id is not None:
            session = self._c._sql_session.open_session()
            try:
                h = host_utils.get_host_by_id(session, host_id)
            except:
                self._c._sql_session.close_session(session)
                self._c._main_window.showNotification(
                    _("Cannot get the selected host details"),
                    _("<br/>Maybe someone else just removed it."),
                    Notification.TYPE_ERROR_MESSAGE)
                self._c.refresh_host_list()
                return
            # The items (job, jobset, file constraint) that are using this
            # host are stored in a list.  Later on, they will be displayed
            # in a popup (WhereWindow).
            self._gathered = list()
            cmd_hosts.whatuses.print_whatuses(session, h, self._gather, self)
            self._c._sql_session.close_session(session)
            if not self._gathered:
                self._c._main_window.showNotification(
                    _("Not used"),
                    _("<br/>%s is not used.") %
                    str(h.hostname.encode('utf-8')))
            else:
                WhereWindow(self._c, self._gathered)

    @staticmethod
    def _gather(self, title, val):
        """Callback method for the L{cmd_hosts.whatuses.print_whatuses}
        function.  This method is used to collect the items that are using
        a host.
        """
        self._gathered.append([title, val])


class HostnameValidator(AbstractValidator):

    """Check that the provided hostname or IP is valid and can be resolved."""

    def __init__(self):
        super(HostnameValidator, self).__init__(_("Cannot resolve"))

    def isValid(self, value):
        """Tests if the given value is a valid IP or a hostname that can be
        resolved.

        None values are always accepted. Values that are not strings are
        converted using L{__str__}.

        @param value:
                 the value to check
        @return: True if the value is a valid IP adress or a valid hostname,
                 False otherwise
        """
        if value is None:
            return True
        if not isinstance(value, str):
            value = str(value)
        try:
            socket.getaddrinfo(value, None)
        except socket.gaierror as (error, message):
            if error != socket.EAI_AGAIN:
                return False
        return True


class TCPPortValidator(AbstractValidator):

    """Check that the provided TCP service is valid and can be resolved."""

    def __init__(self):
        super(TCPPortValidator, self).__init__(
            _("Invalid number or unknown service"))

    def isValid(self, value):
        """Tests if the given value is a valid TCP port number or a service
        name that can be resolved.

        None values are always accepted. Values that are not strings are
        converted using L{__str__}.

        @param value:
                 the value to check
        @return: true if the value is a valid port number or service name,
                 false otherwise
        """
        if value is None:
            return True
        if not isinstance(value, str):
            value = str(value)
        try:
            socket.getaddrinfo(None, value)
        except socket.gaierror as (error, message):
            if error != socket.EAI_AGAIN:
                return False
        return True


class ApproveHost(IClickListener):

    """Approve pending host requests window."""

    _bt_captions = [_("Cancel"), _("Approve")]

    def __init__(self, hosts_tab_obj):
        """Create and display the Approve window.

        @param hosts_tab_obj:
                    the associated L{HostsTab} object.
        """
        super(ApproveHost, self).__init__()

        self._c = hosts_tab_obj

        self._w = Window(_("Approve New Hosts"))
        self._w.setWidth("315px")
        self._w.setHeight("315px")

        # VerticalLayout as content by default
        v = self._w.getContent()
        v.setSizeFull()
        v.setMargin(True)
        v.setSpacing(True)

        # List of pending requests
        l = ListSelect(_("Pending hosts requests:"))
        l.setSizeFull()
        l.setNullSelectionAllowed(True)
        l.setMultiSelect(True)
        v.addComponent(l)
        v.setExpandRatio(l, 1.0)
        self._list = l

        # Populate the list
        for h in get_pending():
            l.addItem(h.encode('utf-8'))

        # Bottom buttons
        h_bt = HorizontalLayout()
        h_bt.setMargin(False)
        h_bt.setSpacing(True)

        for caption in self._bt_captions:
            b = Button(_(caption))
            b.addListener(self, IClickListener)
            h_bt.addComponent(b)

        v.addComponent(h_bt)
        v.setComponentAlignment(h_bt, Alignment.BOTTOM_RIGHT)

        self._c._main_window.addWindow(self._w)
        self._w.center()

    def buttonClick(self, event):
        # First button is Cancel
        if event.getButton().getCaption() != _(self._bt_captions[0]):
            err_list = list()
            for h in self._list.getValue():
                msg = register(h)
                if msg:
                    err_list.append(msg)
            if err_list:
                self._c._main_window.showNotification(
                    _("Some registration failed<br/>"),
                    '<br/>'.join(err_list),
                    Notification.TYPE_ERROR_MESSAGE)
                # Refresh the list
                self._list.removeAllItems()
                for h in get_pending():
                    self._list.addItem(h.encode('utf-8'))
                # Refresh the main list because maybe some registrations
                # succeeded
                self._c.refresh_host_list()
                return
        # Disable the Approve button if there is no more pending
        # requests
        if not get_pending():
            self._c._approve_group.setEnabled(False)
        # Close the window
        self._c._main_window.removeWindow(self._w)
        # Refresh the main view
        self._c.refresh_host_list()


class NewHost(IClickListener, ITextChangeListener):

    """Edit/Declare window for a host."""

    _bt_captions = [_('Cancel'), _('OK')]

    def __init__(self, hosts_tab_obj, host_id=None):
        """Create and display the Edit/Declare window.

        @param hosts_tab_obj:
                    the associated L{HostsTab} object.
        @param host_id:
                    when the window is used to edit an existing host, this is
                    the database host ID.
        """
        super(NewHost, self).__init__()

        self._c = hosts_tab_obj
        self._host_id = host_id
        self._certificate_ok = True

        self._certfile = None

        if host_id is None:
            title = _('New host')
            host_obj = None
        else:
            title = _('Edit host')
            try:
                host_obj = host_utils.get_host_by_id(self._c._sql_session,
                                                     host_id)
            except:
                self._c._main_window.showNotification(
                    _("Cannot get the selected host details"),
                    _("<br/>Maybe someone else just removed it."),
                    Notification.TYPE_ERROR_MESSAGE)
                self._c.refresh_host_list()
                return
        self._w = Window(title)
        self._w.setWidth("520px")
        self._w.setHeight("520px")

        # VerticalLayout as content by default
        v = self._w.getContent()
        v.setSizeFull()
        v.setMargin(False)
        v.setSpacing(False)

        ts = TabSheet()
        ts.setSizeFull()
        v.addComponent(ts)
        v.setExpandRatio(ts, 1.0)
        self._ts = ts

        t1 = VerticalLayout()
        t1.setSizeFull()
        t1.setSpacing(True)
        t1.setMargin(True)
        ts.addTab(t1, _('General'))
        self._t1 = t1

        t2 = VerticalLayout()
        t2.setSizeFull()
        t2.setSpacing(True)
        t2.setMargin(True)
        ts.addTab(t2, _('SSL'))
        self._t2 = t2

        # General
        t = TextField(_('Host name:'))
        t.setWidth('100%')
        t.setDescription(_('Host name or IP address'))
        t.addValidator(HostnameValidator())
        t.setImmediate(True)
        if host_obj:
            t.setValue(host_obj.hostname.encode('utf-8'))
        self._hostname = t
        t1.addComponent(t)

        t = TextField(_('TCP port:'))
        t.setWidth('100%')
        t.setDescription(_('Network port name or number'))
        t.addValidator(TCPPortValidator())
        t.setImmediate(True)
        if host_obj:
            if isinstance(host_obj.portnum, int):
                t.setValue(str(host_obj.portnum))
            else:
                t.setValue(host_obj.portnum.encode('utf-8'))
        else:
            t.setValue('2006')
        self._port = t
        t1.addComponent(t)

        t = TextArea(_('Description:'))
        t.setSizeFull()
        t.setDescription(_('Free text as a comment or description'))
        if host_obj:
            t.setValue(host_obj.description.encode('utf-8'))
        self._descr = t
        t1.addComponent(t)
        t1.setExpandRatio(t, 1.0)

        # SSL
        t = CheckBox(_('Use SSL'))
        t.setWidth('100%')
        t.setDescription(_('Whether SSL must be used to encrypt network'))
        if host_obj:
            t.setValue(bool(host_obj.sslenable))
        self._ssl = t
        t2.addComponent(t)

        t = TextArea(_('Custom SSL Certificate:'))
        t.setSizeFull()
        t.setDescription(_('Paste the agent custom SSL certificate here. \
                Leave blank if the certificate for the agent has been signed \
                by the schedwi server authority as the regular registration \
                process.'))
        t.setStyleName('sslcert')
        t.setImmediate(True)
        if host_obj:
            t.setValue(host_obj.sslcert.encode('utf-8'))
        t.setTextChangeEventMode(TextChangeEventMode.LAZY)
        t.setTextChangeTimeout(200)
        t.addListener(self, ITextChangeListener)
        self._sslcert = t
        t2.addComponent(t)
        t2.setExpandRatio(t, 1.0)

        g = GridLayout(2, 4)
        g.setSpacing(True)
        g.setStyleName('cert')
        t2.addComponent(g)
        t2.setComponentAlignment(g, Alignment.MIDDLE_CENTER)

        self._label_cert = Label()
        g.addComponent(self._label_cert, 0, 0, 1, 0)
        g.setComponentAlignment(self._label_cert, Alignment.MIDDLE_CENTER)

        l = Label(_("Cert alt names:"))
        g.addComponent(l, 0, 1)
        g.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        self._label_alt = Label()
        g.addComponent(self._label_alt, 1, 1)
        g.setComponentAlignment(self._label_alt, Alignment.MIDDLE_LEFT)

        l = Label(_("Cert activation:"))
        g.addComponent(l, 0, 2)
        g.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        self._label_activ = Label()
        g.addComponent(self._label_activ, 1, 2)
        g.setComponentAlignment(self._label_activ, Alignment.MIDDLE_LEFT)

        l = Label(_("Cert expiration:"))
        g.addComponent(l, 0, 3)
        g.setComponentAlignment(l, Alignment.MIDDLE_RIGHT)
        self._label_expir = Label()
        g.addComponent(self._label_expir, 1, 3)
        g.setComponentAlignment(self._label_expir, Alignment.MIDDLE_LEFT)
        self.set_cert_details()

        # Bottom buttons
        h_bt = HorizontalLayout()
        h_bt.setMargin(True)
        h_bt.setSpacing(True)

        for caption in self._bt_captions:
            b = Button(_(caption))
            b.addListener(self, IClickListener)
            h_bt.addComponent(b)

        v.addComponent(h_bt)
        v.setComponentAlignment(h_bt, Alignment.BOTTOM_RIGHT)

        self._c._main_window.addWindow(self._w)
        self._w.center()

    def set_cert_details(self, val=None):
        """Display the provided text certificate details.

        The associated labels are updated in the form with the certificate
        details. In case of errror (wrong certificate format or expired
        certificate), the associated labels are displayed in red.

        @param val:
                the text certificate in PEM format. If None, it is retrieved
                from the form.
        """
        if val:
            crt = val
        else:
            crt = self._sslcert.getValue()
        if not crt:
            self._label_cert.setValue('')
            self._label_cert.removeStyleName('certerror')
            self._label_alt.setValue('-')
            self._label_activ.setValue('-')
            self._label_activ.removeStyleName('certerror')
            self._label_expir.setValue('-')
            self._label_expir.removeStyleName('certerror')
            self._certificate_ok = True
        else:
            try:
                certX509 = cmd_hosts.cert.schedwiX509(crt)
            except:
                self._label_cert.setValue(_('Invalid certificate'))
                self._label_cert.addStyleName('certerror')
                self._label_alt.setValue('-')
                self._label_activ.setValue('-')
                self._label_activ.removeStyleName('certerror')
                self._label_expir.setValue('-')
                self._label_expir.removeStyleName('certerror')
                self._certificate_ok = False
            else:
                self._certificate_ok = True
                self._label_cert.setValue(certX509.get_subject())
                self._label_cert.removeStyleName('certerror')
                dns = certX509.get_dns()
                if dns:
                    self._label_alt.setValue(dns)
                else:
                    self._label_alt.setValue('-')
                self._label_activ.setValue(certX509.get_not_before())
                if certX509.not_yet_active():
                    self._label_activ.addStyleName('certerror')
                    self._certificate_ok = False
                else:
                    self._label_activ.removeStyleName('certerror')
                self._label_expir.setValue(certX509.get_not_after())
                if certX509.has_expired():
                    self._label_expir.addStyleName('certerror')
                    self._certificate_ok = False
                else:
                    self._label_expir.removeStyleName('certerror')

    def textChange(self, event):
        """Callback for when the certificate is updated by the user in the
        form.
        """
        self.set_cert_details(event.getText())

    def buttonClick(self, event):
        # First button is Cancel
        if event.getButton().getCaption() != _(self._bt_captions[0]):
            # Retrieve the values from the form
            hostname = self._hostname.getValue().strip()
            if not hostname:
                self._c._main_window.showNotification(
                    _("Host name is empty"),
                    _("<br/>The host name cannot \
                       be empty or contain just spaces."),
                    Notification.TYPE_ERROR_MESSAGE)
                self._ts.setSelectedTab(self._t1)
                self._hostname.focus()
                return
            port = self._port.getValue().strip()
            if not port:
                port = '2006'
            descr = self._descr.getValue().strip()
            ssl = self._ssl.getValue()
            sslcert = self._sslcert.getValue().strip()
            if ssl and sslcert:
                if not self._certificate_ok:
                    self._c._main_window.showNotification(
                        _("SSL is set but the agent certificate is invalid"),
                        _("<br/>The provided certificate is not readable, has \
                            expired or is not yet valid."),
                        Notification.TYPE_ERROR_MESSAGE)
                    self._ts.setSelectedTab(self._t2)
                    self._sslcert.focus()
                    return
                certX509 = cmd_hosts.cert.schedwiX509(sslcert)
                if not certX509.matches_hostname(hostname):
                    self._c._main_window.showNotification(
                        _("SSL is set but the agent hostname does not match \
                           the certificate"),
                        _("<br/>None of the hostnames (CN and alt names) in \
                           the certificate match the hostname."),
                        Notification.TYPE_ERROR_MESSAGE)
                    self._ts.setSelectedTab(self._t2)
                    self._sslcert.focus()
                    return

            # Try to retrieve the host from the database
            if ':' in hostname:
                # IPv6 addresses must be between square brackets
                full_hostname = '[%s]:%s' % (hostname, port)
            else:
                full_hostname = '%s:%s' % (hostname, port)
            session = self._c._sql_session.open_session()
            try:
                h_lst = host_utils.name2host_list(session, full_hostname)
            except:
                # Not found.
                h_lst = None
            if h_lst:
                # Already in the database
                # If it was an Add Window, print an error message.
                if self._host_id is None or self._host_id != h_lst[0].id:
                    self._c._sql_session.close_session(session)
                    self._c._main_window.showNotification(
                        _("Host already defined"),
                        _("<br/>The host %s is already \
                        defined in the database.") % full_hostname,
                        Notification.TYPE_ERROR_MESSAGE)
                    self._ts.setSelectedTab(self._t1)
                    self._hostname.focus()
                    return

            # Add the new host in the database
            if self._host_id is None:
                h = hosts(hostname, port, int(ssl), sslcert, descr)
                session.add(h)

            # Edit the host
            else:
                if not h_lst:
                    # Retrieve the old host
                    try:
                        h = host_utils.get_host_by_id(session, self._host_id)
                    except:
                        # Humm, it should have been in the database...
                        # So recreating the host
                        h = hosts(hostname, port, int(ssl), sslcert,
                                  descr)
                        session.add(h)
                else:
                    h = h_lst[0]
                h.hostname = hostname.decode('utf-8')
                h.portnum = port.decode('utf-8')
                h.sslenable = int(ssl)
                h.sslcert = sslcert.decode('utf-8')
                h.description = descr.decode('utf-8')
            self._c._sql_session.close_session(session)
            self._c.refresh_host_list(h.id)

        # Close the window
        self._c._main_window.removeWindow(self._w)
        # Refresh the main view
        self._c.refresh_main()


class WhereWindow(IClickListener):

    """Window to display the host associations."""

    def __init__(self, hosts_tab_obj, obj_list):
        """Create and display the window.

        @param hosts_tab_obj:
                    the associated L{HostsTab} object.
        @param obj_list:
                    list of objects.  Each object is an array of two
                    strings.  The first one is the type of object (job,
                    jobset, constraint file) and the second one is its name.
        """
        super(WhereWindow, self).__init__()

        self._c = hosts_tab_obj

        self._w = Window(_("Host references"))
        self._w.setWidth("320px")
        self._w.setHeight("320px")

        # VerticalLayout as content by default
        v = self._w.getContent()
        v.setSizeFull()
        v.setMargin(True)
        v.setSpacing(True)

        t = Table()
        t.setSizeFull()
        t.setColumnReorderingAllowed(False)
        t.setColumnCollapsingAllowed(False)
        t.setSortDisabled(False)
        t.setSelectable(False)
        t.setMultiSelect(False)
        c = IndexedContainer()
        c.addContainerProperty('type', str, None)
        c.addContainerProperty('value', str, None)
        for k, val in obj_list:
            val = val.encode('utf-8')
            item = c.addItem(c.generateId())
            item.getItemProperty('type').setValue(k)
            item.getItemProperty('value').setValue(val)
        t.setContainerDataSource(c)
        t.setColumnExpandRatio('value', 1.0)
        v.addComponent(t)
        v.setExpandRatio(t, 1.0)

        # Close button
        ok = Button(_('Close'))
        ok.addListener(self, IClickListener)
        v.addComponent(ok)
        v.setComponentAlignment(ok, Alignment.BOTTOM_RIGHT)

        self._c._main_window.addWindow(self._w)
        self._w.center()

    def buttonClick(self, event):
        self._c._main_window.removeWindow(self._w)


class EnvSelected(IValueChangeListener):

    """Callback for when an environment group is selected in the list."""

    def __init__(self, hosts_tab_obj):
        """Initialize the callback.

        @param hosts_tab_obj:
                    the associated L{HostsTab} object.
        """
        super(EnvSelected, self).__init__()
        self._c = hosts_tab_obj

    def valueChange(self, event):
        # Enable/disable action buttons
        self._c.set_state_button_env()


class EnvAdd(IClickListener):

    """Callback for the Add new environment button."""

    def __init__(self, hosts_tab_obj):
        """Initialize the callback.

        @param hosts_tab_obj:
                    the associated L{HostsTab} object.
        """
        super(EnvAdd, self).__init__()
        self._c = hosts_tab_obj

    def buttonClick(self, event):
        AddEnvGroup(self._c)


class DeleteEnvConfirmed(BoxCloseListener):

    """Remove an environment group confirmation callback."""

    def __init__(self, hosts_tab_obj, env_id):
        """Initialize the object.

        @param hosts_tab_obj
                    the associated L{HostsTab} object.
        @param env_id:
                    database ID of the environment group to remove.
        """
        super(DeleteEnvConfirmed, self).__init__()
        self._c = hosts_tab_obj
        self._env_id = env_id

    def boxClose(self, event):
        """Called when the user confirms the deletion."""
        c = self._c.get_datasource_env()
        item_id_to_select = c.prevItemId(self._env_id)
        if item_id_to_select is None:
            item_id_to_select = c.nextItemId(self._env_id)
        c.removeItem(self._env_id)
        self._c.select_env_group(item_id_to_select)
        self._c.write_database_env()


class EnvDel(IClickListener):

    """Callback for the Remove environment group button."""

    def __init__(self, hosts_tab_obj):
        """Initialize the callback.

        @param hosts_tab_obj:
                    the associated L{HostsTab} object.
        """
        super(EnvDel, self).__init__()
        self._c = hosts_tab_obj

    def buttonClick(self, event):
        d = DeleteBox(self._c._main_window,
                      _("Delete"),
                      _('Are you sure you want to remove this \
                         environment group from the host?'))
        d.addListener(DeleteEnvConfirmed(self._c, self._c.get_selected_env()))


class EnvTop(IClickListener):

    """Callback for the Move environment group to top button."""

    def __init__(self, hosts_tab_obj):
        """Initialize the callback.

        @param hosts_tab_obj:
                    the associated L{HostsTab} object.
        """
        super(EnvTop, self).__init__()
        self._c = hosts_tab_obj

    def buttonClick(self, event):
        env_group_id = self._c.get_selected_env()
        if env_group_id is not None:
            c = self._c.get_datasource_env()
            if not c.isFirstId(env_group_id):
                item = c.getItem(env_group_id)
                name = item.getItemProperty('name').getValue()
                c.removeItem(env_group_id)
                new_item = c.addItemAt(0, env_group_id)
                new_item.getItemProperty('name').setValue(name)
                self._c.select_env_group(env_group_id)
                self._c.write_database_env()


class EnvUp(IClickListener):

    """Callback for the Move environment group up button."""

    def __init__(self, hosts_tab_obj):
        """Initialize the callback.

        @param hosts_tab_obj:
                    the associated L{HostsTab} object.
        """
        super(EnvUp, self).__init__()
        self._c = hosts_tab_obj

    def buttonClick(self, event):
        env_group_id = self._c.get_selected_env()
        if env_group_id is not None:
            c = self._c.get_datasource_env()
            prev_item_id = c.prevItemId(env_group_id)
            if prev_item_id is not None:
                item = c.getItem(env_group_id)
                name = item.getItemProperty('name').getValue()
                c.removeItem(env_group_id)
                idx = c.indexOfId(prev_item_id)
                new_item = c.addItemAt(idx, env_group_id)
                new_item.getItemProperty('name').setValue(name)
                self._c.select_env_group(env_group_id)
                self._c.write_database_env()


class EnvDown(IClickListener):

    """Callback for the Move environment group down button."""

    def __init__(self, hosts_tab_obj):
        """Initialize the callback.

        @param hosts_tab_obj:
                    the associated L{HostsTab} object.
        """
        super(EnvDown, self).__init__()
        self._c = hosts_tab_obj

    def buttonClick(self, event):
        env_group_id = self._c.get_selected_env()
        if env_group_id is not None:
            c = self._c.get_datasource_env()
            next_item_id = c.nextItemId(env_group_id)
            if next_item_id is not None:
                item = c.getItem(env_group_id)
                name = item.getItemProperty('name').getValue()
                c.removeItem(env_group_id)
                idx = c.indexOfId(next_item_id)
                new_item = c.addItemAt(idx + 1, env_group_id)
                new_item.getItemProperty('name').setValue(name)
                self._c.select_env_group(env_group_id)
                self._c.write_database_env()


class EnvBottom(IClickListener):

    """Callback for the Move environment group to bottom button."""

    def __init__(self, hosts_tab_obj):
        """Initialize the callback.

        @param hosts_tab_obj:
                    the associated L{HostsTab} object.
        """
        super(EnvBottom, self).__init__()
        self._c = hosts_tab_obj

    def buttonClick(self, event):
        env_group_id = self._c.get_selected_env()
        if env_group_id is not None:
            c = self._c.get_datasource_env()
            if not c.isLastId(env_group_id):
                item = c.getItem(env_group_id)
                name = item.getItemProperty('name').getValue()
                c.removeItem(env_group_id)
                new_item = c.addItem(env_group_id)
                new_item.getItemProperty('name').setValue(name)
                self._c.select_env_group(env_group_id)
                self._c.write_database_env()


class HostPing(IClickListener):

    """Callback for the Check connection button."""

    def __init__(self, hosts_tab_obj):
        """Initialize the callback.

        @param hosts_tab_obj:
                    the associated L{HostsTab} object.
        """
        super(HostPing, self).__init__()
        self._c = hosts_tab_obj

    def buttonClick(self, event):
        details = self._c.get_item_details_host()
        if details:
            arg = '[' + details['hostname'] + ']:' + details['port']
            p = subprocess.Popen([self._c._schedwiping_prog, arg],
                                 stderr=subprocess.STDOUT,
                                 stdout=subprocess.PIPE)
            ret = p.wait()
            msg = p.stdout.read()
            if ret:
                self._c._main_window.showNotification(
                    _("Connection failed"),
                    "<br/>" + msg,
                    Notification.TYPE_ERROR_MESSAGE)
            else:
                self._c._main_window.showNotification(_("Success"),
                                                      "<br/>" + msg)


class AddEnvGroup(IClickListener):

    """Display the add an environment group popup."""

    _bt_captions = [_('Cancel'), _('OK')]

    def __init__(self, hosts_tab_obj):
        """Create the popup.

        @param hosts_tab_obj:
                the associated L{HostsTab} object.
        """
        super(AddEnvGroup, self).__init__()

        self._c = hosts_tab_obj

        self._w = Window('Add Environment group')
        self._w.setWidth("300px")

        # VerticalLayout as content by default
        v = self._w.getContent()
        v.setSizeFull()
        v.setMargin(True)
        v.setSpacing(True)

        t = SelectEnvGroup(hosts_tab_obj._sql_session,
                           _('Environment group:'),
                           _('Select an environment group'),
                           workload=hosts_tab_obj._workload)
        t.setWidth('100%')
        self._name = t
        v.addComponent(t)
        v.setExpandRatio(t, 1.0)

        # Button box
        h_bt = HorizontalLayout()
        h_bt.setMargin(False)
        h_bt.setSpacing(True)

        for caption in self._bt_captions:
            b = Button(_(caption))
            b.addListener(self, IClickListener)
            h_bt.addComponent(b)

        v.addComponent(h_bt)
        v.setComponentAlignment(h_bt, Alignment.BOTTOM_RIGHT)

        self._c._main_window.addWindow(self._w)
        self._w.center()

    def buttonClick(self, event):
        # First button is Cancel
        if event.getButton().getCaption() != _(self._bt_captions[0]):
            # Retrieve the selected item
            item_id, name = self._name.getSelected()
            c = self._c.get_datasource_env()
            if c.containsId(item_id):
                self._c._main_window.showNotification(
                    _("Environment group already in host"),
                    _("<br/>The selected environment group is already \
                       associated with this host."),
                    Notification.TYPE_ERROR_MESSAGE)
                return
            item = c.addItem(item_id)
            # try clause to get around a strange exception in
            # indexed_container.py, in firePropertyValueChange:
            # self._singlePropertyValueChangeListeners.get(source._propertyId)
            # AttributeError: 'NoneType' object has no attribute 'get'
            try:
                item.getItemProperty('name').setValue(name)
            except:
                pass
            self._c.select_env_group(item_id)
        # Close the window
        self._c._main_window.removeWindow(self._w)
        self._c.write_database_env()
