# Schedwi
# Copyright (C) 2012, 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Manage environment groups and variables window."""

import re

from muntjac.api import (Window, HorizontalSplitPanel, Button, Alignment,
                         VerticalLayout, HorizontalLayout, ListSelect,
                         TextField, TextArea, Table, Label, GridLayout)
from muntjac.ui.button import IClickListener
from muntjac.data.property import IValueChangeListener
from muntjac.ui.window import Notification
from muntjac.ui.themes import BaseTheme
from muntjac.terminal.theme_resource import ThemeResource
from muntjac.terminal.sizeable import ISizeable
from muntjac.data.util.indexed_container import IndexedContainer
from muntjac.event.shortcut_action import KeyCode
from muntjac.data.validators.regexp_validator import RegexpValidator

from web.boxes import DeleteBox, BoxCloseListener
from web.cutcopypastemenu import GenericContextMenu
import environments_utils
import cmd_env.whatuses
from tables.environments import environments
from tables.environments_s import environments_s
from tables.environment_var import environment_var
from tables.environment_var_s import environment_var_s


class EnvWindow(Window, IClickListener):

    """Environment group window."""

    def __init__(self, main_window, sql_session, workload=None,
                 env_group_id_to_select=None):
        """Build the environment group window.

        @param main_window:
                    the L{web.main.SchedwiWindow} object (the main window)
        @param sql_session:
                    SQLAlchemy session.
        @param workload:
                    workload to consider.
        @param env_group_id_to_select:
                    database ID of the environment group to select by default.
        """
        super(EnvWindow, self).__init__()

        self._main_window = main_window
        self._sql_session = sql_session
        self._workload = workload

        # Layout
        self.setCaption(_('Environments'))
        self.setHeight('450px')
        self.setWidth('600px')

        v = self.getContent()
        v.setSizeFull()
        v.setMargin(False)
        v.setSpacing(False)

        horiz = HorizontalSplitPanel()
        horiz.setSplitPosition(160, ISizeable.UNITS_PIXELS, False)
        horiz.setSizeFull()
        v.addComponent(horiz)
        v.setExpandRatio(horiz, 1.0)

        h = HorizontalLayout()
        h.setWidth('100%')
        h.setMargin(True)
        h.setSpacing(False)
        h.setStyleName('popupbtbox')
        v.addComponent(h)

        ok = Button(_('Close'))
        ok.addListener(self, IClickListener)
        h.addComponent(ok)
        h.setComponentAlignment(ok, Alignment.BOTTOM_RIGHT)

        vert_left = VerticalLayout()
        vert_left.setSizeFull()
        vert_left.setMargin(True)
        horiz.addComponent(vert_left)

        list_group = ListSelect(_("Environment groups:"))
        list_group.setSizeFull()
        list_group.setNullSelectionAllowed(False)
        list_group.addListener(EnvGroupSelected(self), IValueChangeListener)
        list_group.setImmediate(True)
        vert_left.addComponent(list_group)
        vert_left.setExpandRatio(list_group, 1.0)
        self._list_group = list_group

        bt_box_group = GridLayout(2, 2)
        bt_box_group.setMargin(False)
        bt_box_group.setSpacing(False)
        vert_left.addComponent(bt_box_group)
        vert_left.setComponentAlignment(bt_box_group, Alignment.MIDDLE_CENTER)

        edit_group = Button()
        edit_group.setIcon(ThemeResource('icons/edit14.png'))
        edit_group.setStyleName('small')
        edit_group.setDescription(_("Edit the selected environment group"))
        edit_group.setEnabled(False)
        edit_group.addListener(EnvGroupEdit(self), IClickListener)
        bt_box_group.addComponent(edit_group, 0, 0)
        self._edit_group = edit_group

        add_group = Button()
        add_group.setIcon(ThemeResource('icons/add14.png'))
        add_group.setStyleName('small')
        add_group.setDescription(_("Add a new environment group"))
        add_group.addListener(EnvGroupAdd(self), IClickListener)
        bt_box_group.addComponent(add_group, 1, 0)

        del_group = Button()
        del_group.setIcon(ThemeResource('icons/delete14.png'))
        del_group.setStyleName('small')
        del_group.setDescription(_("Delete the selected environment group"))
        del_group.setEnabled(False)
        del_group.addListener(EnvGroupDel(self), IClickListener)
        bt_box_group.addComponent(del_group, 0, 1)
        self._del_group = del_group

        where_group = Button()
        where_group.setIcon(ThemeResource('icons/search14.png'))
        where_group.setStyleName('small')
        where_group.setDescription(_("Environment group references"))
        where_group.setEnabled(False)
        where_group.addListener(EnvGroupWhere(self), IClickListener)
        bt_box_group.addComponent(where_group, 1, 1)
        self._where_group = where_group

        vert_right = VerticalLayout()
        vert_right.setSizeFull()
        vert_right.setMargin(True)
        vert_right.setSpacing(True)
        horiz.addComponent(vert_right)

        name = Label()
        name.setStyleName('envname')
        name.setContentMode(Label.CONTENT_TEXT)
        vert_right.addComponent(name)
        self._name = name

        description = Label()
        description.setContentMode(Label.CONTENT_TEXT)
        vert_right.addComponent(description)
        self._descr = description

        variables = HorizontalLayout()
        variables.setSizeFull()
        vert_right.addComponent(variables)
        vert_right.setExpandRatio(variables, 1.0)

        var_table = Table(_("Variables:"))
        var_table.setSizeFull()
        var_table.setColumnReorderingAllowed(False)
        var_table.setColumnCollapsingAllowed(False)
        var_table.setSortDisabled(True)
        var_table.setSelectable(True)
        var_table.setMultiSelect(False)
        var_table.addActionHandler(VarMenu(self))
        var_table.setImmediate(True)
        var_table.addListener(VarSelected(self), IValueChangeListener)
        variables.addComponent(var_table)
        variables.setExpandRatio(var_table, 1.0)
        self._var_table = var_table

        var_bt = VerticalLayout()
        var_bt.setSpacing(True)
        var_bt.setMargin(False)
        var_bt.setWidth('20px')
        variables.addComponent(var_bt)
        variables.setComponentAlignment(var_bt, Alignment.MIDDLE_CENTER)

        b = Button()
        b.setStyleName(BaseTheme.BUTTON_LINK)
        b.setIcon(ThemeResource('icons/edit14.png'))
        b.setDescription(_("Edit the variable"))
        b.addListener(VarEdit(self), IClickListener)
        var_bt.addComponent(b)
        self._btedit = b

        b = Button()
        b.setStyleName(BaseTheme.BUTTON_LINK)
        b.setIcon(ThemeResource('icons/add14.png'))
        b.setDescription(_("Add a variable"))
        b.addListener(VarAdd(self), IClickListener)
        var_bt.addComponent(b)
        self._btadd = b

        b = Button()
        b.setStyleName(BaseTheme.BUTTON_LINK)
        b.setIcon(ThemeResource('icons/delete14.png'))
        b.setDescription(_("Remove the selected variable"))
        b.addListener(VarDel(self), IClickListener)
        var_bt.addComponent(b)
        self._btdel = b

        b = Button()
        b.setStyleName(BaseTheme.BUTTON_LINK)
        b.setIcon(ThemeResource('icons/go-top14.png'))
        b.setDescription(_("Move to the top"))
        b.addListener(VarTop(self), IClickListener)
        var_bt.addComponent(b)
        self._bttop = b

        b = Button()
        b.setStyleName(BaseTheme.BUTTON_LINK)
        b.setIcon(ThemeResource('icons/go-up14.png'))
        b.setDescription(_("Raise the variable"))
        b.addListener(VarUp(self), IClickListener)
        var_bt.addComponent(b)
        self._btup = b

        b = Button()
        b.setStyleName(BaseTheme.BUTTON_LINK)
        b.setIcon(ThemeResource('icons/go-down14.png'))
        b.setDescription(_("Lower the variable"))
        b.addListener(VarDown(self), IClickListener)
        var_bt.addComponent(b)
        self._btdown = b

        b = Button()
        b.setStyleName(BaseTheme.BUTTON_LINK)
        b.setIcon(ThemeResource('icons/go-bottom14.png'))
        b.setDescription(_("Move to the bottom"))
        b.addListener(VarBottom(self), IClickListener)
        var_bt.addComponent(b)
        self._btbottom = b

        # Fill the environment group list
        self._set_env_group_container()

        # Initialize the variable table
        self._set_variable_container()
        self.disable_button_var()

        # Show the window
        main_window.addWindow(self)
        self.center()

        if env_group_id_to_select is not None:
            self._list_group.setValue(env_group_id_to_select)

    def _set_variable_container(self, selected_env_group_id=None):
        """Build and set the container for variables list table.

        @param selected_env_group_id:
                        the database ID of the environment group for which
                        the variables must be displayed in the table.
                        If None (default), an empty container is used.
        """
        c = IndexedContainer()
        c.addContainerProperty('env_key', str, None)
        c.addContainerProperty('env_value', str, None)
        if selected_env_group_id is not None:
            session = self._sql_session.open_session()
            try:
                e = environments_utils.get_env(session, selected_env_group_id,
                                               self._workload)
            except:
                pass
            else:
                for v in e.environment_var:
                    item = c.addItem(c.generateId())
                    item.getItemProperty('env_key').setValue(
                        v.env_key.encode('utf-8'))
                    item.getItemProperty('env_value').setValue(
                        v.env_value.encode('utf-8'))
            self._sql_session.close_session(session)
        self._var_table.setContainerDataSource(c)
        self._var_table.setVisibleColumns(['env_key', 'env_value'])
        self._var_table.setColumnHeaders([_('Name'), _('Value')])

    def _set_env_group_container(self):
        """Build and set the container for the environment group list."""
        c = IndexedContainer()
        c.addContainerProperty('env', str, None)
        c.addContainerProperty('descr', str, None)
        for e in environments_utils.name2env_list(self._sql_session, '*',
                                                  self._workload):
            item = c.addItem(e.id)
            item.getItemProperty('env').setValue(e.name.encode('utf-8'))
            item.getItemProperty('descr').setValue(
                e.description.encode('utf-8'))
        self._list_group.setContainerDataSource(c)
        self._list_group.setItemCaptionPropertyId('env')

    def get_selected_env_group(self):
        """Return the currently selected environment group ID.

        @return:    the database ID of the selected environment group.
        """
        return self._list_group.getValue()

    def refresh_env_group(self, env_group_id_to_select=None):
        """Rebuild the environment group list.

        @param env_group_id_to_select:
                        database ID of the environment group to select after
                        the refresh.  If None (default), the previously
                        selected group will be re-selected.
        """
        # Save the selection so we can restore it
        if env_group_id_to_select is None:
            env_group_id_to_select = self.get_selected_env_group()
        self._set_env_group_container()
        self._list_group.setValue(env_group_id_to_select)

    def set_state_button_env_group(self):
        """Enable or disable (greyed out) action buttons whether an
        environment group is selected or not.
        """
        selected_item_id = self.get_selected_env_group()
        if selected_item_id is None:
            self._edit_group.setEnabled(False)
            self._del_group.setEnabled(False)
            self._where_group.setEnabled(False)
            self.disable_button_var()
        else:
            self._del_group.setEnabled(True)
            self._edit_group.setEnabled(True)
            self._where_group.setEnabled(True)
            self.enable_button_var()

    def propagate_select_env_group(self):
        """Display the selected environment group details (name, description
        and variables) in the right pane of the window.
        """
        selected_item_id = self.get_selected_env_group()
        if selected_item_id is None:
            self._name.setValue('')
            self._descr.setValue('')
            self._descr.removeStyleName('envdescription')
        else:
            c = self._list_group.getContainerDataSource()
            item = c.getItem(selected_item_id)
            self._name.setValue(item.getItemProperty('env').getValue())
            d = item.getItemProperty('descr').getValue().strip()
            self._descr.setValue(d)
            if d:
                self._descr.addStyleName('envdescription')
            else:
                self._descr.removeStyleName('envdescription')
        self._set_variable_container(selected_item_id)

    def get_item_details_env_group(self, item_id=None):
        """Get the specified environment group details.

        @param item_id:
                    the database ID of the environment group.
                    If None (default), the details of the selected group
                    are retrieved.
        @return:    a dictionnary containing the group details.
        """
        if item_id is None:
            item_id = self.get_selected_env_group()
        c = self._list_group.getContainerDataSource()
        item = c.getItem(item_id)
        return {'env':   item.getItemProperty('env').getValue(),
                'descr': item.getItemProperty('descr').getValue(),
                'id':    item_id}

    def get_previous_env_group_id(self, item_id):
        """Return the database ID of the previous environment group in the
           list.

        @param item_id:
                    the database ID of the environment group.
        @return:    the ID of the previous group in the list.
        """
        c = self._list_group.getContainerDataSource()
        i = c.prevItemId(item_id)
        return i if i is not None else c.nextItemId(item_id)

    def get_selected_var(self):
        """Return the currently selected variable.

        @return:    the container ID of the selected variable.
        """
        return self._var_table.getValue()

    def get_datasource_var(self):
        """Return the container data source.

        @return:        the container data source associated with the
                        variable table.
        """
        return self._var_table.getContainerDataSource()

    def refresh_var(self):
        """Refresh the variable table."""
        self._var_table.setVisibleColumns(['env_key', 'env_value'])

    def copy_item_var(self, old_item, new_item):
        """Copy a variable item.

        @param old_item:    the source container item.
        @param new_item:    the destination container item
        """
        k = old_item.getItemProperty('env_key').getValue()
        v = old_item.getItemProperty('env_value').getValue()
        new_item.getItemProperty('env_key').setValue(k)
        new_item.getItemProperty('env_value').setValue(v)

    def select_item_var(self, var_id):
        """Selecte the specified item ID in the variable table.

        @param var_id:  containet item ID to select.
        """
        self._var_table.select(var_id)

    def get_item_details_var(self, item_id=None):
        """Get the specified variable details.

        @param item_id:
                    the container ID of the variable.  If None (default),
                    the details of the selected variable are retrieved.
        @return:    a dictionnary containing the variable details.
        """
        if item_id is None:
            item_id = self.get_selected_var()
        c = self.get_datasource_var()
        item = c.getItem(item_id)
        return {'env_key':   item.getItemProperty('env_key').getValue(),
                'env_value': item.getItemProperty('env_value').getValue(),
                'id':        item_id}

    def disable_button_var(self):
        """Disable (greyed out) the variable table action buttons."""
        self._btadd.setEnabled(False)
        self._btedit.setEnabled(False)
        self._btdel.setEnabled(False)
        self._bttop.setEnabled(False)
        self._btup.setEnabled(False)
        self._btdown.setEnabled(False)
        self._btbottom.setEnabled(False)

    def enable_button_var(self):
        """Enable the variable table action buttons."""
        self._btadd.setEnabled(True)
        self.set_state_button_var()

    def set_state_button_var(self):
        """Enable or disable (greyed out) action buttons whether a
        variable is selected or not.
        """
        selected_item_id = self.get_selected_var()
        if selected_item_id is None:
            self._btedit.setEnabled(False)
            self._btdel.setEnabled(False)
            self._bttop.setEnabled(False)
            self._btup.setEnabled(False)
            self._btdown.setEnabled(False)
            self._btbottom.setEnabled(False)
        else:
            self._btedit.setEnabled(True)
            self._btdel.setEnabled(True)
            c = self.get_datasource_var()
            if c.isFirstId(selected_item_id):
                self._bttop.setEnabled(False)
                self._btup.setEnabled(False)
            else:
                self._bttop.setEnabled(True)
                self._btup.setEnabled(True)
            if c.isLastId(selected_item_id):
                self._btdown.setEnabled(False)
                self._btbottom.setEnabled(False)
            else:
                self._btdown.setEnabled(True)
                self._btbottom.setEnabled(True)

    def write_database_var(self):
        """Write the variables to the database."""
        env_group_id = self.get_selected_env_group()
        if env_group_id is not None:
            # Build the list of environment variables from the table
            new_list = list()
            c = self.get_datasource_var()
            for idx in range(c.size()):
                i = c.getIdByIndex(idx)
                item = c.getItem(i)
                k = item.getItemProperty('env_key').getValue()
                v = item.getItemProperty('env_value').getValue()
                if self._workload is None:
                    ev = environment_var(idx + 1, k, v)
                else:
                    ev = environment_var_s(idx + 1, k, v, self._workload)
                new_list.append(ev)

            # Retrieve the environment group from the database
            # and update the variables
            session = self._sql_session.open_session()
            try:
                e = environments_utils.get_env(session, env_group_id,
                                               self._workload)
            except:
                self._sql_session.close_session(session)
                self._main_window.showNotification(
                    _("Cannot get the selected environment group details"),
                    _("<br/>Maybe someone else just removed it."),
                    Notification.TYPE_ERROR_MESSAGE)
                self.refresh_env_group()
                return
            e.environment_var = new_list
            self._sql_session.close_session(session)

    def buttonClick(self, event):
        self._main_window.removeWindow(self)


class EnvGroupSelected(IValueChangeListener):

    """Callback for when an environment group is selected in the list."""

    def __init__(self, env_window_obj):
        """Initialize the callback.

        @param env_window_obj:
                    the associated L{EnvWindow} object.
        """
        super(EnvGroupSelected, self).__init__()
        self._c = env_window_obj

    def valueChange(self, event):
        # Enable/disable action buttons
        self._c.set_state_button_env_group()
        # Display the selected environment group details
        self._c.propagate_select_env_group()


class EnvGroupEdit(IClickListener):

    """Callback for the Edit environment group button."""

    def __init__(self, env_window_obj):
        """Initialize the callback.

        @param env_window_obj:
                    the associated L{EnvWindow} object.
        """
        super(EnvGroupEdit, self).__init__()
        self._c = env_window_obj

    def buttonClick(self, event):
        r = self._c.get_item_details_env_group()
        NewEnvGroup(self._c, r)


class EnvGroupAdd(IClickListener):

    """Callback for the Add new environment group button."""

    def __init__(self, env_window_obj):
        """Initialize the callback.

        @param env_window_obj:
                    the associated L{EnvWindow} object.
        """
        super(EnvGroupAdd, self).__init__()
        self._c = env_window_obj

    def buttonClick(self, event):
        NewEnvGroup(self._c)


class DeleteConfirmed(BoxCloseListener):

    """Delete confirmation callback."""

    def __init__(self, env_window_obj, env_id):
        """Initialize the object.

        @param env_window_obj
                    the associated L{EnvWindow} object.
        @param env_id:
                    database ID of the environment group to remove.
        """
        super(DeleteConfirmed, self).__init__()
        self._c = env_window_obj
        self._env_id = env_id

    def boxClose(self, event):
        """Called when the user confirmes the deletion."""
        session = self._c._sql_session.open_session()
        try:
            e = environments_utils.get_env(session, self._env_id,
                                           self._c._workload)
        except:
            pass
        else:
            session.delete(e)
        self._c._sql_session.close_session(session)
        self._c.refresh_env_group(
            self._c.get_previous_env_group_id(self._env_id))


class EnvGroupDel(IClickListener):

    """Callback for the Delete environment group button."""

    def __init__(self, env_window_obj):
        """Initialize the callback.

        @param env_window_obj:
                    the associated L{EnvWindow} object.
        """
        super(EnvGroupDel, self).__init__()
        self._c = env_window_obj

    def buttonClick(self, event):
        selected_item_id = self._c.get_selected_env_group()
        if selected_item_id is not None:
            session = self._c._sql_session.open_session()
            try:
                e = environments_utils.get_env(session, selected_item_id,
                                               self._c._workload)
            except:
                self._c._sql_session.close_session(session)
                self._c.refresh_env_group()
                return
            name = e.name.encode('utf-8')
            if cmd_env.whatuses.is_used(session, e, self._c._workload):
                self._c._sql_session.close_session(session)
                self._c._main_window.showNotification(
                    _("This environment is still in used"),
                    _("<br/>%s is in used somewhere else and cannot be \
                       deleted from the database.") % str(name),
                    Notification.TYPE_ERROR_MESSAGE)
            else:
                self._c._sql_session.close_session(session)
                d = DeleteBox(self._c._main_window,
                              _("Delete"),
                              _('Are you sure you want to delete this \
                                 environment group?'),
                              _('%s and all its variables will be destroyed.')
                              % str(name))
                d.addListener(DeleteConfirmed(self._c, selected_item_id))


class EnvGroupWhere(IClickListener):

    """Callback for the Where used button."""

    def __init__(self, env_window_obj):
        """Initialize the callback.

        @param env_window_obj:
                    the associated L{EnvWindow} object.
        """
        super(EnvGroupWhere, self).__init__()
        self._c = env_window_obj

    def buttonClick(self, event):
        selected_item_id = self._c.get_selected_env_group()
        if selected_item_id is not None:
            session = self._c._sql_session.open_session()
            try:
                e = environments_utils.get_env(session, selected_item_id,
                                               self._c._workload)
            except:
                self._c._sql_session.close_session(session)
                self._c._main_window.showNotification(
                    _("Cannot get the selected environment group details"),
                    _("<br/>Maybe someone else just removed it."),
                    Notification.TYPE_ERROR_MESSAGE)
                self._c.refresh_env_group()
                return
            # The items (job, jobset, host) that are using this environment
            # group are stored in a list.  Later on, they will be displayed
            # in a popup.
            self._gathered = list()
            cmd_env.whatuses.print_whatuses(session, e, self._c._workload,
                                            self._gather, self)
            self._c._sql_session.close_session(session)
            if not self._gathered:
                self._c._main_window.showNotification(
                    _("Not used"),
                    _("<br/>%s is not used.") % str(e.name.encode('utf-8')))
            else:
                WhereWindow(self._c, self._gathered)

    @staticmethod
    def _gather(self, title, val):
        """Callback method for the L{cmd_env.whatuses.print_whatuses}
        function.  This method is used to collect the items that are using
        an environment group.
        """
        self._gathered.append([title, val])


class NewEnvGroup(IClickListener):

    """Edit/Add window for an environment group."""

    _bt_captions = [_('Cancel'), _('OK')]

    def __init__(self, env_window_obj, params=None):
        """Create and display the Edit/Add window.

        @param env_window_obj:
                    the associated L{EnvWindow} object.
        @param params:
                    when the window is used to edit an existing environment,
                    this is the environment details.  This parameter is
                    a dictionnary as returned by
                    L{EnvWindow.get_item_details_env_group}.
        """
        super(NewEnvGroup, self).__init__()

        self._c = env_window_obj
        self._params = params

        if params is None:
            title = _('New Environment group')
        else:
            title = _('Edit Environment group')
        self._w = Window(title)
        self._w.setWidth("360px")

        # VerticalLayout as content by default
        v = self._w.getContent()
        v.setSizeFull()
        v.setMargin(True)
        v.setSpacing(True)

        t = TextField(_('Name:'))
        t.setWidth('100%')
        t.setDescription(_('Environment group name'))
        if params and 'env' in params:
            t.setValue(params['env'])
        self._name = t
        v.addComponent(t)

        t = TextArea(_('Description:'))
        t.setSizeFull()
        t.setDescription(_('Free text as a comment or description'))
        if params and 'descr' in params:
            t.setValue(params['descr'])
        self._descr = t
        v.addComponent(t)
        v.setExpandRatio(t, 1.0)

        # Button box
        h_bt = HorizontalLayout()
        h_bt.setMargin(False)
        h_bt.setSpacing(True)

        for caption in self._bt_captions:
            b = Button(_(caption))
            b.addListener(self, IClickListener)
            h_bt.addComponent(b)

        v.addComponent(h_bt)
        v.setComponentAlignment(h_bt, Alignment.BOTTOM_RIGHT)

        self._c._main_window.addWindow(self._w)
        self._w.center()

    def buttonClick(self, event):
        # First button is Cancel
        if event.getButton().getCaption() != _(self._bt_captions[0]):
            # Retrieve the values from the form
            name = self._name.getValue().strip()
            if not name:
                self._c._main_window.showNotification(
                    _("Name is empty"),
                    _("<br/>The environment group name cannot \
                       be empty or contain just spaces."),
                    Notification.TYPE_ERROR_MESSAGE)
                self._name.focus()
                return
            descr = self._descr.getValue().strip()

            # Try to retrieve the environment from the database
            session = self._c._sql_session.open_session()
            try:
                e = environments_utils.name2env(session, name,
                                                self._c._workload)
            except:
                # Not found.
                e = None
            else:
                # Already in the database
                # If it was an Add Window, print an error message.
                if self._params is None or self._params['id'] != e.id:
                    self._c._sql_session.close_session(session)
                    self._c._main_window.showNotification(
                        _("This name is already used"),
                        _("<br/>The environment group %s is already \
                        defined in the database.") % name,
                        Notification.TYPE_ERROR_MESSAGE)
                    return

            # Add the new environment group in the database
            if self._params is None:
                if self._c._workload is None:
                    e = environments(name, descr)
                else:
                    e = environments_s(name, descr, self._c._workload)
                session.add(e)

            # Edit the environment group
            else:
                if e is None:
                    # Retrieve the old environment
                    try:
                        e = environments_utils.get_env(session,
                                                       self._params['id'],
                                                       self._c._workload)
                    except:
                        # Humm, it should have been in the database...
                        # So recreating the environment group
                        if self._c._workload is None:
                            e = environments(name, descr)
                        else:
                            e = environments_s(name, descr, self._c._workload)
                        session.add(e)
                e.name = name.decode('utf-8')
                e.description = descr.decode('utf-8')
            self._c._sql_session.close_session(session)
            self._c.refresh_env_group(e.id)

        # Close the window
        self._c._main_window.removeWindow(self._w)


class WhereWindow(IClickListener):

    """Window to display by what an environment group is used."""

    def __init__(self, env_window_obj, obj_list):
        """Create and display the window.

        @param env_window_obj:
                    the associated L{EnvWindow} object.
        @param obj_list:
                    list of objects.  Each object is an array of two
                    strings.  The first one is the type of object (job,
                    jobset, host) and the second one is the value.
        """
        super(WhereWindow, self).__init__()

        self._c = env_window_obj

        self._w = Window(_("Environment group references"))
        self._w.setWidth("320px")
        self._w.setHeight("320px")

        # VerticalLayout as content by default
        v = self._w.getContent()
        v.setSizeFull()
        v.setMargin(True)
        v.setSpacing(True)

        t = Table()
        t.setSizeFull()
        t.setColumnReorderingAllowed(False)
        t.setColumnCollapsingAllowed(False)
        t.setSortDisabled(False)
        t.setSelectable(False)
        t.setMultiSelect(False)
        c = IndexedContainer()
        c.addContainerProperty('type', str, None)
        c.addContainerProperty('value', str, None)
        for k, val in obj_list:
            val = unicode(val)
            item = c.addItem(c.generateId())
            item.getItemProperty('type').setValue(k)
            item.getItemProperty('value').setValue(val.encode('utf-8'))
        t.setContainerDataSource(c)
        t.setColumnExpandRatio('value', 1.0)
        v.addComponent(t)
        v.setExpandRatio(t, 1.0)

        # Close button
        ok = Button(_('Close'))
        ok.addListener(self, IClickListener)
        v.addComponent(ok)
        v.setComponentAlignment(ok, Alignment.BOTTOM_RIGHT)

        self._c._main_window.addWindow(self._w)
        self._w.center()

    def buttonClick(self, event):
        self._c._main_window.removeWindow(self._w)


class VarMenu(GenericContextMenu):

    """Variable table context menu."""

    def __init__(self, env_window_obj):
        """Initialize the menu.

        @param env_window_obj:
                    the associated L{EnvWindow} object.
        """
        super(VarMenu, self).__init__()
        self._c = env_window_obj

    def getActions(self, target, sender):
        if not target:
            return [self._ACTION_ADD]
        else:
            return [self._ACTION_EDIT, self._ACTION_DELETE]

    def handleAction(self, a, sender, target):
        if a.getCaption() == self._ACTION_ADD.getCaption():
            VarPopup(self._c)
        elif a.getCaption() == self._ACTION_EDIT.getCaption():
            r = self._c.get_item_details_var(target)
            VarPopup(self._c, params=r)
        elif a.getCaption() == self._ACTION_DELETE.getCaption():
            d = DeleteBox(
                self._c._main_window,
                _("Delete"),
                _('Are you sure you want to remove this variable?'))
            d.addListener(DeleteVarConfirmed(self._c, target))


class VarSelected(IValueChangeListener):

    """Callback for when a variable is selected in the table."""

    def __init__(self, env_window_obj):
        """Initialize the callback.

        @param env_window_obj:
                    the associated L{EnvWindow} object.
        """
        super(VarSelected, self).__init__()
        self._c = env_window_obj

    def valueChange(self, event):
        # Enable/disable action buttons
        self._c.set_state_button_var()


class VarEdit(IClickListener):

    """Callback for the Edit variable button."""

    def __init__(self, env_window_obj):
        """Initialize the callback.

        @param env_window_obj:
                    the associated L{EnvWindow} object.
        """
        super(VarEdit, self).__init__()
        self._c = env_window_obj

    def buttonClick(self, event):
        r = self._c.get_item_details_var()
        #VarPopup(self._c, event.getClientX(), event.getClientY(), r)
        VarPopup(self._c, params=r)


class VarAdd(IClickListener):

    """Callback for the Add new variable button."""

    def __init__(self, env_window_obj):
        """Initialize the callback.

        @param env_window_obj:
                    the associated L{EnvWindow} object.
        """
        super(VarAdd, self).__init__()
        self._c = env_window_obj

    def buttonClick(self, event):
        #VarPopup(self._c, event.getClientX(), event.getClientY())
        VarPopup(self._c)


class DeleteVarConfirmed(BoxCloseListener):

    """Delete a variable confirmation callback."""

    def __init__(self, env_window_obj, var_id):
        """Initialize the object.

        @param env_window_obj
                    the associated L{EnvWindow} object.
        @param var_id:
                    container ID of the variable to remove.
        """
        super(DeleteVarConfirmed, self).__init__()
        self._c = env_window_obj
        self._var_id = var_id

    def boxClose(self, event):
        """Called when the user confirms the deletion."""
        c = self._c.get_datasource_var()
        item_id_to_select = c.prevItemId(self._var_id)
        if item_id_to_select is None:
            item_id_to_select = c.nextItemId(self._var_id)
        c.removeItem(self._var_id)
        self._c.refresh_var()
        self._c.select_item_var(item_id_to_select)
        self._c.write_database_var()


class VarDel(IClickListener):

    """Callback for the Delete variable button."""

    def __init__(self, env_window_obj):
        """Initialize the callback.

        @param env_window_obj:
                    the associated L{EnvWindow} object.
        """
        super(VarDel, self).__init__()
        self._c = env_window_obj

    def buttonClick(self, event):
        d = DeleteBox(self._c._main_window,
                      _("Delete"),
                      _('Are you sure you want to remove this variable?'))
        d.addListener(DeleteVarConfirmed(self._c, self._c.get_selected_var()))


class VarTop(IClickListener):

    """Callback for the Move variable to top button."""

    def __init__(self, env_window_obj):
        """Initialize the callback.

        @param env_window_obj:
                    the associated L{EnvWindow} object.
        """
        super(VarTop, self).__init__()
        self._c = env_window_obj

    def buttonClick(self, event):
        var_id = self._c.get_selected_var()
        if var_id is not None:
            c = self._c.get_datasource_var()
            if not c.isFirstId(var_id):
                old_item = c.getItem(var_id)
                i = c.generateId()
                new_item = c.addItemAt(0, i)
                self._c.copy_item_var(old_item, new_item)
                c.removeItem(var_id)
                self._c.select_item_var(i)
                self._c.write_database_var()


class VarUp(IClickListener):

    """Callback for the Move variable up button."""

    def __init__(self, env_window_obj):
        """Initialize the callback.

        @param env_window_obj:
                    the associated L{EnvWindow} object.
        """
        super(VarUp, self).__init__()
        self._c = env_window_obj

    def buttonClick(self, event):
        var_id = self._c.get_selected_var()
        if var_id is not None:
            c = self._c.get_datasource_var()
            prev_item_id = c.prevItemId(var_id)
            if prev_item_id is not None:
                old_item = c.getItem(var_id)
                idx = c.indexOfId(prev_item_id)
                i = c.generateId()
                new_item = c.addItemAt(idx, i)
                self._c.copy_item_var(old_item, new_item)
                c.removeItem(var_id)
                self._c.select_item_var(i)
                self._c.write_database_var()


class VarDown(IClickListener):

    """Callback for the Move variable down button."""

    def __init__(self, env_window_obj):
        """Initialize the callback.

        @param env_window_obj:
                    the associated L{EnvWindow} object.
        """
        super(VarDown, self).__init__()
        self._c = env_window_obj

    def buttonClick(self, event):
        var_id = self._c.get_selected_var()
        if var_id is not None:
            c = self._c.get_datasource_var()
            next_item_id = c.nextItemId(var_id)
            if next_item_id is not None:
                old_item = c.getItem(var_id)
                idx = c.indexOfId(next_item_id)
                i = c.generateId()
                new_item = c.addItemAt(idx + 1, i)
                self._c.copy_item_var(old_item, new_item)
                c.removeItem(var_id)
                self._c.select_item_var(i)
                self._c.write_database_var()


class VarBottom(IClickListener):

    """Callback for the Move variable to bottom button."""

    def __init__(self, env_window_obj):
        """Initialize the callback.

        @param env_window_obj:
                    the associated L{EnvWindow} object.
        """
        super(VarBottom, self).__init__()
        self._c = env_window_obj

    def buttonClick(self, event):
        var_id = self._c.get_selected_var()
        if var_id is not None:
            c = self._c.get_datasource_var()
            if not c.isLastId(var_id):
                old_item = c.getItem(var_id)
                i = c.generateId()
                new_item = c.addItem(i)
                self._c.copy_item_var(old_item, new_item)
                c.removeItem(var_id)
                self._c.select_item_var(i)
                self._c.write_database_var()


class VarPopup(IClickListener):

    """Display the add/edit variable popup."""

    def __init__(self, env_window_obj, x=None, y=None, params=None):
        """Create the popup.

        @param env_window_obj:
                the associated L{EnvWindow} object.
        @param x:
                the X coordinate of the mouse which is used to position
                the popup.  By default, the popup is centered.
        @param y:
                the Y coordinate of the mouse which is used to position
                the popup.  By default, the popup is centered.
        @param params:
                when the window is used to edit an existing variable,
                this is the variable details.  This parameter is
                a dictionnary as returned by L{EnvWindow.get_item_details_var}.
        """
        super(VarPopup, self).__init__()
        self._c = env_window_obj
        self._params = params

        if params is None:
            title = _('New variable')
        else:
            title = _('Edit variable')
        h = HorizontalLayout()
        h.setWidth('100%')
        h.setMargin(True)
        h.setSpacing(True)

        self._key = TextField(_('Name:'))
        self._key.focus()
        self._key.setWidth('100%')
        self._key.setImmediate(True)
        self._key.addValidator(RegexpValidator(
            '^[A-Za-z_][A-Za-z_0-9]*$',
            True,
            _("Only letters, numbers, and underscores, and \
               beginning with a letter or underscore")))
        h.addComponent(self._key)
        h.setExpandRatio(self._key, 1.0)

        l = Label('=')
        l.setWidth('15px')
        h.addComponent(l)
        h.setComponentAlignment(l, Alignment.BOTTOM_CENTER)

        self._val = TextField(_('Value:'))
        self._val.setWidth('100%')
        h.addComponent(self._val)
        h.setExpandRatio(self._val, 2.0)

        if params is not None:
            self._key.setValue(params['env_key'])
            self._val.setValue(params['env_value'])

        ok = Button(_('OK'))
        ok.addListener(self, IClickListener)
        ok.setClickShortcut(KeyCode.ENTER)
        h.addComponent(ok)
        h.setComponentAlignment(ok, Alignment.BOTTOM_RIGHT)

        self._w = Window(title, h)
        self._w.setWidth('450px')
        self._c._main_window.addWindow(self._w)
        if x and y:
            self._w.setPositionX(x)
            self._w.setPositionY(y)
        else:
            self._w.center()

    def buttonClick(self, event):
        # Retrieve the entered values
        name = self._key.getValue().strip()
        if not name or not re.match('^[A-Za-z_][A-Za-z_0-9]*$', name):
            self._c._main_window.showNotification(
                _("Invalid variable name"),
                _("<br/>The variable name cannot be empty. It must contain \
                   only <br/>letters, numbers, and underscores, and begin \
                   with a <br/>letter or underscore."),
                Notification.TYPE_ERROR_MESSAGE)
            self._key.focus()
            return
        val = self._val.getValue()
        c = self._c.get_datasource_var()
        # New
        if self._params is None:
            item = c.addItem(c.generateId())
            self._c.refresh_var()
        # Update
        else:
            item = c.getItem(self._params['id'])
        item.getItemProperty('env_key').setValue(name)
        item.getItemProperty('env_value').setValue(val)
        self._c._main_window.removeWindow(self._w)
        self._c.write_database_var()
