# Schedwi
# Copyright (C) 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Database queries for the job_stat table."""

import time
import sqlalchemy.orm.session

from tables.job_stat import job_stat
import status_utils


def sql_get_stat(sql_session, job_id):
    """Retrieve a job or jobset statistics.

    @param sql_session:
                SQLAlchemy session (it can be an opened session)
    @param job_id:
                the job ID to look for in the database.
    @return:    the statistics.
    @rtype:     L{tables.job_stat.job_stat}
    @raise sqlalchemy.orm.exc.NoResultFound:
                the given job ID is not in the database.
    """
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
    else:
        session = sql_session

    try:
        ret = session.query(job_stat).filter_by(job_id=job_id).one()
    except:
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)
        raise

    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return ret


def sql_get_stat_ratio_completion(sql_session, job_id, workload):
    """Return the completion ration of the given job or jobset.

    @param sql_session:
                SQLAlchemy session (it can be an opened session)
    @param job_id:
                the job ID to look for in the database.
    @param workload:
                workload to consider.
    @return:    the completion ratio of the running job (between 0 and 1)
                or None if the job/jobset is not running or if no statistics
                are available.
    @raise sqlalchemy.orm.exc.NoResultFound:
                the given job ID is not in the database.
    """
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
    else:
        session = sql_session

    # Retrieve the current status of the job/jobset
    s = status_utils.get_status(session, job_id, workload)
    if not s or s.status != status_utils.RUNNING:
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)
        return None

    # Retrieve the stats for the given job/jobset
    try:
        stats = sql_get_stat(session, job_id)
    except:
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)
        return None
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    if stats.num_success == 0 or stats.total_duration == 0:
        return None
    ratio = (time.time() - s.start_time) / \
            (stats.total_duration / stats.num_success)
    if ratio > 1.0:
        ratio = 1.0
    return ratio
