# Schedwi
# Copyright (C) 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to force the status of jobs and jobsets."""

import sys
import getopt

import path
import status_utils
from help import print_trim


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: force [OPTION]... JOB|JOBSET...
Force the status of JOBs and JOBSETs.  If no option is given, the status is
set the completed.

Options:
  -c, --completed  force the status of JOB|JOBSET to completed.  This is the
                   default.
  -f, --failed     force the status of JOB|JOBSET to failed.
  -r, --running    force the status of JOB|JOBSET to running.
  -w, --waiting    force the status of JOB|JOBSET to waiting.
  -h, --help       display this help.
    """))


def force(sql_session, current_cwd, arguments, workload):
    """Force the status of jobs and jobsets.

    @param sql_session:
                SQLAlchemy session.
    @param current_cwd:
                current working jobset.
    @type current_cwd:
                L{path.Path}
    @param arguments:
                list of arguments given to the force command.
    @param workload:
                workload to consider.
    @return:
                0 on success and 1 on error (an error message has been
                displayed)
    """
    try:
        optlist, args = getopt.getopt(arguments,
                                      "cfrwh",
                                      [
                                          "completed",
                                          "failed",
                                          "running",
                                          "waiting",
                                          "help"])
    except getopt.GetoptError, err:
        sys.stderr.write(_("force: ") + str(err) + "\n")
        return 1
    st = status_utils.COMPLETED
    for o, a in optlist:
        if o in ("-c", "--completed"):
            st = status_utils.COMPLETED
        elif o in ("-f", "--failed"):
            st = status_utils.FAILED
        elif o in ("-r", "--running"):
            st = status_utils.RUNNING
        elif o in ("-w", "--waiting"):
            st = status_utils.WAITING
        elif o in ("-h", "--help"):
            usage()
            return 0
    paths = list()
    if args:
        for arg in args:
            p = path.get_paths(sql_session, arg, current_cwd,
                               workload=workload)
            # An argument is an unknow path
            if not p:
                sys.stderr.write(_("force: `%s': no such job or jobset\n")
                                 % arg)
                return 1
            paths.extend(p)
    else:
        sys.stderr.write(_("force: missing job or jobset operand\n"))
        return 1
    session = sql_session.open_session()
    for p in paths:
        try:
            status_utils.set_status(session, p.id[-1], workload, st)
        except:
            sys.stderr.write(_("force: `%s': failed to set the status\n"))
    sql_session.close_session(session)
    return 0
