# Schedwi
# Copyright (C) 2011-2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to list jobs and jobsets."""

import sys
import getopt

import config
import console_size
import path
import path_cal
import status_utils
import sql_hierarchy
import cluster_utils
import host_utils
from tables.job_main import job_main
from tables.job_cluster import job_cluster
from tables.job_main_s import job_main_s
from tables.job_cluster_s import job_cluster_s
from help import print_trim


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: ls [OPTION]... [JOB|JOBSET]...
List information about the JOBs or JOBSETs (the current jobset by default).
Sort entries alphabetically.

Options:
  -l  use a long listing format according to the following schema:
      - or j  whether the entry is a job or a jobset
      - or d  whether the job/jobset is enabled or disabled
      HHhMM   start time
      CAL     calendar name
      HOST    cluster or host name and TCP port (if not the default port 2006)
      NAME    job/jobset name
      [--S--> JOB/JOBSET]
              link (the specified JOB/JOBSET must have the status S for the
              listed job/jobset to start). S can be
                w  for Waiting
                r  for Running
                c  for completed
                f  for failed
                d  for done (completed or failed)
              A link of `-----> ...' means that the job/jobset depends on more
              than one other job/jobset (use `cat job' to get the full list)
  -h, --help  display this help.

Example:
> ls -l
-- 18h14 Every Friday  flower.example.com     closeweek  ----->  ...
-- 00h00 Every Monday  cherry.example.com     initday    --c--> /test/new_week
-- 00h00 Every Monday  localhost              new_week
-- 00h00 Every day     plum.example.com/2007  report
-- 18h14 Every Friday  apple.example.com      send_week_files
j- 00h00 Every day     localhost              sub_reports

report    is a job that starts at 00:00 every day (according to the calendar
          named `Every day') on the host plum.example.com (TCP port 2007)
initday   is a job that starts at 00:00 but when the job /test/new_week is
          successful (completed)
closeweek  is a job that starts at 18:14 and depends of the completion of
           several other jobs/jobsets (`cat closeweek' list all these jobs)
    """))

_INTERSPACE = 2


def _max(a, b):
    i = len(b.name)
    if i > a:
        return i
    return a


def _print1job(session, job, parent_id, long_listing, workload=None):
    """Print the provided job/jobset name."""
    if not long_listing:
        print job.name
        return
    # Start
    if job.start_time >= 0:
        h = job.start_time / 100
        m = job.start_time % 100
    else:
        for i in range(len(parent_id) - 2, -1, -1):
            if workload is None:
                query = session.query(job_main.start_time)
            else:
                query = session.query(job_main_s.start_time)
                query = query.filter(job_main_s.workload_date == workload)
            start_time, = query.filter_by(id=parent_id[i]).one()
            if start_time >= 0:
                h = start_time / 100
                m = start_time % 100
                break
        else:
            h = m = 0
    # host/cluster
    if job.job_cluster is not None:
        cluster = cluster_utils.get_cluster_by_id(session,
                                                  job.job_cluster.cluster_id)
        host_str = 'c(' + cluster.name + ')'
    elif job.job_host is not None:
        host = host_utils.get_host_by_id(session, job.job_host.host_id)
        host_str = str(host)
    else:
        in_object, obj = sql_hierarchy.get_cluster_or_host_by_id(session,
                                                                 parent_id,
                                                                 workload)
        if obj is None:
            host_str = "None"
        else:
            if isinstance(obj, job_cluster) or isinstance(obj, job_cluster_s):
                cluster = cluster_utils.get_cluster_by_id(session,
                                                          obj.cluster_id)
                host_str = 'c(' + cluster.name + ')'
            else:
                host = host_utils.get_host_by_id(session, obj.host_id)
                host_str = str(host)
    # calendar
    if job.cal_id == 0:
        for i in range(len(parent_id) - 2, -1, -1):
            if workload is None:
                query = session.query(job_main.cal_id)
            else:
                query = session.query(job_main_s.cal_id)
                query = query.filter(job_main_s.workload_date == workload)
            cal_id, = query.filter_by(id=parent_id[i]).one()
            if cal_id != 0:
                cal_full_name = path_cal.PathCal(session, id=cal_id,
                                                 workload=workload)
                break
        else:
            cal_full_name = _("Every day")
    else:
        cal_full_name = path_cal.PathCal(session, id=job.cal_id,
                                         workload=workload)
    cal_name = cal_full_name.path.rsplit('/')[-1]
    # links
    num_links = len(job.links)
    if num_links == 0:
        link = ""
    elif num_links > 1:
        link = "-----> ..."
    else:
        p = path.id2path(session, job.links[0].job_id_destination, workload)
        link = "--%s--> %s" % (
            status_utils.status2char(job.links[0].required_status),
            p)
    # print the result
    print "%(jobset)s%(enabled)s %(h)02dh%(m)02d %(cal)-13s %(host)-15s \
%(name)s %(link)s " % \
        {"jobset":   'j' if job.type == 0 else '-',
         "enabled":  'd' if not job.enabled else '-',
         "h":        h,
         "m":        m,
         "cal":      cal_name,
         "host":     host_str,
         "name":     job.name,
         "link":     link}


def _print_jobs(session, jobs, parent_id, long_listing, workload=None):
    """Print the given jobs."""
    if config.ISATTY_STDOUT and not long_listing:
        m = reduce(_max, jobs, 0) + _INTERSPACE
        w = console_size.width
        n = w / m
        i = 0
        for job in jobs:
            f = m - len(job.name)
            sys.stdout.write(job.name + ' ' * f)
            i += 1
            if i == n:
                i = 0
                sys.stdout.write("\n")
        if i:
            sys.stdout.write("\n")
    else:
        for job in jobs:
            p = list(parent_id)
            p.append(job.id)
            _print1job(session, job, p, long_listing, workload)


def ls(sql_session, current_cwd, arguments, workload=None):
    """List jobs and jobsets.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working jobset (a path.Path object)
    arguments -- list of arguments given to the ls command (list
                 of jobs/jobsets)
    workload -- workload to use

    """
    try:
        optlist, args = getopt.getopt(arguments, 'lh', ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write(_("ls: ") + str(err) + "\n")
        return 1
    long_listing = False
    for o, a in optlist:
        if o == "-l":
            long_listing = True
        elif o in ("-h", "--help"):
            usage()
            return 0
    paths = list()
    if args:
        for arg in args:
            p = path.get_paths(sql_session, arg, current_cwd,
                               workload=workload)
            if not p:
                sys.stderr.write(_("ls: `%s': no such job or jobset\n") % arg)
            else:
                paths.extend(p)
    else:
        paths.append(current_cwd)
    if not paths:
        sys.stderr.write(_("ls: no such job or jobset\n"))
        return 1
    session = sql_session.open_session()
    previous = 0
    error = False
    for p in paths:
        if workload is None:
            query = session.query(job_main)
        else:
            query = session.query(job_main_s)
            query = query.filter(job_main_s.workload_date == workload)
        try:
            job = query.filter_by(id=p.id[-1]).one()
        except:
            sys.stderr.write(_("ls: no such job or jobset\n"))
            error = True
            continue
        if job.type == 1:
            if previous == 1:  # The previous item was a jobset
                sys.stdout.write("\n")
            _print1job(session, job, p.id, long_listing, workload)
            previous = 2
        else:
            if len(paths) > 1:
                if previous:
                    sys.stdout.write("\n")
                print "%s:" % p
            previous = 1
            if workload is None:
                query = session.query(job_main)
            else:
                query = session.query(job_main_s)
                query = query.filter(job_main_s.workload_date == workload)
            jobs = query.filter_by(parent=p.id[-1]).all()
            _print_jobs(session, jobs, p.id, long_listing, workload)
    sql_session.close_session(session)
    return 0 if not error else 1
