# Schedwi
# Copyright (C) 2012 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module for readline auto-completion of jobs and jobsets."""

import shlex

import path
from tables.job_main import job_main
from tables.job_main_s import job_main_s


def complete(sql_session, text, line, cwd, workload=None, only_jobset=False):
    """readline complete function.

    @param sql_session:
                SQLAlchemy session.
    @param text:
                string prefix we are attempting to match.
    @param line:
                current input line (full command)
    @param cwd:
                current working directory (jobset)
    @param workload:
                workload to consider.
    @param only_jobset:
                try to match only jobsets (and not jobs)
    """
    # Remove the command and option parts
    args = shlex.split(line)
    if len(args) == 1 or args[-1][0] == '-':
        dir_to_look = '.'
    else:
        dir_to_look = args[-1].decode('utf-8')
    if text:
        text = text.decode('utf-8')
        d, foo, foo = dir_to_look.rpartition(text)
        if d:
            p = path.Path(sql_session, d, cwd, workload=workload)
        else:
            p = path.Path(sql_session, '.', cwd, workload=workload)
    else:
        p = path.Path(sql_session, dir_to_look, cwd, workload=workload)
    session = sql_session.open_session()
    if workload is None:
        q = session.query(job_main)
        q = q.filter(job_main.name.like(text + '%'))
        q = q.filter(job_main.parent == p.id[-1])
        if only_jobset:
            q = q.filter(job_main.type == 0)
        q = q.order_by(job_main.name)
    else:
        q = session.query(job_main_s)
        q = q.filter(job_main_s.workload_date == workload)
        q = q.filter(job_main_s.name.like(text + '%'))
        q = q.filter(job_main_s.parent == p.id[-1])
        if only_jobset:
            q = q.filter(job_main_s.type == 0)
        q = q.order_by(job_main_s.name)
    lst = q.all()
    sql_session.close_session(session)
    if len(lst) == 1:
        name = lst[0].name.encode('utf-8')
        if lst[0].type == 0:
            if ' ' in name:
                return ['"' + name + '"/']
            else:
                return [name + '/']
        if ' ' in name:
            return ['"' + name + '" ']
        else:
            return [name + ' ']
    a = map(lambda x: '"' + x.name.encode('utf-8') + '"' if ' ' in x.name
            else x.name.encode('utf-8'), lst)
    return a
