# Schedwi
# Copyright (C) 2011, 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to get the jobs, jobsets and constraint files that use a host."""

import sys
import getopt

import path
from tables.job_main import job_main
from tables.job_host import job_host
from tables.constraint_file import constraint_file
from tables.job_host_s import job_host_s
from tables.constraint_file_s import constraint_file_s
from tables.job_main_s import job_main_s
from tables.clusters import clusters
from tables.host_clusters import host_clusters
from tables.job_status import job_status
import host_utils
import status_utils
from help import print_trim


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: whatuses HOST...
        List the jobs, jobsets, constraint files and clusters that use HOST.
        HOST must have the following format:
           HOSTNAME[:PORT]
        For IPv6, the address must be enclosed between square brackets (ie.
        [fe80::210:a2ff:fa1d:aabb:ccdd]:2006)

        Option:
          -h, --help  display this help.
    """))


def is_used(session, host):
    """Tell if the host is used.

    Arguments:
    session -- SQLAlchemy session
    host -- host object

    """
    if session.query(job_host).filter(job_host.host_id == host.id).count():
        return True
    if session.query(job_host_s).filter(job_host_s.host_id == host.id).count():
        return True
    q = session.query(job_status).filter(job_status.host_id == host.id)
    if q.filter(job_status.status == status_utils.RUNNING).count():
        return True
    q = session.query(constraint_file)
    if q.filter(constraint_file.host_id == host.id).count():
        return True
    q = session.query(constraint_file_s)
    if q.filter(constraint_file_s.host_id == host.id).count():
        return True
    q = session.query(host_clusters)
    if q.filter(host_clusters.host_id == host.id).count():
        return True
    return False


_FIELD_WIDTH = 15


def _print_val(foo, title, val):
    """Print a parameter value.

    Arguments:
    title -- parameter name
    val -- value

    """
    print "%s :" % title.rjust(_FIELD_WIDTH), val


def print_whatuses(session, host, print_function=_print_val, user_data=None):
    """Print the jobs, jobsets and constraint files that use a host.

    Arguments:
    session -- SQLAlchemy session
    host -- host object

    """
    # Jobs and jobsets
    q = session.query(job_main).filter(job_host.host_id == host.id)
    q = q.filter(job_host.job_id == job_main.id)
    for job in q.order_by(job_main.parent):
        print_function(user_data,
                       _("Job") if job.type else _("Jobset"),
                       "%s" % path.id2path(session, job.id))
    # Jobs and jobsets (in a workload)
    q = session.query(job_main_s).filter(job_host_s.host_id == host.id)
    q = q.filter(job_host_s.job_id == job_main_s.id)
    for job in q.order_by(job_main_s.workload_date, job_main_s.parent):
        print_function(user_data,
                       _("Job") if job.type else _("Jobset"),
                       "%s:%s" % (job.workload_date,
                                  path.id2path(session,
                                               job.id,
                                               job.workload_date)))
    # Running jobs
    q = session.query(job_main_s).filter(job_main_s.id == job_status.job_id)
    q = q.filter(job_main_s.workload_date == job_status.workload_date)
    q = q.filter(job_status.host_id == host.id)
    q = q.filter(job_status.status == status_utils.RUNNING)
    for job in q.order_by(job_main_s.workload_date, job_main_s.parent):
        print_function(user_data,
                       _("Running job") if job.type else _("Running jobset"),
                       "%d:%s" % (job.workload_date,
                                  path.id2path(session,
                                               job.id,
                                               job.workload_date)))
    # Constraint files
    q = session.query(constraint_file).filter(constraint_file.host_id ==
                                              host.id)
    for f in q.order_by(constraint_file.job_id):
        print_function(user_data,
                       _("Constraint file"),
                       _("%s: %s must %sexist") %
                       (path.id2path(session, f.job_id),
                        f.filename,
                        '' if f.exist else 'NOT '))
    # Constraint files (in a workload)
    q = session.query(constraint_file_s).filter(constraint_file_s.host_id ==
                                                host.id)
    for f in q.order_by(constraint_file_s.workload_date,
                        constraint_file_s.job_id):
        print_function(user_data,
                       _("Constraint file"),
                       _("%s:%s: %s must %sexist") %
                       (f.workload_date,
                        path.id2path(session, f.job_id, f.workload_date),
                        f.filename,
                        '' if f.exist else _('NOT ')))
    # Clusters
    q = session.query(clusters).filter(host_clusters.host_id == host.id)
    q = q.filter(host_clusters.cluster_id == clusters.id)
    for c in q.order_by(clusters.name):
        print_function(user_data, _("Cluster"), "%s" % c.name)


def whatuses(sql_session, arguments):
    """Show what is referencing a host.

    Arguments:
    sql_session -- SQLAlchemy session
    arguments -- list of arguments given to the whatuses command (list
                 of hosts)

    """
    try:
        optlist, args = getopt.getopt(arguments, "h", ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write(_("whatuses: ") + str(err) + "\n")
        return 1
    for o, a in optlist:
        if o in ("-h", "--help"):
            usage()
            return 0
    if not args:
        sys.stderr.write(_("whatuses: missing operand\n"))
        return 1
    session = sql_session.open_session()
    error = False
    for a in args:
        host_list = host_utils.name2host_list(session, a)
        if not host_list:
            sys.stderr.write(_("whatuses: `%s': no such host\n") % a)
            error = True
        else:
            first = True
            for h in host_list:
                if not first:
                    print
                print "%s:" % str(h)
                print_whatuses(session, h)
                first = False
    sql_session.close_session(session)
    return 0 if not error else 1
