# Schedwi
# Copyright (C) 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to get the jobs, jobsets that use a cluster."""

import sys
import getopt

import path
from tables.job_main import job_main
from tables.job_cluster import job_cluster
from tables.job_cluster_s import job_cluster_s
from tables.job_main_s import job_main_s
from tables.host_clusters import host_clusters
from tables.hosts import hosts
import cluster_utils
from help import print_trim


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: whatuses CLUSTER...
        List the jobs, jobsets and hosts that use CLUSTER.

        Option:
          -h, --help  display this help.
    """))


def is_used(session, cluster):
    """Tell if the cluster is used by a job/jobset.

    @param session:
                    SQLAlchemy session
    @param cluster:
                    The cluster object
    @type cluster:
                    L{tables.clusters.clusters}
    @return:
                    True if the cluster is used.  False otherwise
    """
    if session.query(job_cluster).filter(job_cluster.cluster_id ==
                                         cluster.id).count():
        return True
    if session.query(job_cluster_s).filter(job_cluster_s.cluster_id ==
                                           cluster.id).count():
        return True
    return False


_FIELD_WIDTH = 15


def _print_val(foo, title, val):
    """Print a parameter value.

    @param title:   parameter name
    @param val:     value
    """
    print "%s :" % title.rjust(_FIELD_WIDTH), val


def print_whatuses(session, cluster,
                   print_function=_print_val, user_data=None):
    """Print the jobs, jobsets and hosts that use a cluster.

    @param session:
                    SQLAlchemy session
    @param cluster:
                    The cluster object
    @type cluster:
                    L{tables.clusters.clusters}
    @param print_function:
                    Function called for each parameter to print
    @param user_data:
                    Data provided by the caller and passed as the first
                    parameter to print_function
    """
    q = session.query(job_main).filter(job_cluster.cluster_id == cluster.id)
    q = q.filter(job_cluster.job_id == job_main.id)
    for job in q.order_by(job_main.parent):
        print_function(user_data,
                       _("Job") if job.type else _("Jobset"),
                       "%s" % path.id2path(session, job.id))
    q = session.query(job_main_s)
    q = q.filter(job_cluster_s.cluster_id == cluster.id)
    q = q.filter(job_cluster_s.job_id == job_main_s.id)
    for job in q.order_by(job_main_s.workload_date, job_main_s.parent):
        print_function(user_data,
                       _("Job") if job.type else _("Jobset"),
                       "%s:%s" % (job.workload_date,
                                  path.id2path(session,
                                               job.id,
                                               job.workload_date)))
    q = session.query(hosts).filter(hosts.id == host_clusters.host_id)
    q = q.filter(host_clusters.cluster_id == cluster.id)
    for host in q.order_by(hosts.hostname):
        if host.sslenable:
            print_function(user_data, _("Host"), "%s (SSL)" % str(host))
        else:
            print_function(user_data, _("Host"), str(host))


def whatuses(sql_session, arguments):
    """Show what is referencing a cluster.

    @param sql_session:
                    SQLAlchemy session
    @param arguments:
                    list of arguments given to the whatuses command (list
                    of clusters)
    @return:
                    0 on success and 1 on error
    """
    try:
        optlist, args = getopt.getopt(arguments, "h", ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write(_("whatuses: ") + str(err) + "\n")
        return 1
    for o, a in optlist:
        if o in ("-h", "--help"):
            usage()
            return 0
    if not args:
        sys.stderr.write(_("whatuses: missing operand\n"))
        return 1
    session = sql_session.open_session()
    error = False
    for a in args:
        cluster_list = cluster_utils.name2cluster_list(session, a)
        if not cluster_list:
            sys.stderr.write(_("whatuses: `%s': no such cluster\n") % a)
            error = True
        else:
            first = True
            for c in cluster_list:
                if not first:
                    print
                print c.name
                print_whatuses(session, c)
                first = False
    sql_session.close_session(session)
    return 0 if not error else 1
