# Schedwi
# Copyright (C) 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to set cluster parameters."""

import getopt
import sys

from sqlalchemy.orm.exc import NoResultFound, MultipleResultsFound

import host_utils
from tables.host_clusters import host_clusters


def usage_options(with_clustername=True):
    """Print the option list on STDOUT."""
    print _("\nOptions:")
    if with_clustername:
        print _("""  --name=NAME                change cluster name.
  --delhost=HOSTNAME[:PORT]  remove the given host from the cluster. This
                             option can be specified several times.""")
    print _(
        """  --addhost=HOSTNAME[:PORT]  add the provided host to the cluster.
                             This option can be specified several times.
  --description=TEXT         a description.
  -h, --help                 display this help.
    """)


def usage():
    """Print a usage message on STDOUT."""
    print _("""Usage: set [OPTION]... CLUSTER
Change cluster parameters.""")
    usage_options()


def parse_args(argv):
    """Parse the argument list.

    @param argv:
                    argument list to parse
    @raise getopt.getopterror:
                    option error (unknow option or missing parameter)
    """
    optlist, args = getopt.getopt(argv, "n:h",
                                  [
                                      "name=",
                                      "addhost=",
                                      "delhost=",
                                      "description=",
                                      "help"
                                  ])
    return (optlist, args)


def set(session, cluster, opts):
    """Change cluster parameters.

    @param session:
                SQLAlchemy session
    @param cluster:
                cluster object to modify
    @type cluster:
                L{tables.clusters.clusters}
    @param opts:
                option list (from getopt.getopt() - see L{parse_args})
    @return:
                0 on success and 1 on error
    """
    hosts = map(lambda h: h.host_id, cluster.host_clusters)
    for o, a in opts:
        if o in ("-n", "--name"):
            cluster.name = a.decode('utf-8')

        elif o == "--description":
            cluster.description = a.decode('utf-8')

        elif o in ("-h", "--help"):
            usage()

        elif o == "--addhost":
            try:
                host = host_utils.name2host(session, a)
            except NoResultFound:
                sys.stderr.write(_("No such host (--addhost=%s)\n") %
                                 a.encode('utf-8'))
                return 1
            except MultipleResultsFound:
                sys.stderr.write(_("More than one host match the provided \
name (--addhost=%s)\n") % a)
                return 1
            if host.id not in hosts:
                hosts.append(host.id)

        elif o == "--delhost":
            try:
                host = host_utils.name2host(session, a)
            except NoResultFound:
                sys.stderr.write(_("No such host (--delhost=%s)\n") % a)
                return 1
            except MultipleResultsFound:
                sys.stderr.write(_("More than one host match the provided \
name (--delhost=%s)\n") % a)
                return 1
            if host.id in hosts:
                hosts.remove(host.id)
    new_hosts = list()
    for host_id in hosts:
        new_hosts.append(host_clusters(host_id))
    cluster.host_clusters = new_hosts
    return 0
