# Schedwi
# Copyright (C) 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to list clusters."""

import sys
import getopt

from tables.clusters import clusters
from tables.host_clusters import host_clusters
import cluster_utils
from help import print_trim


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: ls [OPTION]... [CLUSTER]...
    List information about the CLUSTERs (all the clusters are listed by
    default).  Sort entries alphabetically.

    Options:
      -l         use a long listing format according to the following schema:
                    NUM     number of hosts associated with the cluster
                    NAME    cluster name
      -h, --help display this help
    """))


def _print_cluster(session, cluster, long_listing):
    """Print a cluster name.

    @param session:
                    SQLAlchemy session
    @param cluster:
                    Cluster object to print
    @param long_listing:
                    If a long listing should be printed (True) or not (False)
    """
    if not long_listing:
        print cluster.name
        return
    q = session.query(host_clusters)
    nb = q.filter(host_clusters.cluster_id == cluster.id).count()
    print "%-6d  %s" % (nb, cluster.name)


def ls(sql_session, arguments):
    """List clusters.

    @param sql_session:
                SQLAlchemy session
    @params arguments:
                List of arguments given to the ls command (list of clusters)
    @return:
                0 on success and 1 on error
    """
    try:
        optlist, args = getopt.getopt(arguments, 'lh', ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write(_("ls: ") + str(err) + "\n")
        return 1
    long_listing = False
    for o, a in optlist:
        if o == "-l":
            long_listing = True
        elif o in ("-h", "--help"):
            usage()
            return 0
    session = sql_session.open_session()
    if args:
        for a in args:
            cluster_list = cluster_utils.name2cluster_list(session, a)
            if not cluster_list:
                sys.stderr.write(_("ls: `%s': no such cluster\n") % a)
            else:
                for c in cluster_list:
                    _print_cluster(session, c, long_listing)
    else:
        for c in session.query(clusters).order_by(clusters.name):
            _print_cluster(session, c, long_listing)
    sql_session.close_session(session)
    return 0
