# Schedwi
# Copyright (C) 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to print the parameters of cluster."""

import sys
import getopt

import cluster_utils
from tables.host_clusters import host_clusters
from tables.hosts import hosts
from help import print_trim


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: cat CLUSTER...
        Print the parameters of the provided CLUSTERs.

        Option:
          -h, --help  display this help.
    """))

_FIELD_WIDTH = 15


def _print_val(title, fmt=None, val=None):
    """Print a parameter value.

    @param title:   Parameter name
    @param fmt:     Format string for the parameter value
    @param val:     Value
    """
    if fmt is not None:
        if val is not None:
            print "%s :" % title.rjust(_FIELD_WIDTH), fmt % val
        else:
            print "%s :" % title.rjust(_FIELD_WIDTH), fmt
    else:
        print "%s :" % title.rjust(_FIELD_WIDTH)


def _print_cluster(session, cluster):
    """Print a cluster parameters.

    @param session: SQLAlchemy session
    @param cluster: Cluster object to print
    """
    _print_val(_("Name"), "%s", cluster.name)
    q = session.query(hosts).filter(host_clusters.host_id == hosts.id)
    q = q.filter(host_clusters.cluster_id == cluster.id)
    for host in q.order_by(hosts.hostname):
        _print_val(_("Host"),
                   "%s (SSL)" if host.sslenable else "%s",
                   str(host))
    _print_val(_("Description"), "%s", cluster.description)


def cat(sql_session, arguments):
    """Print cluster parameters.

    @param sql_session:
                SQLAlchemy session
    @param arguments:
                List of arguments given to the cat command (list of clusters)
    @return:
                0 on success and 1 on error
    """
    try:
        optlist, args = getopt.getopt(arguments, "h", ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write(_("cat: ") + str(err) + "\n")
        return 1
    for o, a in optlist:
        if o in ("-h", "--help"):
            usage()
            return 0
    if not args:
        sys.stderr.write(_("cat: missing operand\n"))
        return 1
    session = sql_session.open_session()
    error = False
    for a in args:
        cluster_list = cluster_utils.name2cluster_list(session, a)
        if not cluster_list:
            sys.stderr.write(_("cat: `%s': no such cluster\n") % a)
            error = True
        else:
            first = True
            for c in cluster_list:
                if not first:
                    print
                _print_cluster(session, c)
                first = False
    sql_session.close_session(session)
    return 0 if not error else 1
