# Schedwi
# Copyright (C) 2011, 2012 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to list calendars and directories in a tree-like format."""

import sys
import getopt

import simple_queries_cal
import path_cal
from help import print_trim


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: tree [OPTION]... [CAL|DIR]...
List contents of directories in a tree-like format (the current directory by
default).

Options:
  -d             list directories only.
  -f             prints the full path prefix for each calendar.
  -F             append  a `/' for directories.
  -i             makes tree not print the indentation lines, useful when used
                 in conjunction with the -f option.
  --dirsfirst    list directories before calendars.
  -L LEVEL       max display depth of the directory tree.
  --noreport     omits printing of the calendar and directory report at the end
                 of the tree listing.
  -h, --help     display this help.

    """))


def _print_cal(session, cal, id, last, prefix, options, workload):
        if options['full_path']:
            name = path_cal.PathCal(session, id=id[-1], workload=workload)
        else:
            name = cal.name
        if options['append_type'] and cal.entry_type == 1:
            type_char = '/'
        else:
            type_char = ''
        if not options['indentation']:
            print "%s%s" % (name, type_char)
        else:
            if last:
                print "%s`--%s%s" % (prefix, name, type_char)
            else:
                print "%s|--%s%s" % (prefix, name, type_char)


def _print_recur(session, id, prefix, level, counters, options, workload):
    cals = simple_queries_cal.sql_get_cal_children(session, id[-1],
                                                   options['only_dir'],
                                                   workload,
                                                   options['sort_by_dirs'])
    l = len(cals)
    for i in range(l):
        cal = cals[i]
        if cal.entry_type == 1:
            counters["dirs"] += 1
        else:
            counters["cals"] += 1
        cal_ids = id + [cal.id]
        _print_cal(session, cal, cal_ids, True if i + 1 == l else False,
                   prefix, options, workload)
        if i + 1 == l:
            sub_prefix = prefix + '    '
        else:
            sub_prefix = prefix + '|   '
        if options['max_level'] == 0 or level < options['max_level']:
            _print_recur(session, cal_ids, sub_prefix, level + 1, counters,
                         options, workload)


def tree(sql_session, current_cwd, arguments, workload=None):
    """List calendars and directories in a tree-like format.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working directory (a path_cal.PathCal object)
    arguments -- list of arguments given to the tree command (list
                 of calendars/directories)
    workload -- workload to use

    """
    try:
        optlist, args = getopt.getopt(arguments, 'dfFiL:h',
                                      ["help", "noreport", "dirsfirst"])
    except getopt.GetoptError, err:
        sys.stderr.write(_("tree: ") + str(err) + "\n")
        return 1
    options = {'only_dir': False,
               'full_path': False,
               'print_report': True,
               'append_type': False,
               'max_level': 0,
               'sort_by_dirs': False,
               'indentation': True
               }
    for o, a in optlist:
        if o == "-d":
            options['only_dir'] = True
        elif o == "-f":
            options['full_path'] = True
        elif o == "-F":
            options['append_type'] = True
        elif o == "-i":
            options['indentation'] = False
        elif o == "-L":
            try:
                level = int(a)
            except ValueError:
                level = -1
            if level < 1:
                sys.stderr.write(
                    _("tree: Invalid level, must be greater than 0.\n"))
                return 1
            options['max_level'] = level
        elif o == "--dirsfirst":
            options['sort_by_dirs'] = True
        elif o == "--noreport":
            options['print_report'] = False
        elif o in ("-h", "--help"):
            usage()
            return 0
    paths = list()
    if args:
        for arg in args:
            p = path_cal.get_paths(sql_session, arg, current_cwd,
                                   only_dir=True, workload=workload)
            if not p:
                sys.stderr.write(_("tree: `%s': no such directory\n") % arg)
            else:
                paths.extend(p)
    else:
        paths.append(current_cwd)
    if not paths:
        sys.stderr.write(_("tree: no such directory\n"))
        return 1
    session = sql_session.open_session()
    counters = {"dirs": 0, "cals": 0}
    for p in paths:
        print unicode(p).encode('utf-8')
        _print_recur(session, p.id, '', 1, counters, options, workload)
    if options['print_report']:
        if options['only_dir']:
            print _("\n%d directories") % counters["dirs"]
        else:
            print _("\n%d directories, %d calendars") % (counters["dirs"],
                                                         counters["cals"])
    sql_session.close_session(session)
    return 0
